<?php
require_once '../config.php';

if (!isset($_SESSION['role']) || $_SESSION['role'] != 'teacher') {
    header('Location: ../login.php');
    exit;
}

// Get teacher's ID
$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT id, CONCAT(first_name, ' ', last_name) as name FROM staff WHERE user_id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher record not found. Please contact your administrator.");
}

$teacher_id = $teacher['id'];
$teacher_name = $teacher['name'];

// Get filter values
$selected_class = $_GET['class'] ?? null;
$selected_term_id = $_GET['term_id'] ?? null;
$selected_academic_year_id = $_GET['academic_year_id'] ?? null;
$selected_course_id = $_GET['course_id'] ?? null;
$ranking_type = $_GET['ranking_type'] ?? 'subject'; // 'subject' or 'overall'

// Get classes taught by this teacher
$stmt = $pdo->prepare("SELECT DISTINCT c.name FROM classes c JOIN class_courses cc ON c.id = cc.class_id JOIN teacher_courses tc ON cc.course_id = tc.course_id WHERE tc.teacher_id = ? ORDER BY c.name");
$stmt->execute([$teacher_id]);
$teacher_classes = $stmt->fetchAll(PDO::FETCH_COLUMN);

// Get courses taught by this teacher
$stmt = $pdo->prepare("SELECT c.id, c.name FROM courses c JOIN teacher_courses tc ON c.id = tc.course_id WHERE tc.teacher_id = ? ORDER BY c.name");
$stmt->execute([$teacher_id]);
$teacher_courses = $stmt->fetchAll();

// Get all terms
$stmt = $pdo->query("SELECT id, name FROM terms ORDER BY start_date DESC");
$active_terms = $stmt->fetchAll();

// Get all academic years
$stmt = $pdo->query("SELECT id, year FROM academic_years ORDER BY start_date DESC");
$active_academic_years = $stmt->fetchAll();

// Get ranking data based on type
if ($ranking_type === 'subject' && $selected_course_id) {
    // Ranking by subject/course
    $query = "
        SELECT
            s.id as student_id,
            CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as student_name,
            s.class as student_class,
            c.name as course_name,
            sub.name as subject_name,
            g.converted_final_mark,
            ROW_NUMBER() OVER (PARTITION BY c.id ORDER BY g.converted_final_mark DESC) as rank_in_course,
            COUNT(*) OVER (PARTITION BY c.id) as total_students_in_course,
            t.name as term_name,
            ay.year as academic_year
        FROM students s
        JOIN enrollments e ON s.id = e.student_id
        JOIN courses c ON e.course_id = c.id
        LEFT JOIN grades g ON e.id = g.enrollment_id
        LEFT JOIN subjects sub ON c.subject_id = sub.id
        LEFT JOIN terms t ON g.term_id = t.id
        LEFT JOIN academic_years ay ON g.academic_year_id = ay.id
        JOIN teacher_courses tc ON c.id = tc.course_id
        WHERE tc.teacher_id = ? AND c.id = ?
    ";

    $params = [$teacher_id, $selected_course_id];

    if ($selected_class) {
        $query .= " AND s.class = ?";
        $params[] = $selected_class;
    }
    if ($selected_term_id) {
        $query .= " AND g.term_id = ?";
        $params[] = $selected_term_id;
    }
    if ($selected_academic_year_id) {
        $query .= " AND g.academic_year_id = ?";
        $params[] = $selected_academic_year_id;
    }

    $query .= " ORDER BY c.id, g.converted_final_mark DESC";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $ranking_data = $stmt->fetchAll();

} elseif ($ranking_type === 'overall') {
    // Overall ranking across all courses
    $query = "
        SELECT
            s.id as student_id,
            CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as student_name,
            s.class as student_class,
            AVG(g.converted_final_mark) as average_score,
            SUM(g.converted_final_mark) as weighted_total_marks,
            COUNT(g.id) as total_grades,
            ROW_NUMBER() OVER (ORDER BY AVG(g.converted_final_mark) DESC) as overall_rank,
            COUNT(*) OVER () as total_students,
            t.name as term_name,
            ay.year as academic_year
        FROM students s
        JOIN enrollments e ON s.id = e.student_id
        JOIN courses c ON e.course_id = c.id
        LEFT JOIN grades g ON e.id = g.enrollment_id
        LEFT JOIN terms t ON g.term_id = t.id
        LEFT JOIN academic_years ay ON g.academic_year_id = ay.id
        JOIN teacher_courses tc ON c.id = tc.course_id
        WHERE tc.teacher_id = ?
    ";

    $params = [$teacher_id];

    if ($selected_class) {
        $query .= " AND s.class = ?";
        $params[] = $selected_class;
    }
    if ($selected_term_id) {
        $query .= " AND g.term_id = ?";
        $params[] = $selected_term_id;
    }
    if ($selected_academic_year_id) {
        $query .= " AND g.academic_year_id = ?";
        $params[] = $selected_academic_year_id;
    }

    $query .= "
        GROUP BY s.id, CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name), s.class, t.name, ay.year
        HAVING COUNT(g.id) > 0
        ORDER BY AVG(g.converted_final_mark) DESC
    ";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $ranking_data = $stmt->fetchAll();
} else {
    $ranking_data = [];
}

include '../includes/teacher_header.php';
?>

<style>
    .ranking-container {
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    .page-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 30px;
        border-radius: 15px;
        margin-bottom: 30px;
        text-align: center;
        box-shadow: 0 10px 30px rgba(102, 126, 234, 0.3);
        position: relative;
        overflow: hidden;
    }

    .page-header h2 {
        margin: 0;
        font-size: 2.5em;
        font-weight: 700;
        position: relative;
        z-index: 1;
    }

    .filters-section {
        background: white;
        padding: 30px;
        border-radius: 15px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        margin-bottom: 30px;
        border: 1px solid #e9ecef;
    }

    .filters-section h3 {
        margin: 0 0 25px 0;
        color: #333;
        font-size: 1.5em;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .filter-form {
        display: flex;
        flex-wrap: wrap;
        gap: 25px;
        align-items: end;
    }

    .filter-group {
        display: flex;
        flex-direction: column;
        min-width: 220px;
        flex: 1;
    }

    .filter-group label {
        font-weight: 600;
        color: #555;
        margin-bottom: 8px;
        font-size: 0.95em;
    }

    .filter-group select {
        padding: 12px 16px;
        border: 2px solid #e9ecef;
        border-radius: 10px;
        font-size: 1em;
        background: white;
        transition: all 0.3s ease;
        cursor: pointer;
    }

    .filter-group select:focus {
        outline: none;
        border-color: #667eea;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        transform: translateY(-1px);
    }

    .filter-btn {
        padding: 14px 30px;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border: none;
        border-radius: 10px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        gap: 8px;
        font-size: 1em;
        min-height: 48px;
    }

    .filter-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
    }

    .ranking-table-container {
        background: white;
        border-radius: 15px;
        overflow: hidden;
        box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        border: 1px solid #e9ecef;
    }

    .ranking-table-container h3 {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        margin: 0;
        padding: 25px 30px;
        font-size: 1.5em;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    table {
        width: 100%;
        border-collapse: collapse;
        font-size: 0.95em;
    }

    th, td {
        padding: 18px 20px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    th {
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        font-weight: 700;
        color: #333;
        text-transform: uppercase;
        font-size: 0.8em;
        letter-spacing: 1px;
        position: sticky;
        top: 0;
        z-index: 10;
    }

    tr:nth-child(even) {
        background: #fafbfc;
    }

    tr:hover {
        background: linear-gradient(135deg, #e3f2fd 0%, #f3e5f5 100%);
        transition: background-color 0.3s ease;
        transform: scale(1.01);
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    }

    .rank-badge {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        width: 40px;
        height: 40px;
        border-radius: 50%;
        font-weight: 700;
        font-size: 1.1em;
        color: white;
        margin-right: 15px;
    }

    .rank-1 { background: linear-gradient(135deg, #ffd700, #ffb347); }
    .rank-2 { background: linear-gradient(135deg, #c0c0c0, #a8a8a8); }
    .rank-3 { background: linear-gradient(135deg, #cd7f32, #a0522d); }
    .rank-other { background: linear-gradient(135deg, #667eea, #764ba2); }

    .student-name {
        font-weight: 600;
        color: #333;
        font-size: 1.05em;
    }

    .score-display {
        font-weight: 700;
        font-size: 1.1em;
        color: #333;
    }

    .no-data {
        text-align: center;
        padding: 80px 20px;
        color: #666;
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    }

    .no-data i {
        font-size: 5em;
        color: #ccc;
        margin-bottom: 25px;
        display: block;
        opacity: 0.7;
    }

    .no-data p {
        margin: 10px 0;
        font-size: 1.1em;
    }

    .no-data p:first-of-type {
        font-weight: 600;
        color: #333;
    }

    /* Responsive Design */
    @media (max-width: 768px) {
        .ranking-container {
            padding: 15px;
        }

        .page-header {
            padding: 25px 20px;
            margin-bottom: 20px;
        }

        .page-header h2 {
            font-size: 2em;
        }

        .filter-form {
            flex-direction: column;
            align-items: stretch;
            gap: 20px;
        }

        .filter-group {
            min-width: auto;
        }

        .filter-btn {
            width: 100%;
            justify-content: center;
        }

        .ranking-table-container h3 {
            padding: 20px;
            font-size: 1.3em;
        }

        table {
            font-size: 0.9em;
        }

        th, td {
            padding: 12px 15px;
        }

        .rank-badge {
            width: 35px;
            height: 35px;
            font-size: 1em;
            margin-right: 10px;
        }
    }
</style>

<div class="ranking-container">
    <div class="page-header">
        <h2>🏆 Students Ranking</h2>
        <p>View student rankings by subject or overall performance</p>
    </div>

    <!-- Filters Section -->
    <div class="filters-section">
        <h3>🔍 Filter Rankings</h3>
        <form method="get" action="students_ranking.php" class="filter-form">
            <div class="filter-group">
                <label for="ranking_type">Ranking Type:</label>
                <select id="ranking_type" name="ranking_type">
                    <option value="subject" <?php echo ($ranking_type == 'subject') ? 'selected' : ''; ?>>By Subject/Course</option>
                    <option value="overall" <?php echo ($ranking_type == 'overall') ? 'selected' : ''; ?>>Overall Ranking</option>
                </select>
            </div>
            <div class="filter-group">
                <label for="class">Class:</label>
                <select id="class" name="class">
                    <option value="">All Classes</option>
                    <?php foreach ($teacher_classes as $class): ?>
                        <option value="<?php echo htmlspecialchars($class); ?>" <?php echo ($selected_class == $class) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($class); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group" id="course-group" style="<?php echo ($ranking_type == 'overall') ? 'display: none;' : ''; ?>">
                <label for="course_id">Course:</label>
                <select id="course_id" name="course_id">
                    <option value="">Select Course</option>
                    <?php foreach ($teacher_courses as $course): ?>
                        <option value="<?php echo $course['id']; ?>" <?php echo ($selected_course_id == $course['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($course['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label for="term_id">Term/Semester:</label>
                <select id="term_id" name="term_id">
                    <option value="">All Terms</option>
                    <?php foreach ($active_terms as $term): ?>
                        <option value="<?php echo $term['id']; ?>" <?php echo ($selected_term_id == $term['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($term['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label for="academic_year_id">Academic Year:</label>
                <select id="academic_year_id" name="academic_year_id">
                    <option value="">All Academic Years</option>
                    <?php foreach ($active_academic_years as $academic_year): ?>
                        <option value="<?php echo $academic_year['id']; ?>" <?php echo ($selected_academic_year_id == $academic_year['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($academic_year['year']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <button type="submit" class="filter-btn">🔍 Apply Filters</button>
        </form>
    </div>

    <!-- Ranking Table -->
    <div class="ranking-table-container">
        <?php if ($ranking_type === 'subject' && $selected_course_id): ?>
            <h3>📚 Subject Ranking - <?php echo htmlspecialchars($teacher_courses[array_search($selected_course_id, array_column($teacher_courses, 'id'))]['name'] ?? 'Selected Course'); ?></h3>
        <?php elseif ($ranking_type === 'overall'): ?>
            <h3>🏆 Overall Student Ranking</h3>
        <?php else: ?>
            <h3>🏆 Student Rankings</h3>
        <?php endif; ?>

        <?php if (empty($ranking_data)): ?>
            <div class="no-data">
                <i>🏆</i>
                <p>No ranking data available</p>
                <p>Please select appropriate filters and ensure students have been graded.</p>
            </div>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>Rank</th>
                        <th>Student Name</th>
                        <th>Class</th>
                        <?php if ($ranking_type === 'subject'): ?>
                            <th>Subject</th>
                            <th>Final Score</th>
                        <?php else: ?>
                            <th>Average Score</th>
                            <th>Weighted Total Marks</th>
                            <th>Total Grades</th>
                        <?php endif; ?>
                        <th>Term</th>
                        <th>Academic Year</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($ranking_data as $student): ?>
                        <tr>
                            <td>
                                <span class="rank-badge rank-<?php echo ($student['rank_in_course'] ?? $student['overall_rank']) <= 3 ? ($student['rank_in_course'] ?? $student['overall_rank']) : 'other'; ?>">
                                    <?php echo $student['rank_in_course'] ?? $student['overall_rank']; ?>
                                </span>
                            </td>
                            <td class="student-name"><?php echo htmlspecialchars($student['student_name']); ?></td>
                            <td><?php echo htmlspecialchars($student['student_class']); ?></td>
                            <?php if ($ranking_type === 'subject'): ?>
                                <td><?php echo htmlspecialchars($student['subject_name'] ?? $student['course_name']); ?></td>
                                <td class="score-display"><?php echo htmlspecialchars($student['converted_final_mark'] ?? 'N/A'); ?></td>
                            <?php else: ?>
                                <td class="score-display"><?php echo $student['average_score'] ? number_format($student['average_score'], 1) . '%' : 'N/A'; ?></td>
                                <td class="score-display"><?php echo $student['weighted_total_marks'] ? number_format($student['weighted_total_marks'], 1) : 'N/A'; ?></td>
                                <td><?php echo htmlspecialchars($student['total_grades']); ?></td>
                            <?php endif; ?>
                            <td><?php echo htmlspecialchars($student['term_name'] ?? 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($student['academic_year'] ?? 'N/A'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const rankingTypeSelect = document.getElementById('ranking_type');
    const courseGroup = document.getElementById('course-group');

    rankingTypeSelect.addEventListener('change', function() {
        if (this.value === 'overall') {
            courseGroup.style.display = 'none';
        } else {
            courseGroup.style.display = 'block';
        }
    });

    // Auto-submit form when filters change
    const filters = document.querySelectorAll('select[name="class"], select[name="term_id"], select[name="academic_year_id"], select[name="course_id"]');
    filters.forEach(filter => {
        filter.addEventListener('change', function() {
            this.closest('form').submit();
        });
    });
});
</script>

<?php
include '../includes/footer.php';
?>
