<?php
require_once '../config.php';

if (!in_array($_SESSION['role'], ['admin', 'teacher'])) {
    header('Location: ../dashboard.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT id FROM staff WHERE user_id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher not found.");
}

$teacher_id = $teacher['id'];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_resource'])) {
    $academic_year_id = $_POST['academic_year_id'] ?? null;
    $term_id = $_POST['term_id'] ?? null;
    $title = trim($_POST['title'] ?? '');
    $file = $_FILES['resource_file'] ?? null;

    $errors = [];

    if (!$academic_year_id) {
        $errors[] = "Academic year is required.";
    }
    if (!$term_id) {
        $errors[] = "Term is required.";
    }
    if (empty($title)) {
        $errors[] = "Title is required.";
    }
    if (!$file) {
        $errors[] = "No file was uploaded.";
    } elseif ($file['error'] !== UPLOAD_ERR_OK) {
        // Provide specific error message based on upload error code
        switch ($file['error']) {
            case UPLOAD_ERR_INI_SIZE:
                $errors[] = "File size exceeds server limit (upload_max_filesize).";
                break;
            case UPLOAD_ERR_FORM_SIZE:
                $errors[] = "File size exceeds form limit (MAX_FILE_SIZE).";
                break;
            case UPLOAD_ERR_PARTIAL:
                $errors[] = "File was only partially uploaded.";
                break;
            case UPLOAD_ERR_NO_FILE:
                $errors[] = "No file was uploaded.";
                break;
            case UPLOAD_ERR_NO_TMP_DIR:
                $errors[] = "Missing temporary folder.";
                break;
            case UPLOAD_ERR_CANT_WRITE:
                $errors[] = "Failed to write file to disk.";
                break;
            case UPLOAD_ERR_EXTENSION:
                $errors[] = "File upload stopped by extension.";
                break;
            default:
                $errors[] = "File upload failed (Error code: " . $file['error'] . ").";
                break;
        }
    }

    if (empty($errors)) {
        $upload_dir = __DIR__ . '/../uploads/resources/exams/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }

        $filename = basename($file['name']);
        $target_path = $upload_dir . uniqid() . '_' . $filename;

        if (move_uploaded_file($file['tmp_name'], $target_path)) {
            $relative_path = 'uploads/resources/exams/' . basename($target_path);

            $insert_stmt = $pdo->prepare("INSERT INTO teacher_resources (teacher_id, type, academic_year_id, term_id, title, file_path) VALUES (?, 'exam', ?, ?, ?, ?)");
            $insert_stmt->execute([$teacher_id, $academic_year_id, $term_id, $title, $relative_path]);

            header("Location: exam_resources.php?success=1");
            exit;
        } else {
            $errors[] = "Failed to move uploaded file.";
        }
    }
}

// Fetch academic years for dropdown
$academic_years = $pdo->query("SELECT id, Year FROM academic_years ORDER BY Year DESC")->fetchAll();

// Fetch terms for dropdown
$terms = $pdo->query("SELECT id, name FROM terms ORDER BY name")->fetchAll();

// Handle filter parameters
$search = $_GET['search'] ?? '';
$academic_year_filter = $_GET['academic_year'] ?? '';
$term_filter = $_GET['term'] ?? '';

// Build query with filters
$query = "SELECT tr.id, tr.title, tr.file_path, ay.Year AS academic_year, t.name AS term, tr.uploaded_at
    FROM teacher_resources tr
    JOIN academic_years ay ON tr.academic_year_id = ay.id
    JOIN terms t ON tr.term_id = t.id
    WHERE tr.teacher_id = ? AND tr.type = 'exam'";

$params = [$teacher_id];

if (!empty($search)) {
    $query .= " AND tr.title LIKE ?";
    $params[] = '%' . $search . '%';
}

if (!empty($academic_year_filter)) {
    $query .= " AND tr.academic_year_id = ?";
    $params[] = $academic_year_filter;
}

if (!empty($term_filter)) {
    $query .= " AND tr.term_id = ?";
    $params[] = $term_filter;
}

$query .= " ORDER BY tr.uploaded_at DESC";

$resources_stmt = $pdo->prepare($query);
$resources_stmt->execute($params);
$resources = $resources_stmt->fetchAll();

$role = $_SESSION['role'];
if ($role == 'admin') {
    include '../includes/admin_header.php';
} else {
    include '../includes/teacher_header.php';
}
?>

<style>
    .resources-container {
        padding: 20px;
        background-color: #f8f9fa;
        min-height: 100vh;
    }
    .page-header {
        background: linear-gradient(135deg, #2980b9 0%, #3498db 100%);
        color: white;
        padding: 30px;
        border-radius: 15px;
        margin-bottom: 30px;
        text-align: center;
        box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    }
    .page-header h2 {
        margin: 0;
        font-size: 2.5em;
        font-weight: 700;
    }
    .resources-section {
        background: white;
        border-radius: 15px;
        padding: 25px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
    }
    .resources-section h3 {
        color: #2c3e50;
        margin-bottom: 20px;
        font-size: 1.8em;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    .btn-primary {
        background-color: #2980b9;
        border: none;
        padding: 10px 20px;
        color: white;
        border-radius: 5px;
        cursor: pointer;
        font-weight: 600;
    }
    .btn-primary:hover {
        background-color: #3498db;
    }
    table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }
    th, td {
        padding: 15px;
        text-align: left;
        border-bottom: 1px solid #ddd;
    }
    th {
        background-color: #2980b9;
        color: white;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 1px;
    }
    tr:nth-child(even) {
        background-color: #f8f9fa;
    }
    tr:hover {
        background-color: #e8f5f8;
    }
    .download-btn {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        background: linear-gradient(135deg, #2980b9 0%, #3498db 100%);
        color: white;
        text-decoration: none;
        padding: 8px 16px;
        border-radius: 6px;
        font-size: 14px;
        font-weight: 600;
        transition: all 0.3s ease;
        border: none;
        cursor: pointer;
    }
    .download-btn:hover {
        background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(41, 128, 185, 0.3);
    }
    .download-btn i {
        font-size: 12px;
    }
    .no-resources {
        text-align: center;
        color: #6c757d;
        font-size: 1.2em;
        padding: 40px;
    }
    .filter-form {
        background: #f8f9fa;
        padding: 20px;
        border-radius: 10px;
        margin-bottom: 20px;
        border: 1px solid #e9ecef;
    }
    .filter-form .form-row {
        display: flex;
        gap: 15px;
        align-items: end;
    }
    .filter-form .form-group {
        flex: 1;
    }
    .filter-form label {
        display: block;
        margin-bottom: 5px;
        font-weight: 600;
        color: #2c3e50;
    }
    .filter-form select,
    .filter-form input[type="text"] {
        width: 100%;
        padding: 10px;
        border: 1px solid #ddd;
        border-radius: 5px;
        font-size: 14px;
    }
    .filter-form .btn-secondary {
        background-color: #6c757d;
        color: white;
        border: none;
        padding: 10px 20px;
        border-radius: 5px;
        cursor: pointer;
        font-weight: 600;
    }
    .filter-form .btn-secondary:hover {
        background-color: #5a6268;
    }
    @media (max-width: 768px) {
        table {
            font-size: 0.9em;
        }
        th, td {
            padding: 10px;
        }
        .page-header h2 {
            font-size: 2em;
        }
        .filter-form .form-row {
            flex-direction: column;
        }
        .filter-form .form-group {
            width: 100%;
        }
    }
</style>

<div class="resources-container">
    <div class="page-header">
        <h2>📁 Exam Resources</h2>
    </div>

    <div class="resources-section">
        <h3>
            My Exam Resources
            <button class="btn-primary" id="addResourceBtn">Add Resource</button>
        </h3>

        <?php if (!empty($_GET['success'])): ?>
            <div style="color: green; margin-bottom: 15px;">Resource added successfully!</div>
        <?php endif; ?>

        <?php if (!empty($errors)): ?>
            <div style="color: red; margin-bottom: 15px;">
                <?php foreach ($errors as $error): ?>
                    <div><?php echo htmlspecialchars($error); ?></div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <!-- Filter Form -->
        <form method="GET" class="filter-form">
            <div class="form-row">
                <div class="form-group">
                    <label for="search">Search by Title:</label>
                    <input type="text" name="search" id="search" value="<?php echo htmlspecialchars($search); ?>" placeholder="Enter resource title...">
                </div>
                <div class="form-group">
                    <label for="academic_year">Academic Year:</label>
                    <select name="academic_year" id="academic_year">
                        <option value="">All Academic Years</option>
                        <?php foreach ($academic_years as $ay): ?>
                            <option value="<?php echo $ay['id']; ?>" <?php echo ($academic_year_filter == $ay['id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars($ay['Year']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label for="term">Term:</label>
                    <select name="term" id="term">
                        <option value="">All Terms</option>
                        <?php foreach ($terms as $term): ?>
                            <option value="<?php echo $term['id']; ?>" <?php echo ($term_filter == $term['id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars($term['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <button type="submit" class="btn-secondary">Filter</button>
                </div>
            </div>
        </form>

        <?php if (empty($resources)): ?>
            <div class="no-resources">
                <p>No exam resources uploaded yet.</p>
            </div>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>Title</th>
                        <th>Academic Year</th>
                        <th>Term</th>
                        <th>Uploaded At</th>
                        <th>Download</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($resources as $resource): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($resource['title']); ?></td>
                        <td><?php echo htmlspecialchars($resource['academic_year']); ?></td>
                        <td><?php echo htmlspecialchars($resource['term']); ?></td>
                        <td><?php echo htmlspecialchars($resource['uploaded_at']); ?></td>
                        <td>
                            <a href="../<?php echo htmlspecialchars($resource['file_path']); ?>" download class="download-btn">
                                <i class="fas fa-download"></i> Download
                            </a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>

<!-- Add Resource Modal -->
<div id="addResourceModal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.6); z-index:1000; backdrop-filter: blur(2px);">
    <div style="background:white; max-width:600px; margin:50px auto; border-radius:15px; box-shadow: 0 20px 60px rgba(0,0,0,0.3); position:relative; overflow:hidden;">
        <!-- Modal Header -->
        <div style="background: linear-gradient(135deg, #2980b9 0%, #3498db 100%); color: white; padding: 20px 30px; position: relative;">
            <h3 style="margin: 0; font-size: 1.5em; font-weight: 600;">📄 Add Exam Resource</h3>
            <button id="closeModalBtn" style="position: absolute; top: 15px; right: 20px; background: none; border: none; color: white; font-size: 24px; cursor: pointer; padding: 0; width: 30px; height: 30px; display: flex; align-items: center; justify-content: center; border-radius: 50%; transition: background-color 0.3s;">&times;</button>
        </div>

        <!-- Modal Body -->
        <div style="padding: 30px;">
            <form method="POST" enctype="multipart/form-data" id="resourceForm">
                <input type="hidden" name="add_resource" value="1" />

                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 20px;">
                    <div class="form-group">
                        <label for="academic_year_id" style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50;">Academic Year *</label>
                        <select name="academic_year_id" id="academic_year_id" required style="width: 100%; padding: 12px; border: 2px solid #e1e8ed; border-radius: 8px; font-size: 14px; transition: border-color 0.3s;">
                            <option value="">Select Academic Year</option>
                            <?php foreach ($academic_years as $ay): ?>
                                <option value="<?php echo $ay['id']; ?>"><?php echo htmlspecialchars($ay['Year']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="term_id" style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50;">Term *</label>
                        <select name="term_id" id="term_id" required style="width: 100%; padding: 12px; border: 2px solid #e1e8ed; border-radius: 8px; font-size: 14px; transition: border-color 0.3s;">
                            <option value="">Select Term</option>
                            <?php foreach ($terms as $term): ?>
                                <option value="<?php echo $term['id']; ?>"><?php echo htmlspecialchars($term['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <div class="form-group" style="margin-bottom: 20px;">
                    <label for="title" style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50;">Resource Title *</label>
                    <input type="text" name="title" id="title" required placeholder="Enter a descriptive title for the resource" style="width: 100%; padding: 12px; border: 2px solid #e1e8ed; border-radius: 8px; font-size: 14px; transition: border-color 0.3s;" />
                </div>

                <div class="form-group" style="margin-bottom: 20px;">
                    <label for="resource_file" style="display: block; margin-bottom: 8px; font-weight: 600; color: #2c3e50;">Select File *</label>
                    <div style="position: relative;">
                        <input type="file" name="resource_file" id="resource_file" accept=".pdf,.doc,.docx,.xls,.xlsx,.ppt,.pptx,.zip,.rar,.7z,.txt" required style="width: 100%; padding: 12px; border: 2px dashed #bdc3c7; border-radius: 8px; background: #f8f9fa; cursor: pointer; transition: all 0.3s;" />
                        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); color: #7f8c8d; pointer-events: none; font-size: 14px;">
                            📎 Click to select a file or drag and drop
                        </div>
                    </div>
                </div>

                <!-- File Preview -->
                <div id="filePreview" style="display: none; background: #f8f9fa; border: 1px solid #e1e8ed; border-radius: 8px; padding: 15px; margin-bottom: 20px;">
                    <h4 style="margin: 0 0 10px 0; color: #2c3e50; font-size: 14px;">📄 File Preview</h4>
                    <div style="display: flex; align-items: center; gap: 10px;">
                        <div id="fileIcon" style="font-size: 24px;">📄</div>
                        <div style="flex: 1;">
                            <div id="fileName" style="font-weight: 600; color: #2c3e50;"></div>
                            <div id="fileSize" style="font-size: 12px; color: #7f8c8d;"></div>
                        </div>
                        <button type="button" id="removeFileBtn" style="background: #e74c3c; color: white; border: none; padding: 5px 10px; border-radius: 4px; cursor: pointer; font-size: 12px;">Remove</button>
                    </div>
                </div>

                <!-- Upload Progress Bar -->
                <div id="uploadProgress" style="display: none; margin-bottom: 20px;">
                    <div style="margin-bottom: 10px;">
                        <span style="font-weight: 600; color: #2c3e50;">Uploading...</span>
                        <span id="progressText" style="float: right; font-size: 12px; color: #7f8c8d;">0%</span>
                    </div>
                    <div style="width: 100%; height: 8px; background: #e1e8ed; border-radius: 4px; overflow: hidden;">
                        <div id="progressBar" style="width: 0%; height: 100%; background: linear-gradient(135deg, #2980b9 0%, #3498db 100%); transition: width 0.3s ease;"></div>
                    </div>
                    <div id="uploadStatus" style="margin-top: 5px; font-size: 12px; color: #7f8c8d;">Preparing upload...</div>
                </div>

                <!-- Modal Footer -->
                <div style="display: flex; justify-content: flex-end; gap: 10px; padding-top: 20px; border-top: 1px solid #e1e8ed;">
                    <button type="button" id="cancelBtn" style="background: #95a5a6; color: white; border: none; padding: 12px 24px; border-radius: 8px; cursor: pointer; font-weight: 600; transition: background-color 0.3s;">Cancel</button>
                    <button type="submit" style="background: linear-gradient(135deg, #2980b9 0%, #3498db 100%); color: white; border: none; padding: 12px 24px; border-radius: 8px; cursor: pointer; font-weight: 600; transition: background-color 0.3s;">Upload Resource</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
    // Modal functionality
    const modal = document.getElementById('addResourceModal');
    const addResourceBtn = document.getElementById('addResourceBtn');
    const cancelBtn = document.getElementById('cancelBtn');
    const closeModalBtn = document.getElementById('closeModalBtn');
    const resourceForm = document.getElementById('resourceForm');
    const fileInput = document.getElementById('resource_file');
    const filePreview = document.getElementById('filePreview');
    const fileName = document.getElementById('fileName');
    const fileSize = document.getElementById('fileSize');
    const fileIcon = document.getElementById('fileIcon');
    const removeFileBtn = document.getElementById('removeFileBtn');

    // Open modal
    addResourceBtn.addEventListener('click', function() {
        modal.style.display = 'block';
        document.body.style.overflow = 'hidden';
    });

    // Close modal functions
    function closeModal() {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto';
        resetForm();
    }

    cancelBtn.addEventListener('click', closeModal);
    closeModalBtn.addEventListener('click', closeModal);

    // Close modal when clicking outside
    modal.addEventListener('click', function(e) {
        if (e.target === modal) {
            closeModal();
        }
    });

    // File input handling
    fileInput.addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            showFilePreview(file);
        } else {
            hideFilePreview();
        }
    });

    // Remove file
    removeFileBtn.addEventListener('click', function() {
        fileInput.value = '';
        hideFilePreview();
    });

    // File preview functions
    function showFilePreview(file) {
        const fileExtension = file.name.split('.').pop().toLowerCase();
        let icon = '📄';

        // Set appropriate icon based on file type
        switch(fileExtension) {
            case 'pdf':
                icon = '📕';
                break;
            case 'doc':
            case 'docx':
                icon = '📝';
                break;
            case 'xls':
            case 'xlsx':
                icon = '📊';
                break;
            case 'ppt':
            case 'pptx':
                icon = '📽️';
                break;
            case 'zip':
            case 'rar':
            case '7z':
                icon = '📦';
                break;
            case 'txt':
                icon = '📄';
                break;
        }

        fileIcon.textContent = icon;
        fileName.textContent = file.name;
        fileSize.textContent = formatFileSize(file.size);

        filePreview.style.display = 'block';
    }

    function hideFilePreview() {
        filePreview.style.display = 'none';
    }

    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    // Form validation and AJAX upload
    resourceForm.addEventListener('submit', function(e) {
        e.preventDefault(); // Prevent default form submission

        const academicYear = document.getElementById('academic_year_id').value;
        const term = document.getElementById('term_id').value;
        const title = document.getElementById('title').value.trim();
        const file = fileInput.files[0];

        let isValid = true;
        let errors = [];

        // Clear previous error styling
        document.querySelectorAll('.form-group').forEach(group => {
            group.classList.remove('error');
        });

        // Validate academic year
        if (!academicYear) {
            errors.push('Academic year is required');
            document.querySelector('label[for="academic_year_id"]').parentElement.classList.add('error');
            isValid = false;
        }

        // Validate term
        if (!term) {
            errors.push('Term is required');
            document.querySelector('label[for="term_id"]').parentElement.classList.add('error');
            isValid = false;
        }

        // Validate title
        if (!title) {
            errors.push('Resource title is required');
            document.querySelector('label[for="title"]').parentElement.classList.add('error');
            isValid = false;
        } else if (title.length < 3) {
            errors.push('Title must be at least 3 characters long');
            document.querySelector('label[for="title"]').parentElement.classList.add('error');
            isValid = false;
        }

        // Validate file
        if (!file) {
            errors.push('Please select a file to upload');
            document.querySelector('label[for="resource_file"]').parentElement.classList.add('error');
            isValid = false;
        } else {
            // Check file size (max 10MB)
            if (file.size > 10 * 1024 * 1024) {
                errors.push('File size must be less than 10MB');
                document.querySelector('label[for="resource_file"]').parentElement.classList.add('error');
                isValid = false;
            }

            // Check file type
            const allowedTypes = ['.pdf','.doc','.docx','.xls','.xlsx','.ppt','.pptx','.zip','.rar','.7z','.txt'];
            const fileExtension = '.' + file.name.split('.').pop().toLowerCase();
            if (!allowedTypes.includes(fileExtension)) {
                errors.push('File type not allowed. Please upload PDF, Word, Excel, PowerPoint, or archive files');
                document.querySelector('label[for="resource_file"]').parentElement.classList.add('error');
                isValid = false;
            }
        }

        if (!isValid) {
            alert('Please fix the following errors:\n\n' + errors.join('\n'));
            return false;
        }

        // Start AJAX upload with progress tracking
        uploadFile(academicYear, term, title, file);
    });

    // AJAX upload function with progress tracking
    function uploadFile(academicYear, term, title, file) {
        const submitBtn = resourceForm.querySelector('button[type="submit"]');
        const progressContainer = document.getElementById('uploadProgress');
        const progressBar = document.getElementById('progressBar');
        const progressText = document.getElementById('progressText');
        const uploadStatus = document.getElementById('uploadStatus');

        // Show progress bar and disable form
        progressContainer.style.display = 'block';
        submitBtn.disabled = true;
        submitBtn.textContent = 'Uploading...';

        // Create FormData
        const formData = new FormData();
        formData.append('add_resource', '1');
        formData.append('academic_year_id', academicYear);
        formData.append('term_id', term);
        formData.append('title', title);
        formData.append('resource_file', file);

        // Create XMLHttpRequest for upload with progress tracking
        const xhr = new XMLHttpRequest();

        // Progress event
        xhr.upload.addEventListener('progress', function(e) {
            if (e.lengthComputable) {
                const percentComplete = Math.round((e.loaded / e.total) * 100);
                progressBar.style.width = percentComplete + '%';
                progressText.textContent = percentComplete + '%';

                // Update status messages
                if (percentComplete < 30) {
                    uploadStatus.textContent = 'Preparing upload...';
                } else if (percentComplete < 70) {
                    uploadStatus.textContent = 'Uploading file...';
                } else if (percentComplete < 100) {
                    uploadStatus.textContent = 'Almost done...';
                } else {
                    uploadStatus.textContent = 'Processing...';
                }
            }
        });

        // Load event (completed)
        xhr.addEventListener('load', function() {
            if (xhr.status === 200) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.success) {
                        // Success - redirect to show success message
                        window.location.href = 'exam_resources.php?success=1';
                    } else {
                        // Server-side validation error
                        alert('Upload failed: ' + (response.error || 'Unknown error'));
                        resetUploadState();
                    }
                } catch (e) {
                    // If response is not JSON, assume success and redirect
                    window.location.href = 'exam_resources.php?success=1';
                }
            } else {
                // HTTP error
                alert('Upload failed. Please try again.');
                resetUploadState();
            }
        });

        // Error event
        xhr.addEventListener('error', function() {
            alert('Upload failed due to network error. Please try again.');
            resetUploadState();
        });

        // Abort event
        xhr.addEventListener('abort', function() {
            alert('Upload was cancelled.');
            resetUploadState();
        });

        // Send the request
        xhr.open('POST', window.location.href);
        xhr.send(formData);
    }

    // Reset upload state
    function resetUploadState() {
        const submitBtn = resourceForm.querySelector('button[type="submit"]');
        const progressContainer = document.getElementById('uploadProgress');
        const progressBar = document.getElementById('progressBar');
        const progressText = document.getElementById('progressText');
        const uploadStatus = document.getElementById('uploadStatus');

        progressContainer.style.display = 'none';
        progressBar.style.width = '0%';
        progressText.textContent = '0%';
        uploadStatus.textContent = 'Preparing upload...';
        submitBtn.disabled = false;
        submitBtn.textContent = 'Upload Resource';
    }

    // Reset form
    function resetForm() {
        resourceForm.reset();
        hideFilePreview();
        document.querySelectorAll('.form-group').forEach(group => {
            group.classList.remove('error');
        });
    }

    // Add responsive styles
    const style = document.createElement('style');
    style.textContent = `
        @media (max-width: 768px) {
            #addResourceModal .modal-content {
                margin: 20px !important;
                max-width: none !important;
            }

            #addResourceModal .grid-form {
                grid-template-columns: 1fr !important;
                gap: 15px !important;
            }

            #addResourceModal .modal-header {
                padding: 15px 20px !important;
            }

            #addResourceModal .modal-header h3 {
                font-size: 1.2em !important;
            }

            #addResourceModal .modal-body {
                padding: 20px !important;
            }

            #filePreview .file-info {
                flex-direction: column !important;
                align-items: flex-start !important;
                gap: 5px !important;
            }
        }

        .form-group.error select,
        .form-group.error input {
            border-color: #e74c3c !important;
            box-shadow: 0 0 0 2px rgba(231, 76, 60, 0.1) !important;
        }

        .form-group.error label {
            color: #e74c3c !important;
        }

        #resource_file:hover {
            border-color: #2980b9 !important;
            background: #ecf0f1 !important;
        }

        #resource_file:focus {
            outline: none;
            border-color: #2980b9 !important;
            box-shadow: 0 0 0 2px rgba(41, 128, 185, 0.1) !important;
        }

        select:focus, input[type="text"]:focus {
            outline: none;
            border-color: #2980b9 !important;
            box-shadow: 0 0 0 2px rgba(41, 128, 185, 0.1) !important;
        }

        button:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }

        #cancelBtn:hover {
            background-color: #7f8c8d !important;
        }

        button[type="submit"]:hover {
            background: linear-gradient(135deg, #3498db 0%, #2980b9 100%) !important;
        }

        #closeModalBtn:hover {
            background-color: rgba(255,255,255,0.1) !important;
        }
    `;
    document.head.appendChild(style);
</script>

<?php
if ($role == 'admin') {
    include '../includes/admin_footer.php';
} else {
    include '../includes/teacher_footer.php';
}
?>
