<?php
require_once '../config.php';
require_once '../includes/access_control.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'teacher') {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch teacher information
$stmt = $pdo->prepare("SELECT *, CONCAT(first_name, ' ', last_name) as name FROM staff WHERE user_id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher not found.");
}

// Get all discussion groups the teacher is part of
$stmt = $pdo->prepare("
    SELECT dg.*, COUNT(DISTINCT dp2.user_id) as participant_count
    FROM discussion_groups dg
    JOIN discussion_participants dp ON dg.id = dp.discussion_group_id
    LEFT JOIN discussion_participants dp2 ON dg.id = dp2.discussion_group_id
    WHERE dp.user_id = ? AND dg.is_active = TRUE
    GROUP BY dg.id
    ORDER BY dg.updated_at DESC
");
$stmt->execute([$user_id]);
$discussion_groups = $stmt->fetchAll();

// Get recent messages for each group
$recent_messages = [];
foreach ($discussion_groups as $group) {
    $stmt = $pdo->prepare("
        SELECT dm.*, CONCAT(t.first_name, ' ', t.last_name) as sender_name, t.profile_image
        FROM discussion_messages dm
        LEFT JOIN staff t ON dm.user_id = t.user_id
        WHERE dm.discussion_group_id = ? AND dm.is_deleted = FALSE
        ORDER BY dm.created_at DESC
        LIMIT 5
    ");
    $stmt->execute([$group['id']]);
    $recent_messages[$group['id']] = array_reverse($stmt->fetchAll());
}

// Handle new message submission
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['send_message'])) {
    $group_id = $_POST['group_id'];
    $message = trim($_POST['message']);

    if (!empty($message) && !empty($group_id)) {
        // Check if user is participant in the group
        $stmt = $pdo->prepare("SELECT id FROM discussion_participants WHERE discussion_group_id = ? AND user_id = ? AND is_active = TRUE");
        $stmt->execute([$group_id, $user_id]);

        if ($stmt->rowCount() > 0) {
            // Insert message
            $insert_stmt = $pdo->prepare("
                INSERT INTO discussion_messages (discussion_group_id, user_id, message, message_type, created_at)
                VALUES (?, ?, ?, 'text', NOW())
            ");
            $insert_stmt->execute([$group_id, $user_id, $message]);

            // Update group updated_at
            $pdo->prepare("UPDATE discussion_groups SET updated_at = NOW() WHERE id = ?")->execute([$group_id]);

            header("Location: discussion.php?group=" . $group_id);
            exit;
        }
    }
}

// Get selected group
$selected_group_id = $_GET['group'] ?? null;
$selected_group = null;
$messages = [];

if ($selected_group_id) {
    $stmt = $pdo->prepare("SELECT * FROM discussion_groups WHERE id = ? AND is_active = TRUE");
    $stmt->execute([$selected_group_id]);
    $selected_group = $stmt->fetch();

    if ($selected_group) {
        // Get all messages for the selected group
        $stmt = $pdo->prepare("
            SELECT dm.*, CONCAT(t.first_name, ' ', t.last_name) as sender_name, t.profile_image
            FROM discussion_messages dm
            LEFT JOIN staff t ON dm.user_id = t.user_id
            WHERE dm.discussion_group_id = ? AND dm.is_deleted = FALSE
            ORDER BY dm.created_at ASC
        ");
        $stmt->execute([$selected_group_id]);
        $messages = $stmt->fetchAll();

        // Get participants
        $stmt = $pdo->prepare("
            SELECT t.*, CONCAT(t.first_name, ' ', t.last_name) as name
            FROM discussion_participants dp
            JOIN staff t ON dp.user_id = t.user_id
            WHERE dp.discussion_group_id = ? AND dp.is_active = TRUE
            ORDER BY t.first_name, t.last_name
        ");
        $stmt->execute([$selected_group_id]);
        $participants = $stmt->fetchAll();
    }
}

include '../includes/teacher_header.php';
?>

<div class="discussion-container">
    <div class="discussion-header">
        <h1>💬 Staff Discussion</h1>
        <p class="header-subtitle">Connect and collaborate with your colleagues</p>
    </div>

    <div class="discussion-layout">
        <!-- Groups Sidebar -->
        <div class="groups-sidebar">
            <div class="sidebar-header">
                <h3>📚 Discussion Groups</h3>
                <div class="group-count"><?php echo count($discussion_groups); ?> groups</div>
            </div>

            <div class="groups-list">
                <?php if (empty($discussion_groups)): ?>
                    <div class="no-groups">
                        <div class="no-groups-icon">💬</div>
                        <h4>No Discussion Groups</h4>
                        <p>You're not part of any discussion groups yet.</p>
                        <p>Contact your head teacher to be added to groups.</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($discussion_groups as $group): ?>
                        <div class="group-item <?php echo ($selected_group_id == $group['id']) ? 'active' : ''; ?>"
                             onclick="selectGroup(<?php echo $group['id']; ?>)">
                            <div class="group-avatar">
                                <?php echo strtoupper(substr($group['name'], 0, 1)); ?>
                            </div>
                            <div class="group-info">
                                <div class="group-name"><?php echo htmlspecialchars($group['name']); ?></div>
                                <div class="group-description"><?php echo htmlspecialchars($group['description'] ?? 'No description'); ?></div>
                                <div class="group-meta">
                                    <span class="participant-count">👥 <?php echo $group['participant_count']; ?></span>
                                    <?php if (!empty($recent_messages[$group['id']])): ?>
                                        <span class="last-message-time">
                                            <?php echo date('M d, g:i A', strtotime($recent_messages[$group['id']][0]['created_at'])); ?>
                                        </span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <?php if (!empty($recent_messages[$group['id']])): ?>
                                <div class="recent-message">
                                    <strong><?php echo htmlspecialchars($recent_messages[$group['id']][0]['sender_name']); ?>:</strong>
                                    <?php echo htmlspecialchars(substr($recent_messages[$group['id']][0]['message'], 0, 50)); ?>
                                    <?php if (strlen($recent_messages[$group['id']][0]['message']) > 50): ?>...<?php endif; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Chat Area -->
        <div class="chat-area">
            <?php if ($selected_group): ?>
                <!-- Chat Header -->
                <div class="chat-header">
                    <div class="chat-group-info">
                        <div class="chat-group-avatar">
                            <?php echo strtoupper(substr($selected_group['name'], 0, 1)); ?>
                        </div>
                        <div class="chat-group-details">
                            <h3><?php echo htmlspecialchars($selected_group['name']); ?></h3>
                            <div class="chat-participants">
                                👥 <?php echo count($participants); ?> participants
                            </div>
                        </div>
                    </div>
                    <div class="chat-actions">
                        <button class="btn-icon" onclick="toggleParticipants()" title="View Participants">
                            👥
                        </button>
                    </div>
                </div>

                <!-- Messages Container -->
                <div class="messages-container" id="messagesContainer">
                    <?php if (empty($messages)): ?>
                        <div class="no-messages">
                            <div class="no-messages-icon">💬</div>
                            <h4>No Messages Yet</h4>
                            <p>Start the conversation by sending the first message!</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($messages as $message): ?>
                            <div class="message-wrapper">
                                <div class="message <?php echo ($message['user_id'] == $user_id) ? 'own' : 'other'; ?>">
                                    <div class="message-sender">
                                        <div class="sender-avatar">
                                            <?php echo strtoupper(substr($message['sender_name'], 0, 1)); ?>
                                        </div>
                                        <div class="sender-info">
                                            <span class="sender-name"><?php echo htmlspecialchars($message['sender_name']); ?></span>
                                            <span class="message-time">
                                                <?php echo date('M d, Y g:i A', strtotime($message['created_at'])); ?>
                                            </span>
                                        </div>
                                    </div>
                                    <div class="message-content">
                                        <?php echo nl2br(htmlspecialchars($message['message'])); ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>

                <!-- Message Input -->
                <div class="message-input-container">
                    <form method="POST" action="discussion.php?group=<?php echo $selected_group_id; ?>" class="message-form">
                        <div class="message-input-wrapper">
                            <input type="hidden" name="group_id" value="<?php echo $selected_group_id; ?>">
                            <textarea name="message" id="messageInput" placeholder="Type your message here..."
                                      rows="1" maxlength="1000" required></textarea>
                            <button type="submit" name="send_message" class="send-btn" id="sendBtn">
                                <span class="btn-icon">📤</span>
                            </button>
                        </div>
                    </form>
                </div>
            <?php else: ?>
                <div class="no-chat-selected">
                    <div class="no-chat-icon">💬</div>
                    <h3>Select a Discussion Group</h3>
                    <p>Choose a group from the sidebar to start chatting with your colleagues.</p>
                </div>
            <?php endif; ?>
        </div>

        <!-- Participants Sidebar -->
        <div class="participants-sidebar" id="participantsSidebar">
            <div class="sidebar-header">
                <h3>👥 Participants</h3>
                <button class="close-participants" onclick="toggleParticipants()">✕</button>
            </div>

            <div class="participants-list">
                <?php if (!empty($participants)): ?>
                    <?php foreach ($participants as $participant): ?>
                        <div class="participant-item">
                            <div class="participant-avatar">
                                <?php echo strtoupper(substr($participant['name'], 0, 1)); ?>
                            </div>
                            <div class="participant-info">
                                <div class="participant-name"><?php echo htmlspecialchars($participant['name']); ?></div>
                                <div class="participant-role">Teacher</div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="no-participants">
                        <p>No participants found.</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<style>
/* Professional CSS Variables for consistent theming */
:root {
    --primary-color: #2c3e50;
    --primary-light: #34495e;
    --secondary-color: #3498db;
    --secondary-light: #5dade2;
    --success-color: #27ae60;
    --warning-color: #f39c12;
    --danger-color: #e74c3c;
    --light-gray: #ecf0f1;
    --medium-gray: #95a5a6;
    --dark-gray: #7f8c8d;
    --card-shadow: 0 4px 12px rgba(0,0,0,0.08);
    --card-shadow-hover: 0 8px 25px rgba(0,0,0,0.12);
    --border-radius: 12px;
    --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    --animation-duration: 0.3s;
}

.discussion-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 20px;
    height: calc(100vh - 100px);
    display: flex;
    flex-direction: column;
    animation: fadeInUp 0.6s ease-out;
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.discussion-header {
    text-align: center;
    margin-bottom: 30px;
    padding-bottom: 20px;
    border-bottom: 2px solid rgba(255,255,255,0.1);
    position: relative;
}

.discussion-header::before {
    content: '';
    position: absolute;
    bottom: -2px;
    left: 50%;
    transform: translateX(-50%);
    width: 100px;
    height: 3px;
    background: var(--secondary-color);
    border-radius: 2px;
    animation: shimmer 2s infinite;
}

@keyframes shimmer {
    0% { opacity: 0.5; }
    50% { opacity: 1; }
    100% { opacity: 0.5; }
}

.discussion-header h1 {
    margin: 0 0 10px 0;
    font-size: 2.5em;
    font-weight: 700;
    background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    animation: titleGlow 3s ease-in-out infinite alternate;
}

@keyframes titleGlow {
    from {
        filter: brightness(1);
    }
    to {
        filter: brightness(1.2);
    }
}

.header-subtitle {
    margin: 0;
    font-size: 1.2em;
    color: #6c757d;
    font-weight: 300;
    animation: fadeInUp 0.6s ease-out 0.2s both;
}

.discussion-layout {
    display: flex;
    flex: 1;
    gap: 20px;
    height: calc(100vh - 200px);
}

/* Groups Sidebar */
.groups-sidebar {
    width: 320px;
    background: #fff;
    border-radius: var(--border-radius);
    box-shadow: var(--card-shadow);
    display: flex;
    flex-direction: column;
    overflow: hidden;
    transition: var(--transition);
    border: 1px solid rgba(255,255,255,0.2);
    backdrop-filter: blur(10px);
}

.groups-sidebar:hover {
    box-shadow: var(--card-shadow-hover);
    transform: translateY(-2px);
}

.sidebar-header {
    padding: 20px;
    background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
    color: white;
    text-align: center;
    position: relative;
    overflow: hidden;
}

.sidebar-header::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="rgba(255,255,255,0.1)"/><circle cx="75" cy="75" r="1" fill="rgba(255,255,255,0.1)"/><circle cx="50" cy="10" r="0.5" fill="rgba(255,255,255,0.05)"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
    opacity: 0.3;
}

.sidebar-header h3 {
    margin: 0 0 5px 0;
    font-size: 1.3em;
    font-weight: 600;
    position: relative;
    z-index: 1;
    animation: slideInLeft 0.5s ease-out;
}

@keyframes slideInLeft {
    from {
        opacity: 0;
        transform: translateX(-20px);
    }
    to {
        opacity: 1;
        transform: translateX(0);
    }
}

.group-count {
    font-size: 0.9em;
    opacity: 0.9;
    position: relative;
    z-index: 1;
    animation: slideInLeft 0.5s ease-out 0.1s both;
}

.groups-list {
    flex: 1;
    overflow-y: auto;
    padding: 10px 0;
    scrollbar-width: thin;
    scrollbar-color: rgba(102, 126, 234, 0.3) transparent;
}

.groups-list::-webkit-scrollbar {
    width: 6px;
}

.groups-list::-webkit-scrollbar-track {
    background: transparent;
}

.groups-list::-webkit-scrollbar-thumb {
    background: rgba(102, 126, 234, 0.3);
    border-radius: 3px;
}

.groups-list::-webkit-scrollbar-thumb:hover {
    background: rgba(102, 126, 234, 0.5);
}

.group-item {
    padding: 15px 20px;
    border-bottom: 1px solid #f0f0f0;
    cursor: pointer;
    transition: var(--transition);
    position: relative;
    display: flex;
    align-items: center;
    gap: 15px;
    animation: fadeInUp 0.4s ease-out;
}

.group-item:nth-child(even) {
    background: rgba(248, 249, 250, 0.3);
}

.group-item:hover {
    background: linear-gradient(135deg, rgba(52, 73, 94, 0.1) 0%, rgba(52, 152, 219, 0.1) 100%);
    transform: translateX(5px);
    border-left: 3px solid var(--primary-color);
}

.group-item.active {
    background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
    color: white;
    border-left: 4px solid rgba(255, 255, 255, 0.3);
    box-shadow: 0 4px 15px rgba(52, 73, 94, 0.3);
}

.group-item.active .group-meta {
    color: rgba(255,255,255,0.8);
}

.group-avatar {
    width: 45px;
    height: 45px;
    border-radius: 50%;
    background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 1.2em;
    margin-right: 0;
    transition: var(--transition);
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    position: relative;
    overflow: hidden;
}

.group-avatar::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.2) 50%, transparent 70%);
    transform: translateX(-100%);
    transition: transform 0.6s;
}

.group-item:hover .group-avatar::before {
    transform: translateX(100%);
}

.group-info {
    flex: 1;
    min-width: 0;
}

.group-name {
    font-weight: 600;
    font-size: 1em;
    margin-bottom: 3px;
    transition: var(--transition);
}

.group-description {
    font-size: 0.85em;
    color: #6c757d;
    margin-bottom: 5px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
    transition: var(--transition);
}

.group-item.active .group-description {
    color: rgba(255,255,255,0.8);
}

.group-meta {
    display: flex;
    justify-content: space-between;
    font-size: 0.75em;
    color: #adb5bd;
    transition: var(--transition);
}

.participant-count {
    display: flex;
    align-items: center;
    gap: 4px;
    background: rgba(255,255,255,0.1);
    padding: 2px 8px;
    border-radius: 10px;
    font-size: 0.7em;
}

.group-item.active .participant-count {
    background: rgba(255,255,255,0.2);
}

.recent-message {
    font-size: 0.85em;
    color: #6c757d;
    margin-top: 5px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
    transition: var(--transition);
    font-style: italic;
}

.group-item.active .recent-message {
    color: rgba(255,255,255,0.7);
}

.no-groups {
    text-align: center;
    padding: 40px 20px;
    color: #6c757d;
    animation: fadeInUp 0.5s ease-out;
}

.no-groups-icon {
    font-size: 3em;
    margin-bottom: 15px;
    opacity: 0.5;
    animation: bounce 2s infinite;
}

@keyframes bounce {
    0%, 20%, 50%, 80%, 100% {
        transform: translateY(0);
    }
    40% {
        transform: translateY(-10px);
    }
    60% {
        transform: translateY(-5px);
    }
}

.no-groups h4 {
    margin: 0 0 10px 0;
    color: #495057;
    font-weight: 600;
}

/* Chat Area */
.chat-area {
    flex: 1;
    background: #fff;
    border-radius: var(--border-radius);
    box-shadow: var(--card-shadow);
    display: flex;
    flex-direction: column;
    overflow: hidden;
    transition: var(--transition);
    border: 1px solid rgba(255,255,255,0.2);
    backdrop-filter: blur(10px);
}

.chat-area:hover {
    box-shadow: var(--card-shadow-hover);
    transform: translateY(-1px);
}

.chat-header {
    padding: 20px;
    background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
    color: white;
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: relative;
    overflow: hidden;
}

.chat-header::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain2" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="20" cy="20" r="1" fill="rgba(255,255,255,0.1)"/><circle cx="80" cy="80" r="1" fill="rgba(255,255,255,0.1)"/><circle cx="50" cy="50" r="0.5" fill="rgba(255,255,255,0.05)"/></pattern></defs><rect width="100" height="100" fill="url(%23grain2)"/></svg>');
    opacity: 0.3;
}

.chat-group-info {
    display: flex;
    align-items: center;
    gap: 15px;
    animation: slideInLeft 0.5s ease-out 0.2s both;
}

.chat-group-avatar {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: rgba(255,255,255,0.2);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 1.5em;
    transition: var(--transition);
    box-shadow: 0 2px 10px rgba(0,0,0,0.2);
    position: relative;
    overflow: hidden;
}

.chat-group-avatar::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.3) 50%, transparent 70%);
    transform: translateX(-100%);
    transition: transform 0.6s;
}

.chat-header:hover .chat-group-avatar::before {
    transform: translateX(100%);
}

.chat-group-details h3 {
    margin: 0 0 5px 0;
    font-size: 1.4em;
    font-weight: 600;
    position: relative;
    z-index: 1;
}

.chat-participants {
    font-size: 0.9em;
    opacity: 0.9;
    position: relative;
    z-index: 1;
    display: flex;
    align-items: center;
    gap: 5px;
}

.chat-actions {
    animation: slideInRight 0.5s ease-out 0.3s both;
}

@keyframes slideInRight {
    from {
        opacity: 0;
        transform: translateX(20px);
    }
    to {
        opacity: 1;
        transform: translateX(0);
    }
}

.btn-icon {
    background: rgba(255,255,255,0.2);
    border: none;
    border-radius: 50%;
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: var(--transition);
    color: white;
    position: relative;
    overflow: hidden;
}

.btn-icon::before {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    width: 0;
    height: 0;
    background: rgba(255,255,255,0.2);
    border-radius: 50%;
    transform: translate(-50%, -50%);
    transition: width 0.3s, height 0.3s;
}

.btn-icon:hover::before {
    width: 100%;
    height: 100%;
}

.btn-icon:hover {
    background: rgba(255,255,255,0.3);
    transform: scale(1.1);
}

.messages-container {
    flex: 1;
    overflow-y: auto;
    padding: 20px;
    display: flex;
    flex-direction: column;
    gap: 15px;
    scrollbar-width: thin;
    scrollbar-color: rgba(102, 126, 234, 0.3) transparent;
}

.messages-container::-webkit-scrollbar {
    width: 6px;
}

.messages-container::-webkit-scrollbar-track {
    background: transparent;
}

.messages-container::-webkit-scrollbar-thumb {
    background: rgba(102, 126, 234, 0.3);
    border-radius: 3px;
}

.messages-container::-webkit-scrollbar-thumb:hover {
    background: rgba(102, 126, 234, 0.5);
}

.message-wrapper {
    display: flex;
    margin-bottom: 10px;
    animation: messageSlideIn 0.4s ease-out;
}

@keyframes messageSlideIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.message {
    max-width: 70%;
    padding: 12px 18px;
    border-radius: 18px;
    position: relative;
    transition: var(--transition);
    backdrop-filter: blur(10px);
}

.message::before {
    content: '';
    position: absolute;
    inset: 0;
    border-radius: 18px;
    padding: 1px;
    background: linear-gradient(135deg, rgba(102, 126, 234, 0.3), rgba(118, 75, 162, 0.3));
    mask: linear-gradient(#fff 0 0) content-box, linear-gradient(#fff 0 0);
    mask-composite: xor;
    opacity: 0;
    transition: opacity 0.3s;
}

.message:hover::before {
    opacity: 1;
}

.message.own {
    margin-left: auto;
    background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.message.other {
    margin-right: auto;
    background: #f1f3f4;
    color: #333;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.message-sender {
    display: flex;
    align-items: center;
    gap: 10px;
    margin-bottom: 8px;
}

.sender-avatar {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 0.9em;
    transition: var(--transition);
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    position: relative;
    overflow: hidden;
}

.sender-avatar::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.2) 50%, transparent 70%);
    transform: translateX(-100%);
    transition: transform 0.6s;
}

.message:hover .sender-avatar::before {
    transform: translateX(100%);
}

.sender-name {
    font-weight: 600;
    font-size: 0.9em;
    transition: var(--transition);
}

.message-time {
    font-size: 0.75em;
    opacity: 0.7;
    transition: var(--transition);
}

.message-content {
    line-height: 1.4;
    word-wrap: break-word;
    transition: var(--transition);
}

.no-messages {
    text-align: center;
    padding: 60px 20px;
    color: #6c757d;
    animation: fadeInUp 0.5s ease-out;
}

.no-messages-icon {
    font-size: 4em;
    margin-bottom: 20px;
    opacity: 0.5;
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0% { transform: scale(1); }
    50% { transform: scale(1.05); }
    100% { transform: scale(1); }
}

.no-chat-selected {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    height: 100%;
    text-align: center;
    color: #6c757d;
    animation: fadeInUp 0.5s ease-out;
}

.no-chat-icon {
    font-size: 5em;
    margin-bottom: 20px;
    opacity: 0.5;
    animation: bounce 2s infinite;
}

.no-chat-selected h3 {
    margin: 0 0 15px 0;
    color: #495057;
    font-size: 1.8em;
    font-weight: 600;
}

/* Message Input */
.message-input-container {
    padding: 20px;
    border-top: 1px solid #e9ecef;
    background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
    position: relative;
    backdrop-filter: blur(10px);
}

.message-input-container::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 1px;
    background: linear-gradient(90deg, transparent 0%, var(--primary-color) 50%, transparent 100%);
    opacity: 0.3;
}

.message-input-wrapper {
    display: flex;
    gap: 10px;
    align-items: flex-end;
    position: relative;
}

.message-input-wrapper textarea {
    flex: 1;
    border: 2px solid #e9ecef;
    border-radius: 20px;
    padding: 12px 18px;
    font-size: 1em;
    resize: none;
    outline: none;
    transition: var(--transition);
    max-height: 120px;
    min-height: 44px;
    background: rgba(255, 255, 255, 0.9);
    backdrop-filter: blur(10px);
    font-family: inherit;
    line-height: 1.4;
}

.message-input-wrapper textarea:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1), 0 4px 20px rgba(102, 126, 234, 0.1);
    background: rgba(255, 255, 255, 1);
    transform: translateY(-1px);
}

.message-input-wrapper textarea::placeholder {
    color: #adb5bd;
    font-style: italic;
}

.send-btn {
    width: 44px;
    height: 44px;
    border: none;
    border-radius: 50%;
    background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
    color: white;
    cursor: pointer;
    transition: var(--transition);
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
    overflow: hidden;
    box-shadow: 0 2px 10px rgba(102, 126, 234, 0.3);
}

.send-btn::before {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    width: 0;
    height: 0;
    background: rgba(255, 255, 255, 0.3);
    border-radius: 50%;
    transform: translate(-50%, -50%);
    transition: width 0.3s, height 0.3s;
}

.send-btn:hover::before {
    width: 100%;
    height: 100%;
}

.send-btn:hover {
    transform: scale(1.1) translateY(-2px);
    box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
}

.send-btn:active {
    transform: scale(0.95) translateY(-1px);
}

.send-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
    transform: none;
    box-shadow: 0 2px 5px rgba(102, 126, 234, 0.2);
}

.send-btn .btn-icon {
    font-size: 1.2em;
    position: relative;
    z-index: 1;
    transition: var(--transition);
}

.send-btn:hover .btn-icon {
    transform: rotate(15deg) scale(1.1);
}

.message-form {
    width: 100%;
}

/* Participants Sidebar */
.participants-sidebar {
    width: 300px;
    background: #fff;
    border-radius: 15px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    display: flex;
    flex-direction: column;
    overflow: hidden;
    position: fixed;
    right: -320px;
    top: 0;
    height: 100vh;
    z-index: 1000;
    transition: right 0.3s ease;
}

.participants-sidebar.open {
    right: 0;
}

.participants-list {
    flex: 1;
    overflow-y: auto;
    padding: 10px 0;
}

.participant-item {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 12px 20px;
    border-bottom: 1px solid #f0f0f0;
}

.participant-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
}

.participant-name {
    font-weight: 600;
    color: #495057;
}

.participant-role {
    font-size: 0.85em;
    color: #6c757d;
}

.close-participants {
    background: none;
    border: none;
    font-size: 1.5em;
    cursor: pointer;
    color: #6c757d;
    padding: 10px;
    margin-left: auto;
}

.close-participants:hover {
    color: #495057;
}

/* Responsive Design */
@media (max-width: 1200px) {
    .participants-sidebar {
        width: 280px;
        right: -300px;
    }
}

@media (max-width: 768px) {
    .discussion-container {
        padding: 10px;
    }

    .discussion-layout {
        flex-direction: column;
        height: auto;
    }

    .groups-sidebar {
        width: 100%;
        max-height: 300px;
    }

    .chat-area {
        min-height: 400px;
    }

    .participants-sidebar {
        position: fixed;
        top: auto;
        bottom: 0;
        left: 0;
        right: 0;
        width: 100%;
        height: 300px;
        border-radius: 15px 15px 0 0;
        right: 0;
        transform: translateY(100%);
    }

    .participants-sidebar.open {
        transform: translateY(0);
    }
}

@media (max-width: 480px) {
    .discussion-header h1 {
        font-size: 2em;
    }

    .chat-header {
        padding: 15px;
    }

    .chat-group-details h3 {
        font-size: 1.2em;
    }

    .messages-container {
        padding: 15px;
    }

    .message {
        max-width: 85%;
    }
}
=======
/* Message Input */
.message-input-container {
    padding: 20px;
    border-top: 1px solid #e9ecef;
    background: #f8f9fa;
    position: relative;
    backdrop-filter: blur(10px);
}

.message-input-container::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 1px;
    background: linear-gradient(90deg, transparent 0%, var(--primary-gradient) 50%, transparent 100%);
    opacity: 0.3;
}

.message-input-wrapper {
    display: flex;
    gap: 10px;
    align-items: flex-end;
    position: relative;
}

.message-input-wrapper textarea {
    flex: 1;
    border: 2px solid #e9ecef;
    border-radius: 20px;
    padding: 12px 18px;
    font-size: 1em;
    resize: none;
    outline: none;
    transition: var(--transition);
    max-height: 120px;
    min-height: 44px;
    background: rgba(255, 255, 255, 0.9);
    backdrop-filter: blur(10px);
    font-family: inherit;
    line-height: 1.4;
}

.message-input-wrapper textarea:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1), 0 4px 20px rgba(102, 126, 234, 0.1);
    background: rgba(255, 255, 255, 1);
    transform: translateY(-1px);
}

.message-input-wrapper textarea::placeholder {
    color: #adb5bd;
    font-style: italic;
}

.send-btn {
    width: 44px;
    height: 44px;
    border: none;
    border-radius: 50%;
    background: var(--primary-gradient);
    color: white;
    cursor: pointer;
    transition: var(--transition);
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
    overflow: hidden;
    box-shadow: 0 2px 10px rgba(102, 126, 234, 0.3);
}

.send-btn::before {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    width: 0;
    height: 0;
    background: rgba(255, 255, 255, 0.3);
    border-radius: 50%;
    transform: translate(-50%, -50%);
    transition: width 0.3s, height 0.3s;
}

.send-btn:hover::before {
    width: 100%;
    height: 100%;
}

.send-btn:hover {
    transform: scale(1.1) translateY(-2px);
    box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
}

.send-btn:active {
    transform: scale(0.95) translateY(-1px);
}

.send-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
    transform: none;
    box-shadow: 0 2px 5px rgba(102, 126, 234, 0.2);
}

.send-btn .btn-icon {
    font-size: 1.2em;
    position: relative;
    z-index: 1;
    transition: var(--transition);
}

.send-btn:hover .btn-icon {
    transform: rotate(15deg) scale(1.1);
}

.message-form {
    width: 100%;
}

/* Participants Sidebar */
.participants-sidebar {
    width: 300px;
    background: #fff;
    border-radius: 15px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    display: flex;
    flex-direction: column;
    overflow: hidden;
    position: fixed;
    right: -320px;
    top: 0;
    height: 100vh;
    z-index: 1000;
    transition: right 0.3s ease;
}

.participants-sidebar.open {
    right: 0;
}

.participants-list {
    flex: 1;
    overflow-y: auto;
    padding: 10px 0;
}

.participant-item {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 12px 20px;
    border-bottom: 1px solid #f0f0f0;
}

.participant-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
}

.participant-name {
    font-weight: 600;
    color: #495057;
}

.participant-role {
    font-size: 0.85em;
    color: #6c757d;
}

.close-participants {
    background: none;
    border: none;
    font-size: 1.5em;
    cursor: pointer;
    color: #6c757d;
    padding: 10px;
    margin-left: auto;
}

.close-participants:hover {
    color: #495057;
}

/* Responsive Design */
@media (max-width: 1200px) {
    .participants-sidebar {
        width: 280px;
        right: -300px;
    }
}

@media (max-width: 768px) {
    .discussion-container {
        padding: 10px;
    }

    .discussion-layout {
        flex-direction: column;
        height: auto;
    }

    .groups-sidebar {
        width: 100%;
        max-height: 300px;
    }

    .chat-area {
        min-height: 400px;
    }

    .participants-sidebar {
        position: fixed;
        top: auto;
        bottom: 0;
        left: 0;
        right: 0;
        width: 100%;
        height: 300px;
        border-radius: 15px 15px 0 0;
        right: 0;
        transform: translateY(100%);
    }

    .participants-sidebar.open {
        transform: translateY(0);
    }
}

@media (max-width: 480px) {
    .discussion-header h1 {
        font-size: 2em;
    }

    .chat-header {
        padding: 15px;
    }

    .chat-group-details h3 {
        font-size: 1.2em;
    }

    .messages-container {
        padding: 15px;
    }

    .message {
        max-width: 85%;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const messageInput = document.getElementById('messageInput');
    const sendBtn = document.getElementById('sendBtn');
    const messagesContainer = document.getElementById('messagesContainer');

    // Auto-resize textarea
    messageInput.addEventListener('input', function() {
        this.style.height = 'auto';
        this.style.height = (this.scrollHeight) + 'px';

        // Enable/disable send button
        sendBtn.disabled = this.value.trim().length === 0;
    });

    // Send message on Enter (Shift+Enter for new line)
    messageInput.addEventListener('keydown', function(e) {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            if (this.value.trim()) {
                this.closest('form').submit();
            }
        }
    });

    // Scroll to bottom when new messages load
    function scrollToBottom() {
        if (messagesContainer) {
            messagesContainer.scrollTop = messagesContainer.scrollHeight;
        }
    }

    scrollToBottom();

    // Auto-scroll when new messages are added (you can implement real-time updates here)
    const observer = new MutationObserver(scrollToBottom);
    if (messagesContainer) {
        observer.observe(messagesContainer, { childList: true });
    }
});

function selectGroup(groupId) {
    window.location.href = 'discussion.php?group=' + groupId;
}

function toggleParticipants() {
    const sidebar = document.getElementById('participantsSidebar');
    sidebar.classList.toggle('open');
}

// Close participants sidebar when clicking outside
document.addEventListener('click', function(e) {
    const sidebar = document.getElementById('participantsSidebar');
    const participantsBtn = e.target.closest('.btn-icon');

    if (!sidebar.contains(e.target) && !participantsBtn && sidebar.classList.contains('open')) {
        sidebar.classList.remove('open');
    }
});
</script>

<?php
include '../includes/teacher_footer.php';
?>
