<?php
require_once '../config.php';
require_once '../auto_enroll_students.php';

if (!in_array($_SESSION['role'], ['admin', 'teacher'])) {
    header('Location: ../dashboard.php');
    exit;
}

$message = '';

// Get filter values
$filter_class = $_GET['class'] ?? '';
$filter_academic_year = $_GET['academic_year'] ?? '';
$filter_term = $_GET['term'] ?? '';
$search_query = $_GET['search'] ?? '';

// Get active academic year and term if not specified
if (!$filter_academic_year) {
    $active_year = $pdo->query("SELECT id FROM academic_years WHERE is_active = 1 LIMIT 1")->fetch();
    $filter_academic_year = $active_year ? $active_year['id'] : '';
}

if (!$filter_term) {
    $active_term = $pdo->query("SELECT id FROM terms WHERE is_active = 1 LIMIT 1")->fetch();
    $filter_term = $active_term ? $active_term['id'] : '';
}

// Get classes based on role
$role = $_SESSION['role'];
if ($role == 'admin') {
    // Admins can see all classes
    $assigned_classes = $pdo->query("SELECT id, name FROM classes ORDER BY name")->fetchAll();
} else {
    // Teachers see only their assigned classes
    $user_id = $_SESSION['user_id'];
    $stmt = $pdo->prepare("SELECT id FROM staff WHERE user_id = ? AND role = 'teaching'");
    $stmt->execute([$user_id]);
    $teacher = $stmt->fetch();

    if (!$teacher) {
        die("Teacher not found.");
    }

    $teacher_id = $teacher['id'];

    $stmt = $pdo->prepare("
        SELECT c.id, c.name
        FROM classes c
        INNER JOIN teacher_classes tc ON c.id = tc.class_id
        WHERE tc.teacher_id = ?
        ORDER BY c.name
    ");
    $stmt->execute([$teacher_id]);
    $assigned_classes = $stmt->fetchAll();
}

// Get all academic years and terms for dropdowns
$academic_years = $pdo->query("SELECT id, year FROM academic_years ORDER BY year DESC")->fetchAll();

$terms = $pdo->query("SELECT id, name FROM terms ORDER BY start_date DESC")->fetchAll();

// Get students and their enrolled courses for the selected class
$students_data = [];
$courses = [];
if ($filter_class) {
    // Auto-enroll all students in this class to ensure they have enrollments
    $auto_enroll_result = autoEnrollClass($filter_class);
    if (!$auto_enroll_result['success']) {
        $message = "Warning: " . $auto_enroll_result['message'];
    }

    // Get all students in the selected class, ordered by gender (males first), then by last_name, first_name
    $stmt = $pdo->prepare("SELECT id, CONCAT(last_name, ', ', first_name, ' ', COALESCE(middle_name, '')) as name, gender FROM students WHERE class = ? ORDER BY CASE WHEN gender = 'male' THEN 1 ELSE 2 END, last_name, first_name");
    $stmt->execute([$filter_class]);
    $students = $stmt->fetchAll();

    // Get all courses allocated to this teacher for the selected class and term
    $stmt = $pdo->prepare("
        SELECT DISTINCT c.id, c.code, c.name
        FROM courses c
        INNER JOIN teacher_courses tc ON c.id = tc.course_id
        INNER JOIN class_courses cc ON c.id = cc.course_id
        INNER JOIN course_terms ct ON c.id = ct.course_id
        WHERE tc.teacher_id = ?
        AND cc.class_id = (SELECT id FROM classes WHERE name = ?)
        AND ct.term_id = ?
        ORDER BY c.name
    ");
    $stmt->execute([$teacher_id, $filter_class, $filter_term]);
    $courses = $stmt->fetchAll();

    // Separate students by gender
    $male_students_data = [];
    $female_students_data = [];

    // For each student, get their grades filtered by term and academic year
    foreach ($students as $student) {
        $student_data = [
            'id' => $student['id'],
            'name' => $student['name'],
            'grades' => []
        ];

        // Get grades for this student in the teacher's courses, filtered by term and academic year
        foreach ($courses as $course) {
            $stmt = $pdo->prepare("
                SELECT g.grade, g.comments, g.assessment_total, g.exam_score, g.final_score
                FROM enrollments e
                LEFT JOIN grades g ON e.id = g.enrollment_id AND g.term_id = ? AND g.academic_year_id = ?
                WHERE e.student_id = ? AND e.course_id = ?
            ");
            $stmt->execute([$filter_term, $filter_academic_year, $student['id'], $course['id']]);
            $grade = $stmt->fetch();

            $student_data['grades'][$course['id']] = [
                'course_code' => $course['code'],
                'course_name' => $course['name'],
                'grade' => $grade ? $grade['grade'] : '',
                'comments' => $grade ? $grade['comments'] : '',
                'assessment_total' => $grade ? $grade['assessment_total'] : '',
                'exam_score' => $grade ? $grade['exam_score'] : '',
                'final_score' => $grade ? $grade['final_score'] : ''
            ];
        }

        if (strtolower($student['gender']) === 'male') {
            $male_students_data[] = $student_data;
        } else {
            $female_students_data[] = $student_data;
        }
    }

    // Filter students by search query if provided
    if ($search_query) {
        $search_lower = strtolower($search_query);
        $male_students_data = array_filter($male_students_data, function($student) use ($search_lower) {
            return strpos(strtolower($student['name']), $search_lower) !== false;
        });
        $female_students_data = array_filter($female_students_data, function($student) use ($search_lower) {
            return strpos(strtolower($student['name']), $search_lower) !== false;
        });
    }
}

$role = $_SESSION['role'];



if ($role == 'admin') {
    include '../includes/admin_header.php';
} else {
    include '../includes/teacher_header.php';
}
?>

<style>
    .assign-grades-container {
        padding: 20px;
        background-color: #f8f9fa;
        min-height: 100vh;
    }

    .page-header {
        background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
        color: white;
        padding: 30px;
        border-radius: 15px;
        margin-bottom: 30px;
        text-align: center;
        box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    }

    .page-header h2 {
        margin: 0;
        font-size: 2.5em;
        font-weight: 700;
    }

    .message {
        padding: 15px 20px;
        border-radius: 8px;
        margin-bottom: 20px;
        font-weight: 500;
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .classes-section {
        background: white;
        border-radius: 15px;
        padding: 25px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
    }

    .classes-section h3 {
        color: #2c3e50;
        margin-bottom: 20px;
        font-size: 1.8em;
    }

    .classes-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
        gap: 20px;
    }

    .class-card {
        background: linear-gradient(135deg, #0c56b6ff 0%, #056aaeff 100%);
        color: white;
        border-radius: 12px;
        padding: 25px;
        text-align: center;
        box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }

    .class-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(255,255,255,0.1);
        opacity: 0;
        transition: opacity 0.3s ease;
    }

    .class-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 10px 25px rgba(0,0,0,0.15);
    }

    .class-card:hover::before {
        opacity: 1;
    }

    .class-card h4 {
        margin: 0 0 20px 0;
        font-size: 1.4em;
        font-weight: 600;
        position: relative;
        z-index: 1;
    }

    .class-card .btn {
        display: inline-block;
        padding: 12px 24px;
        background: rgba(255,255,255,0.2);
        color: white;
        text-decoration: none;
        border-radius: 25px;
        font-weight: 600;
        transition: all 0.3s ease;
        border: 2px solid rgba(255,255,255,0.3);
        position: relative;
        z-index: 1;
    }

    .class-card .btn:hover {
        background: white;
        color: #667eea;
        text-decoration: none;
        transform: scale(1.05);
    }

    .filter-form {
        background: white;
        padding: 25px;
        border-radius: 15px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
        display: flex;
        flex-wrap: wrap;
        gap: 20px;
        align-items: center;
    }

    .filter-form label {
        font-weight: 600;
        color: #2c3e50;
        margin-right: 10px;
    }

    .filter-form select {
        padding: 10px 15px;
        border: 2px solid #ddd;
        border-radius: 8px;
        font-size: 1em;
        background-color: #fff;
        transition: border-color 0.3s ease;
    }

    .filter-form select:focus {
        outline: none;
        border-color: #27ae60;
    }

    .filter-form .btn {
        padding: 10px 20px;
        background: #27ae60;
        color: white;
        text-decoration: none;
        border-radius: 8px;
        font-weight: 600;
        transition: background-color 0.3s ease;
    }

    .filter-form .btn:hover {
        background: #219150;
        color: white;
        text-decoration: none;
    }

    .grades-table-container {
        background: white;
        border-radius: 15px;
        padding: 25px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
        overflow-x: auto;
    }

    .grades-table-container h3 {
        color: #2c3e50;
        margin-bottom: 20px;
        font-size: 1.8em;
    }

    table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    th, td {
        padding: 15px;
        text-align: left;
        border-bottom: 1px solid #ddd;
    }

    th {
        background-color: #27ae60;
        color: white;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    tr:nth-child(even) {
        background-color: #f8f9fa;
    }

    tr:hover {
        background-color: #e8f5e8;
    }

    .grade-not-assigned {
        color: #dc3545;
        font-weight: 600;
    }

    .btn-small {
        padding: 8px 16px;
        background: #007bff;
        color: white;
        text-decoration: none;
        border-radius: 6px;
        font-size: 0.9em;
        font-weight: 600;
        transition: all 0.3s ease;
        display: inline-block;
        border: none;
        cursor: pointer;
    }

    .btn-small:hover {
        background: #0056b3;
        color: white;
        text-decoration: none;
        transform: scale(1.05);
    }



    @media (max-width: 768px) {
        .classes-grid {
            grid-template-columns: 1fr;
        }

        .filter-form {
            flex-direction: column;
            align-items: stretch;
        }

        .filter-form label {
            margin-bottom: 5px;
        }

        table {
            font-size: 0.9em;
        }

        th, td {
            padding: 10px;
        }

        .modal-content {
            width: 95%;
            margin: 20% auto;
        }

        .tab-navigation {
            flex-wrap: wrap;
        }

        .tab-button {
            padding: 10px 15px;
            font-size: 0.8em;
        }
    }
</style>

<div class="assign-grades-container">
    <div class="page-header">
        <h2>📊 Assign Grades</h2>
    </div>

    <?php if ($message): ?>
        <div class="message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if (empty($assigned_classes)): ?>
        <div class="classes-section">
            <h3>Your Assigned Classes</h3>
            <p>No classes assigned to you.</p>
        </div>
    <?php else: ?>
        <div class="classes-section">
            <h3>Your Assigned Classes</h3>
            <div class="classes-grid">
                <?php foreach ($assigned_classes as $class): ?>
                    <div class="class-card">
                        <h4><?php echo htmlspecialchars($class['name']); ?></h4>
                        <a href="assign_grades.php?class=<?php echo urlencode($class['name']); ?>" class="btn">View Students for Grading</a>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    <?php endif; ?>

    <?php if ($filter_class): ?>
        <div class="grades-table-container">
            <h3>Grading Students in Class: <?php echo htmlspecialchars($filter_class); ?></h3>

            <div class="filter-form">
                <form method="GET" action="assign_grades.php" style="display: inline-flex; gap: 20px; align-items: center;">
                    <input type="hidden" name="class" value="<?php echo htmlspecialchars($filter_class); ?>">
                    <input type="hidden" name="academic_year" value="<?php echo htmlspecialchars($filter_academic_year); ?>">
                    <input type="hidden" name="term" value="<?php echo htmlspecialchars($filter_term); ?>">

                    <label for="search">Search Student:</label>
                    <input type="text" name="search" id="search" value="<?php echo htmlspecialchars($search_query); ?>" placeholder="Enter student name..." style="padding: 10px 15px; border: 2px solid #ddd; border-radius: 8px; font-size: 1em; background-color: #fff; transition: border-color 0.3s ease;">

                    <button type="submit" class="btn">🔍 Search</button>
                </form>

                <a href="assign_grades.php" class="btn">Back to Classes</a>
            </div>

            <?php if (empty($male_students_data) && empty($female_students_data)): ?>
                <p>No students found in this class.</p>
            <?php else: ?>
                <?php if (!empty($male_students_data)): ?>
                    <h4>Male Students</h4>
                    <table>
                        <thead>
                            <tr>
                                <th>Student Name</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($male_students_data as $student): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($student['name']); ?></td>
                                <td>
                                    <a href="add_grades.php?student_id=<?php echo $student['id']; ?>&class=<?php echo urlencode($filter_class); ?>" class="btn-small">📝 Add Marks</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>

                <?php if (!empty($female_students_data)): ?>
                    <h4>Female Students</h4>
                    <table>
                        <thead>
                            <tr>
                                <th>Student Name</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($female_students_data as $student): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($student['name']); ?></td>
                                <td>
                                    <a href="add_grades.php?student_id=<?php echo $student['id']; ?>&class=<?php echo urlencode($filter_class); ?>" class="btn-small">📝 Add Marks</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    <?php endif; ?>

</div>





<?php
if ($role == 'admin') {
    include '../includes/admin_footer.php';
} else {
    include '../includes/teacher_footer.php';
}
?>
