<?php
require_once '../config.php';

requireFeatureAccess('subscription_management');

$message = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['add_subscription'])) {
        // Add new subscription
        $client_name = trim($_POST['client_name']);
        $school_name = trim($_POST['school_name']);
        $contact_email = trim($_POST['contact_email']);
        $contact_phone = trim($_POST['contact_phone']);
        $package_id = $_POST['package_id'];
        $start_date = $_POST['start_date'];
        $end_date = $_POST['end_date'];

        if (empty($client_name) || empty($package_id)) {
            $error = 'Client name and package are required.';
        } else {
            try {
                $stmt = $pdo->prepare("INSERT INTO client_subscriptions (client_name, school_name, contact_email, contact_phone, package_id, start_date, end_date) VALUES (?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$client_name, $school_name, $contact_email, $contact_phone, $package_id, $start_date, $end_date]);
                $message = 'Subscription added successfully!';
            } catch (PDOException $e) {
                $error = 'Error adding subscription: ' . $e->getMessage();
            }
        }
    } elseif (isset($_POST['update_status'])) {
        // Update subscription status
        $subscription_id = $_POST['subscription_id'];
        $status = $_POST['status'];

        try {
            $stmt = $pdo->prepare("UPDATE client_subscriptions SET subscription_status = ? WHERE id = ?");
            $stmt->execute([$status, $subscription_id]);
            $message = 'Subscription status updated successfully!';
        } catch (PDOException $e) {
            $error = 'Error updating status: ' . $e->getMessage();
        }
    }
}

// Get all subscriptions with package details
$stmt = $pdo->query("
    SELECT cs.*, sp.name as package_name, sp.price_ghs
    FROM client_subscriptions cs
    JOIN subscription_packages sp ON cs.package_id = sp.id
    ORDER BY cs.created_at DESC
");
$subscriptions = $stmt->fetchAll();

// Get all packages for dropdown
$stmt = $pdo->query("SELECT * FROM subscription_packages ORDER BY name");
$packages = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Subscriptions - School Management System</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="../css/admin_custom.css">
    <style>
        .subscription-form {
            background: #fff;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }

        .subscription-form h3 {
            margin-bottom: 20px;
            color: #2c3e50;
        }

        .form-row {
            display: flex;
            gap: 15px;
            margin-bottom: 15px;
        }

        .form-row .form-group {
            flex: 1;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
            color: #555;
        }

        .form-group input, .form-group select {
            width: 100%;
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }

        .btn-submit {
            background: #3498db;
            color: #fff;
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
        }

        .btn-submit:hover {
            background: #2980b9;
        }

        .subscriptions-table {
            background: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
        }

        .subscriptions-table table {
            width: 100%;
            border-collapse: collapse;
        }

        .subscriptions-table th, .subscriptions-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }

        .subscriptions-table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #2c3e50;
        }

        .status-badge {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: 500;
        }

        .status-active {
            background: #d4edda;
            color: #155724;
        }

        .status-inactive {
            background: #f8d7da;
            color: #721c24;
        }

        .status-suspended {
            background: #fff3cd;
            color: #856404;
        }

        .status-expired {
            background: #e2e3e5;
            color: #383d41;
        }

        .action-btn {
            padding: 6px 12px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
            margin-right: 5px;
        }

        .btn-edit {
            background: #ffc107;
            color: #212529;
        }

        .btn-activate {
            background: #28a745;
            color: #fff;
        }

        .btn-suspend {
            background: #fd7e14;
            color: #fff;
        }

        .btn-expire {
            background: #6c757d;
            color: #fff;
        }

        .message {
            padding: 10px;
            margin-bottom: 20px;
            border-radius: 4px;
        }

        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        @media (max-width: 768px) {
            .form-row {
                flex-direction: column;
            }

            .subscriptions-table {
                overflow-x: auto;
            }
        }
    </style>
</head>
<body>
    <?php include '../includes/admin_header.php'; ?>

    <div class="main-content">
        <div class="container">
            <h2><i class="fas fa-crown"></i> Manage Subscriptions</h2>

            <?php if ($message): ?>
                <div class="message success"><?php echo htmlspecialchars($message); ?></div>
            <?php endif; ?>

            <?php if ($error): ?>
                <div class="message error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>

            <!-- Add New Subscription Form -->
            <div class="subscription-form">
                <h3><i class="fas fa-plus"></i> Add New Subscription</h3>
                <form method="post" action="manage_subscriptions.php">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="client_name">Client Name *</label>
                            <input type="text" id="client_name" name="client_name" required>
                        </div>
                        <div class="form-group">
                            <label for="school_name">School Name</label>
                            <input type="text" id="school_name" name="school_name">
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label for="contact_email">Contact Email</label>
                            <input type="email" id="contact_email" name="contact_email">
                        </div>
                        <div class="form-group">
                            <label for="contact_phone">Contact Phone</label>
                            <input type="text" id="contact_phone" name="contact_phone">
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label for="package_id">Subscription Package *</label>
                            <select id="package_id" name="package_id" required>
                                <option value="">Select Package</option>
                                <?php foreach ($packages as $package): ?>
                                    <option value="<?php echo $package['id']; ?>">
                                        <?php echo htmlspecialchars($package['name']); ?> - ₵<?php echo number_format($package['price_ghs'], 2); ?>/year
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label for="start_date">Start Date</label>
                            <input type="date" id="start_date" name="start_date" value="<?php echo date('Y-m-d'); ?>">
                        </div>
                        <div class="form-group">
                            <label for="end_date">End Date</label>
                            <input type="date" id="end_date" name="end_date" value="<?php echo date('Y-m-d', strtotime('+1 year')); ?>">
                        </div>
                    </div>
                    <button type="submit" name="add_subscription" class="btn-submit">
                        <i class="fas fa-plus"></i> Add Subscription
                    </button>
                </form>
            </div>

            <!-- Subscriptions List -->
            <div class="subscriptions-table">
                <table>
                    <thead>
                        <tr>
                            <th>Client</th>
                            <th>School</th>
                            <th>Package</th>
                            <th>Status</th>
                            <th>Start Date</th>
                            <th>End Date</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($subscriptions as $subscription): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($subscription['client_name']); ?></td>
                                <td><?php echo htmlspecialchars($subscription['school_name']); ?></td>
                                <td><?php echo htmlspecialchars($subscription['package_name']); ?></td>
                                <td>
                                    <span class="status-badge status-<?php echo $subscription['subscription_status']; ?>">
                                        <?php echo ucfirst($subscription['subscription_status']); ?>
                                    </span>
                                </td>
                                <td><?php echo $subscription['start_date']; ?></td>
                                <td><?php echo $subscription['end_date']; ?></td>
                                <td>
                                    <form method="post" action="manage_subscriptions.php" style="display: inline;">
                                        <input type="hidden" name="subscription_id" value="<?php echo $subscription['id']; ?>">
                                        <select name="status" style="padding: 4px; margin-right: 5px;">
                                            <option value="active" <?php echo $subscription['subscription_status'] == 'active' ? 'selected' : ''; ?>>Active</option>
                                            <option value="inactive" <?php echo $subscription['subscription_status'] == 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                                            <option value="suspended" <?php echo $subscription['subscription_status'] == 'suspended' ? 'selected' : ''; ?>>Suspended</option>
                                            <option value="expired" <?php echo $subscription['subscription_status'] == 'expired' ? 'selected' : ''; ?>>Expired</option>
                                        </select>
                                        <button type="submit" name="update_status" class="action-btn btn-edit">
                                            <i class="fas fa-save"></i> Update
                                        </button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script src="../js/script.js"></script>
</body>
</html>
