<?php
require_once '../config.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$username = $_SESSION['username'];

// Get current active subscription
$current_subscription = null;
$stmt = $pdo->prepare("SELECT cs.*, sp.name as package_name FROM client_subscriptions cs JOIN subscription_packages sp ON cs.package_id = sp.id WHERE cs.user_id = ? AND cs.subscription_status = 'active' LIMIT 1");
$stmt->execute([$user_id]);
$current_subscription = $stmt->fetch();

// Get all packages
$stmt = $pdo->query("SELECT * FROM subscription_packages ORDER BY price_ghs ASC");
$packages = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Subscription Dashboard - <?php echo htmlspecialchars($username); ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            color: #333;
            background-color: #f4f4f4;
        }

        .header {
            background: linear-gradient(135deg, #8e44ad 0%, #9b59b6 100%);
            color: #fff;
            padding: 20px 0;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .header .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header .logo {
            font-size: 1.5em;
            font-weight: bold;
            display: flex;
            align-items: center;
        }

        .header .logo i {
            margin-right: 10px;
            font-size: 1.8em;
        }

        .header .user-info {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .header .user-info span {
            font-weight: 500;
        }

        .header .logout-btn {
            background: rgba(255,255,255,0.2);
            color: #fff;
            padding: 8px 16px;
            border-radius: 5px;
            text-decoration: none;
            font-weight: 500;
            transition: background 0.3s ease;
        }

        .header .logout-btn:hover {
            background: rgba(255,255,255,0.3);
        }

        .header .manage-btn {
            background: rgba(255,255,255,0.2);
            color: #fff;
            padding: 8px 16px;
            border-radius: 5px;
            text-decoration: none;
            font-weight: 500;
            transition: background 0.3s ease;
        }

        .header .manage-btn:hover {
            background: rgba(255,255,255,0.3);
        }

        .main-content {
            max-width: 1200px;
            margin: 30px auto;
            padding: 0 20px;
        }

        .welcome-section {
            background: #fff;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 30px;
            text-align: center;
        }

        .welcome-section h1 {
            color: #8e44ad;
            margin-bottom: 10px;
        }

        .welcome-section p {
            color: #666;
            font-size: 1.1em;
        }

        .current-subscription {
            background: #e8f5e8;
            border: 1px solid #4caf50;
            color: #2e7d32;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            text-align: center;
        }

        .packages-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
        }

        .package-card {
            background: #fff;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
            transition: transform 0.3s ease;
        }

        .package-card:hover {
            transform: translateY(-5px);
        }

        .package-header {
            background: linear-gradient(135deg, #8e44ad 0%, #9b59b6 100%);
            color: #fff;
            padding: 20px;
            text-align: center;
        }

        .package-header h3 {
            font-size: 1.5em;
            margin-bottom: 10px;
        }

        .package-price {
            font-size: 2em;
            font-weight: bold;
            margin-bottom: 10px;
        }

        .package-price span {
            font-size: 0.5em;
            font-weight: normal;
        }

        .package-body {
            padding: 20px;
        }

        .package-description {
            color: #666;
            margin-bottom: 20px;
            text-align: center;
        }

        .package-features {
            list-style: none;
            margin-bottom: 20px;
        }

        .package-features li {
            padding: 5px 0;
            display: flex;
            align-items: center;
        }

        .package-features li i {
            color: #4caf50;
            margin-right: 10px;
        }

        .package-limits {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }

        .package-limits p {
            margin: 5px 0;
            font-size: 0.9em;
        }

        .activate-btn {
            display: block;
            width: 100%;
            padding: 12px;
            background: linear-gradient(135deg, #4caf50 0%, #45a049 100%);
            color: #fff;
            text-decoration: none;
            text-align: center;
            border-radius: 5px;
            font-weight: 500;
            transition: background 0.3s ease;
        }

        .activate-btn:hover {
            background: linear-gradient(135deg, #45a049 0%, #4caf50 100%);
        }

        .current-package {
            background: #fff3cd;
            border: 1px solid #ffc107;
            color: #856404;
            padding: 12px;
            text-align: center;
            border-radius: 5px;
            font-weight: 500;
        }

        .no-subscription {
            background: #f8d7da;
            border: 1px solid #dc3545;
            color: #721c24;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            text-align: center;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .header .container {
                flex-direction: column;
                gap: 15px;
            }

            .packages-grid {
                grid-template-columns: 1fr;
            }

            .main-content {
                padding: 0 15px;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="container">
            <div class="logo">
                <i class="fas fa-crown"></i>
                <span>Subscription Dashboard</span>
            </div>
            <div class="user-info">
                <span>Welcome, <?php echo htmlspecialchars($username); ?></span>
                <a href="manage_packages.php" class="manage-btn">
                    <i class="fas fa-box"></i> Manage Packages
                </a>
                <a href="logout.php" class="logout-btn">
                    <i class="fas fa-sign-out-alt"></i> Logout
                </a>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="welcome-section">
            <h1>Choose Your Subscription Package</h1>
            <p>Select the package that best fits your school's needs</p>
        </div>

        <?php if ($current_subscription): ?>
            <div class="current-subscription">
                <i class="fas fa-check-circle"></i>
                You are currently subscribed to: <strong><?php echo htmlspecialchars($current_subscription['package_name']); ?></strong>
            </div>
        <?php else: ?>
            <div class="no-subscription">
                <i class="fas fa-exclamation-triangle"></i>
                You don't have an active subscription. Please select a package below.
            </div>
        <?php endif; ?>

        <div class="packages-grid">
            <?php foreach ($packages as $package): ?>
                <div class="package-card">
                    <div class="package-header">
                        <h3><?php echo htmlspecialchars($package['name']); ?></h3>
                        <div class="package-price">
                            <?php
                            $features = json_decode($package['features'], true);
                            $pricing_details = $features['pricing_details'] ?? [];
                            ?>
                            <?php if (!empty($pricing_details) && is_array($pricing_details) && is_numeric($pricing_details['one_time_license'])): ?>
                                ₵<?php echo number_format($pricing_details['one_time_license'], 2); ?><span>/year</span>
                            <?php elseif ($package['price_ghs']): ?>
                                ₵<?php echo number_format($package['price_ghs'], 2); ?><span>/year</span>
                            <?php else: ?>
                                Custom Pricing
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="package-body">
                        <p class="package-description"><?php echo htmlspecialchars($package['description']); ?></p>

                        <?php
                        $features = json_decode($package['features'], true);
                        if ($features):
                        ?>
                        <ul class="package-features">
                            <?php foreach ($features as $feature => $enabled): ?>
                                <?php if ($enabled): ?>
                                    <li><i class="fas fa-check"></i> <?php echo ucwords(str_replace('_', ' ', $feature)); ?></li>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </ul>
                        <?php endif; ?>

                        <div class="package-limits">
                            <p><strong>Max Students:</strong> <?php echo $package['max_students'] ? $package['max_students'] : 'Unlimited'; ?></p>
                            <p><strong>Max Teachers:</strong> <?php echo $package['max_teachers'] ? $package['max_teachers'] : 'Unlimited'; ?></p>
                        </div>

                        <?php if ($current_subscription && $current_subscription['package_id'] == $package['id']): ?>
                            <div class="current-package">
                                <i class="fas fa-star"></i> Current Package
                            </div>
                        <?php else: ?>
                            <a href="activate.php?package_id=<?php echo $package['id']; ?>" class="activate-btn">
                                <i class="fas fa-rocket"></i> Activate Package
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>

    <script src="../js/script.js"></script>
</body>
</html>
