<?php
require_once '../config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'student') {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch student information
$student_info = $pdo->prepare("SELECT * FROM students WHERE user_id = ?");
$student_info->execute([$user_id]);
$student = $student_info->fetch();

if (!$student) {
    echo "Student not found.";
    exit;
}

$student_id = $student['id'];
$student_class = $student['class'];

// Build query to get courses assigned to the student's class
$query = "
    SELECT DISTINCT
        c.*,
        s.name as subject_name,
        s.code as subject_code,
        t.name as teacher_name
    FROM courses c
    LEFT JOIN subjects s ON c.subject_id = s.id
    LEFT JOIN staff t ON c.teacher_id = t.id
    LEFT JOIN class_courses cc ON c.id = cc.course_id
    LEFT JOIN classes cl ON cc.class_id = cl.id
    WHERE cl.name = ? OR c.class_id = (SELECT id FROM classes WHERE name = ?)
    ORDER BY c.name ASC
";

$params = [$student_class, $student_class];

try {
    $courses_query = $pdo->prepare($query);
    $courses_query->execute($params);
    $enrolled_courses = $courses_query->fetchAll();

} catch (PDOException $e) {
    error_log("Error fetching courses for class: " . $e->getMessage());
    $enrolled_courses = [];
}

include '../includes/student_header.php';
?>

<div class="courses-container">
    <div class="page-header">
        <h1>📚 My Courses</h1>
        <p>View all the courses assigned to your class</p>
    </div>

    <!-- Courses Display -->
    <?php if (!empty($enrolled_courses)): ?>
        <div class="courses-section">
            <div class="courses-grid">
                <?php foreach ($enrolled_courses as $course): ?>
                    <div class="course-card">
                        <div class="course-header">
                            <div class="course-icon">📖</div>
                            <div class="course-info">
                                <h4><?php echo htmlspecialchars($course['name']); ?></h4>
                                <p class="subject-code"><?php echo htmlspecialchars($course['subject_code']); ?> - <?php echo htmlspecialchars($course['subject_name']); ?></p>
                            </div>
                        </div>

                        <div class="course-details">
                            <?php if ($course['description']): ?>
                                <div class="course-description">
                                    <strong>Description:</strong>
                                    <p><?php echo htmlspecialchars($course['description']); ?></p>
                                </div>
                            <?php endif; ?>

                            <div class="course-meta">
                                <div class="meta-item">
                                    <span class="meta-label">Credits:</span>
                                    <span class="meta-value"><?php echo htmlspecialchars($course['credits']); ?></span>
                                </div>

                                <?php if ($course['teacher_name']): ?>
                                    <div class="meta-item">
                                        <span class="meta-label">Teacher:</span>
                                        <span class="meta-value"><?php echo htmlspecialchars($course['teacher_name']); ?></span>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <div class="course-actions">
                            <a href="view_grades.php?course_id=<?php echo $course['id']; ?>" class="action-btn view-grades">
                                📊 View Grades
                            </a>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    <?php endif; ?>

    <?php if (empty($enrolled_courses)): ?>
        <div class="no-courses">
            <div class="no-courses-icon">📚</div>
            <h3>No Courses Found</h3>
            <p>You are not currently enrolled in any courses. Please contact your administrator if you believe this is an error.</p>
        </div>
    <?php endif; ?>
</div>

<style>
.courses-container {
    padding: 20px;
}

.page-header h1 {
    margin-bottom: 5px;
    color: #2c3e50;
}

.page-header p {
    color: #7f8c8d;
    margin-bottom: 30px;
}



.courses-section {
    margin-bottom: 40px;
}

.courses-section h3 {
    margin: 0 0 20px 0;
    color: #2c3e50;
    border-bottom: 2px solid #3498db;
    padding-bottom: 10px;
}

.courses-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 20px;
}

.course-card {
    background: #fff;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    border: 1px solid #e9ecef;
    overflow: hidden;
    transition: transform 0.3s ease, box-shadow 0.3s ease;
}

.course-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
}

.course-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    padding: 20px;
    display: flex;
    align-items: center;
    color: white;
}

.course-icon {
    font-size: 24px;
    margin-right: 15px;
}

.course-info h4 {
    margin: 0 0 5px 0;
    font-size: 1.2em;
    font-weight: 600;
}

.subject-code {
    margin: 0;
    opacity: 0.9;
    font-size: 0.9em;
}

.course-details {
    padding: 20px;
}

.course-description {
    margin-bottom: 20px;
}

.course-description strong {
    color: #2c3e50;
    display: block;
    margin-bottom: 5px;
}

.course-description p {
    color: #555;
    margin: 0;
    line-height: 1.5;
}

.course-meta {
    display: grid;
    grid-template-columns: 1fr;
    gap: 10px;
}

.meta-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 8px 0;
    border-bottom: 1px solid #f0f0f0;
}

.meta-item:last-child {
    border-bottom: none;
}

.meta-label {
    font-weight: 600;
    color: #2c3e50;
    font-size: 0.9em;
}

.meta-value {
    color: #34495e;
    text-align: right;
    font-size: 0.9em;
}

.course-actions {
    padding: 15px 20px;
    background: #f8f9fa;
    border-top: 1px solid #e9ecef;
}

.action-btn {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 8px 16px;
    background: #3498db;
    color: white;
    text-decoration: none;
    border-radius: 6px;
    font-size: 0.9em;
    font-weight: 500;
    transition: background-color 0.3s ease;
}

.action-btn:hover {
    background: #2980b9;
}

.no-courses {
    text-align: center;
    background: #fff;
    border-radius: 12px;
    padding: 60px 20px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
}

.no-courses-icon {
    font-size: 4em;
    margin-bottom: 20px;
    opacity: 0.5;
}

.no-courses h3 {
    margin: 0 0 10px 0;
    color: #2c3e50;
}

.no-courses p {
    margin: 0;
    color: #7f8c8d;
    max-width: 400px;
    margin: 0 auto;
}

/* Mobile Responsive */
@media (max-width: 768px) {
    .courses-container {
        padding: 15px;
    }

    .page-header h1 {
        font-size: 1.8em;
    }

    .courses-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }

    .course-card {
        margin: 0 -5px;
    }

    .course-header {
        padding: 15px;
    }

    .course-info h4 {
        font-size: 1.1em;
    }

    .course-details {
        padding: 15px;
    }

    .no-courses {
        padding: 40px 15px;
    }

    .no-courses-icon {
        font-size: 3em;
    }
}

@media (max-width: 480px) {
    .courses-container {
        padding: 10px;
    }

    .page-header h1 {
        font-size: 1.5em;
    }

    .filters-form {
        flex-direction: column;
        gap: 15px;
    }

    .filter-group {
        min-width: auto;
    }

    .filter-actions {
        justify-content: center;
    }

    .courses-grid {
        gap: 10px;
    }

    .course-header {
        padding: 12px;
    }

    .course-info h4 {
        font-size: 1em;
    }

    .course-details {
        padding: 12px;
    }

    .no-courses {
        padding: 30px 10px;
    }

    .no-courses h3 {
        font-size: 1.2em;
    }
}
</style>

<?php
include '../includes/student_footer.php';
?>
