<?php
require_once '../config.php';

// Check if user is logged in and is librarian or admin
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['librarian', 'admin'])) {
    header('Location: ../login.php');
    exit();
}

$message = '';
$message_type = '';
$books = [];
$search_performed = false;

// Handle search
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['search_books'])) {
    $title = trim($_POST['title'] ?? '');
    $author = trim($_POST['author'] ?? '');
    $isbn = trim($_POST['isbn'] ?? '');
    $category_id = intval($_POST['category_id'] ?? 0);

    $conditions = [];
    $params = [];

    if (!empty($title)) {
        $conditions[] = "lb.title LIKE ?";
        $params[] = "%$title%";
    }
    if (!empty($author)) {
        $conditions[] = "lb.author LIKE ?";
        $params[] = "%$author%";
    }
    if (!empty($isbn)) {
        $conditions[] = "lb.isbn LIKE ?";
        $params[] = "%$isbn%";
    }
    if ($category_id > 0) {
        $conditions[] = "lb.category_id = ?";
        $params[] = $category_id;
    }

    if (!empty($conditions)) {
        $where_clause = "WHERE " . implode(" AND ", $conditions);
        $search_performed = true;
    } else {
        $where_clause = "";
        $search_performed = true;
    }

    try {
        $stmt = $pdo->prepare("SELECT lb.*, lc.name as category_name FROM library_books lb LEFT JOIN library_categories lc ON lb.category_id = lc.id $where_clause ORDER BY lb.title");
        $stmt->execute($params);
        $books = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        $message = "Error searching books: " . $e->getMessage();
        $message_type = "error";
    }
}

// Fetch categories for dropdown
try {
    $categories_stmt = $pdo->query("SELECT * FROM library_categories ORDER BY name");
    $categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $categories = [];
}

$page_title = "Search Books - Library";
include '../includes/library_header.php';
?>

<div class="container-fluid">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2><i class="fas fa-search"></i> Search Books</h2>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
            <?php echo htmlspecialchars($message); ?>
            <button type="button" class="close" data-dismiss="alert">
                <span>&times;</span>
            </button>
        </div>
    <?php endif; ?>

    <!-- Search Form -->
    <div class="card mb-4">
        <div class="card-header">
            <h5><i class="fas fa-filter"></i> Search Criteria</h5>
        </div>
        <div class="card-body">
            <form method="POST" class="row g-3">
                <div class="col-md-3">
                    <label for="title" class="form-label">Title</label>
                    <input type="text" class="form-control" id="title" name="title" value="<?php echo htmlspecialchars($_POST['title'] ?? ''); ?>">
                </div>
                <div class="col-md-3">
                    <label for="author" class="form-label">Author</label>
                    <input type="text" class="form-control" id="author" name="author" value="<?php echo htmlspecialchars($_POST['author'] ?? ''); ?>">
                </div>
                <div class="col-md-3">
                    <label for="isbn" class="form-label">ISBN</label>
                    <input type="text" class="form-control" id="isbn" name="isbn" value="<?php echo htmlspecialchars($_POST['isbn'] ?? ''); ?>">
                </div>
                <div class="col-md-3">
                    <label for="category_id" class="form-label">Category</label>
                    <select class="form-control" id="category_id" name="category_id">
                        <option value="">All Categories</option>
                        <?php foreach ($categories as $category): ?>
                            <option value="<?php echo $category['id']; ?>" <?php echo (intval($_POST['category_id'] ?? 0) === $category['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($category['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-12">
                    <button type="submit" name="search_books" class="btn btn-primary">
                        <i class="fas fa-search"></i> Search Books
                    </button>
                    <button type="button" class="btn btn-secondary ml-2" onclick="clearSearch()">
                        <i class="fas fa-times"></i> Clear
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Search Results -->
    <?php if ($search_performed): ?>
        <div class="card">
            <div class="card-header">
                <h5><i class="fas fa-list"></i> Search Results (<?php echo count($books); ?> books found)</h5>
            </div>
            <div class="card-body">
                <?php if (empty($books)): ?>
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i> No books found matching your search criteria.
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead class="thead-dark">
                                <tr>
                                    <th>Title</th>
                                    <th>Author</th>
                                    <th>ISBN</th>
                                    <th>Category</th>
                                    <th>Publisher</th>
                                    <th>Year</th>
                                    <th>Total Copies</th>
                                    <th>Available Copies</th>
                                    <th>Location</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($books as $book): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($book['title']); ?></td>
                                        <td><?php echo htmlspecialchars($book['author']); ?></td>
                                        <td><?php echo htmlspecialchars($book['isbn']); ?></td>
                                        <td><?php echo htmlspecialchars($book['category_name']); ?></td>
                                        <td><?php echo htmlspecialchars($book['publisher']); ?></td>
                                        <td><?php echo htmlspecialchars($book['publication_year']); ?></td>
                                        <td><?php echo htmlspecialchars($book['total_copies']); ?></td>
                                        <td><?php echo htmlspecialchars($book['available_copies']); ?></td>
                                        <td><?php echo htmlspecialchars($book['location']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    <?php endif; ?>
</div>

<script>
function clearSearch() {
    document.getElementById('title').value = '';
    document.getElementById('author').value = '';
    document.getElementById('isbn').value = '';
    document.getElementById('category_id').value = '';
}
</script>

<?php include '../includes/library_footer.php'; ?>
