<?php
/**
 * Enhanced Access Control System
 * Provides role-based permission checking for the school management system
 */

// Check if user has specific permission
function hasPermission($permission_name) {
    if (!isset($_SESSION['user_id'])) {
        return false;
    }

    try {
        global $pdo;

        // Get user's roles (both primary role and assigned roles)
        $stmt = $pdo->prepare("
            SELECT DISTINCT r.role_name
            FROM roles r
            INNER JOIN role_permissions rp ON r.id = rp.role_id
            INNER JOIN permissions p ON rp.permission_id = p.id
            WHERE p.permission_name = ?
            AND (
                r.role_name = (SELECT role FROM users WHERE id = ?) OR
                r.id IN (SELECT role_id FROM user_roles WHERE user_id = ?)
            )
        ");

        $stmt->execute([$permission_name, $_SESSION['user_id'], $_SESSION['user_id']]);
        return $stmt->fetch() !== false;

    } catch (PDOException $e) {
        // If database is not available, fall back to basic role checking
        return hasBasicRoleAccess($permission_name);
    }
}

// Check if user has access to a feature (for backward compatibility) - RENAMED to avoid conflict
function hasFeatureAccessNew($feature_name) {
    // Map feature names to permissions
    $feature_permission_map = [
        'enrollment_management' => 'enrollment_management',
        'student_management' => 'student_management',
        'student_promotion' => 'student_promotion',
        'teacher_management' => 'teacher_management',
        'fee_management' => 'fee_management',
        'assign_fees' => 'assign_fees',
        'payment_management' => 'payment_management',
        'fee_reports' => 'fee_reports',
        'reports_analytics' => 'reports_analytics',
        'inventory_management' => 'inventory_management',
        'facilities_management' => 'facilities_management'
    ];

    if (isset($feature_permission_map[$feature_name])) {
        return hasPermission($feature_permission_map[$feature_name]);
    }

    return false;
}

// Basic role-based access checking (fallback when database is not available)
function hasBasicRoleAccess($permission_name) {
    if (!isset($_SESSION['user_id']) || !isset($_SESSION['role'])) {
        return false;
    }

    $user_role = $_SESSION['role'];

    // Define basic role permissions
    $basic_permissions = [
        // Dashboard access - Only admin and super_admin can access admin panel
        'dashboard_access' => ['super_admin', 'admin'],

        // User management
        'user_management' => ['super_admin', 'admin'],

        // Student management
        'enrollment_management' => ['super_admin', 'admin'],
        'student_management' => ['super_admin', 'admin', 'head_teacher', 'teacher'],
        'student_logins' => ['super_admin', 'admin'],
        'student_promotion' => ['super_admin', 'admin', 'head_teacher'],

        // Class management
        'class_management' => ['super_admin', 'admin', 'head_teacher'],

        // Course management
        'course_management' => ['super_admin', 'admin'],
        'subject_view' => ['super_admin', 'admin', 'head_teacher', 'teacher'],

        // Staff management
        'teacher_management' => ['super_admin', 'admin', 'head_teacher'],
        'librarian_management' => ['super_admin', 'admin'],
        'non_teaching_management' => ['super_admin', 'admin'],
        'director_management' => ['super_admin', 'admin'],
        'head_teacher_management' => ['super_admin', 'admin'],

        // Fee management
        'fee_management' => ['super_admin', 'admin', 'accountant'],
        'assign_fees' => ['super_admin', 'admin', 'accountant'],
        'payment_management' => ['super_admin', 'admin', 'accountant'],
        'fee_reports' => ['super_admin', 'admin', 'accountant'],

        // Academic management
        'academic_years' => ['super_admin', 'admin', 'head_teacher'],
        'terms_management' => ['super_admin', 'admin', 'head_teacher'],
        'levels_management' => ['super_admin', 'admin', 'head_teacher'],

        // Reports
        'reports_analytics' => ['super_admin', 'admin', 'head_teacher', 'accountant', 'director'],

        // Inventory
        'inventory_management' => ['super_admin', 'admin', 'librarian', 'non_teaching'],

        // Facilities
        'facilities_management' => ['super_admin', 'admin', 'librarian', 'non_teaching'],

        // System
        'grading_system' => ['super_admin', 'admin', 'head_teacher'],
        'settings' => ['super_admin', 'admin']
    ];

    return isset($basic_permissions[$permission_name]) &&
           in_array($user_role, $basic_permissions[$permission_name]);
}

// Check if user has any of the specified permissions
function hasAnyPermission($permissions) {
    if (!is_array($permissions)) {
        $permissions = [$permissions];
    }

    foreach ($permissions as $permission) {
        if (hasPermission($permission)) {
            return true;
        }
    }

    return false;
}

// Check if user has all of the specified permissions
function hasAllPermissions($permissions) {
    if (!is_array($permissions)) {
        $permissions = [$permissions];
    }

    foreach ($permissions as $permission) {
        if (!hasPermission($permission)) {
            return false;
        }
    }

    return true;
}

// Get user's accessible menu items based on permissions
function getAccessibleMenuItems() {
    $menu_items = [
        'dashboard' => [
            'name' => 'Dashboard',
            'icon' => '📊',
            'url' => 'admin_dashboard.php',
            'permission' => 'dashboard_access'
        ],
        'user_management' => [
            'name' => 'User Management',
            'icon' => '👥',
            'url' => 'user_management.php',
            'permission' => 'user_management'
        ],
        'students' => [
            'name' => 'Students',
            'icon' => '👨‍🎓',
            'submenu' => [
                'enroll_students' => [
                    'name' => 'Enroll Students',
                    'url' => 'enroll_students.php',
                    'permission' => 'enrollment_management'
                ],
                'manage_students' => [
                    'name' => 'Manage Students',
                    'url' => 'manage_students.php',
                    'permission' => 'student_management'
                ],
                'student_logins' => [
                    'name' => 'Student Logins',
                    'url' => 'manage_student_logins.php',
                    'permission' => 'student_logins'
                ],
                'promote_students' => [
                    'name' => 'Promote Students',
                    'url' => 'promote_students.php',
                    'permission' => 'student_promotion'
                ]
            ]
        ],
        'classes' => [
            'name' => 'Class Management',
            'icon' => '🏫',
            'submenu' => [
                'view_classes' => [
                    'name' => 'View Classes',
                    'url' => 'view_classes.php',
                    'permission' => 'class_management'
                ]
            ]
        ],
        'courses' => [
            'name' => 'Courses / Subjects',
            'icon' => '📚',
            'submenu' => [
                'manage_courses' => [
                    'name' => 'Add Courses/Subjects',
                    'url' => 'manage_courses.php',
                    'permission' => 'course_management'
                ],
                'view_subjects' => [
                    'name' => 'View Subjects/Courses',
                    'url' => 'subjects.php',
                    'permission' => 'subject_view'
                ]
            ]
        ],
        'staff' => [
            'name' => 'Staff',
            'icon' => '👥',
            'submenu' => [
                'teachers' => [
                    'name' => 'Teachers',
                    'submenu' => [
                        'manage_teachers' => [
                            'name' => 'Manage Teachers',
                            'url' => 'manage_teachers.php',
                            'permission' => 'teacher_management'
                        ],
                        'assign_class' => [
                            'name' => 'Assign Class',
                            'url' => 'assign_class.php',
                            'permission' => 'teacher_management'
                        ]
                    ]
                ],
                'librarians' => [
                    'name' => 'Librarians',
                    'url' => 'manage_librarians.php',
                    'permission' => 'librarian_management'
                ],
                'non_teaching' => [
                    'name' => 'Non-teaching',
                    'url' => 'manage_non_teaching.php',
                    'permission' => 'non_teaching_management'
                ],
                'directors' => [
                    'name' => 'Directors',
                    'url' => 'manage_directors.php',
                    'permission' => 'director_management'
                ],
                'head_teachers' => [
                    'name' => 'Head Teachers',
                    'url' => 'manage_head_teachers.php',
                    'permission' => 'head_teacher_management'
                ]
            ]
        ],
        'fees' => [
            'name' => 'Fees Management',
            'icon' => '💰',
            'submenu' => [
                'create_fees' => [
                    'name' => 'Create Fees',
                    'url' => 'add_fee.php',
                    'permission' => 'fee_management'
                ],
                'assign_fees' => [
                    'name' => 'Assign Fees',
                    'url' => 'assign_fees.php',
                    'permission' => 'assign_fees'
                ],
                'record_payments' => [
                    'name' => 'Record Payments',
                    'url' => 'record_payments.php',
                    'permission' => 'payment_management'
                ],
                'fee_reports' => [
                    'name' => 'Fee Reports',
                    'url' => 'fee_reports.php',
                    'permission' => 'fee_reports'
                ]
            ]
        ],
        'academic' => [
            'name' => 'Academic Management',
            'icon' => '📅',
            'submenu' => [
                'academic_years' => [
                    'name' => 'Academic Years',
                    'url' => 'manage_academic_years.php',
                    'permission' => 'academic_years'
                ],
                'terms' => [
                    'name' => 'Terms/Semesters',
                    'url' => 'manage_terms.php',
                    'permission' => 'terms_management'
                ],
                'levels' => [
                    'name' => 'Manage Levels',
                    'url' => 'manage_levels.php',
                    'permission' => 'levels_management'
                ]
            ]
        ],
        'reports' => [
            'name' => 'Reports',
            'icon' => '📈',
            'url' => 'view_reports.php',
            'permission' => 'reports_analytics'
        ],
        'inventory' => [
            'name' => 'Inventory',
            'icon' => '📦',
            'submenu' => [
                'manage_items' => [
                    'name' => 'Manage Items',
                    'url' => 'manage_items.php',
                    'permission' => 'inventory_management'
                ],
                'stock_management' => [
                    'name' => 'Stock Management',
                    'url' => 'manage_stock.php',
                    'permission' => 'inventory_management'
                ]
            ]
        ],
        'facilities' => [
            'name' => 'Facilities',
            'icon' => '🏢',
            'submenu' => [
                'manage_facilities' => [
                    'name' => 'Manage Facilities',
                    'url' => 'manage_facilities.php',
                    'permission' => 'facilities_management'
                ],
                'booking_management' => [
                    'name' => 'Booking Management',
                    'url' => 'manage_bookings.php',
                    'permission' => 'facilities_management'
                ]
            ]
        ],
        'system' => [
            'name' => 'System',
            'icon' => '⚙️',
            'submenu' => [
                'grading_system' => [
                    'name' => 'Grading System',
                    'url' => 'manage_grading_system.php',
                    'permission' => 'grading_system'
                ],
                'settings' => [
                    'name' => 'Settings',
                    'url' => 'settings.php',
                    'permission' => 'settings'
                ]
            ]
        ]
    ];

    $accessible_items = [];

    foreach ($menu_items as $key => $item) {
        if (isset($item['permission'])) {
            if (hasPermission($item['permission'])) {
                $accessible_items[$key] = $item;
            }
        } elseif (isset($item['submenu'])) {
            // Check if user has access to any submenu item
            $has_access = false;
            $accessible_submenu = [];

            foreach ($item['submenu'] as $sub_key => $sub_item) {
                if (isset($sub_item['permission']) && hasPermission($sub_item['permission'])) {
                    $accessible_submenu[$sub_key] = $sub_item;
                    $has_access = true;
                } elseif (isset($sub_item['submenu'])) {
                    // Handle nested submenus
                    $accessible_nested = [];
                    foreach ($sub_item['submenu'] as $nested_key => $nested_item) {
                        if (isset($nested_item['permission']) && hasPermission($nested_item['permission'])) {
                            $accessible_nested[$nested_key] = $nested_item;
                        }
                    }
                    if (!empty($accessible_nested)) {
                        $sub_item['submenu'] = $accessible_nested;
                        $accessible_submenu[$sub_key] = $sub_item;
                        $has_access = true;
                    }
                }
            }

            if ($has_access) {
                $item['submenu'] = $accessible_submenu;
                $accessible_items[$key] = $item;
            }
        }
    }

    return $accessible_items;
}

// Get user's role information
function getUserRoles() {
    if (!isset($_SESSION['user_id'])) {
        return [];
    }

    try {
        global $pdo;

        $stmt = $pdo->prepare("
            SELECT DISTINCT r.role_name, r.description
            FROM roles r
            INNER JOIN role_permissions rp ON r.id = rp.role_id
            INNER JOIN permissions p ON rp.permission_id = p.id
            WHERE r.role_name = (SELECT role FROM users WHERE id = ?)
            OR r.id IN (SELECT role_id FROM user_roles WHERE user_id = ?)
        ");

        $stmt->execute([$_SESSION['user_id'], $_SESSION['user_id']]);
        return $stmt->fetchAll();

    } catch (PDOException $e) {
        // Return basic role if database is not available
        return [['role_name' => $_SESSION['role'], 'description' => 'Basic role']];
    }
}

// Check if user is super admin
function isSuperAdmin() {
    return isset($_SESSION['role']) && $_SESSION['role'] === 'super_admin';
}

// Check if user is admin or super admin
function isAdmin() {
    return isset($_SESSION['role']) && ($_SESSION['role'] === 'admin' || $_SESSION['role'] === 'super_admin');
}

// Require specific permission or redirect
function requirePermission($permission_name, $redirect_url = '../login.php') {
    if (!hasPermission($permission_name)) {
        header('Location: ' . $redirect_url);
        exit;
    }
}

// Require any of the specified permissions or redirect
function requireAnyPermission($permissions, $redirect_url = '../login.php') {
    if (!hasAnyPermission($permissions)) {
        header('Location: ' . $redirect_url);
        exit;
    }
}
?>
