<?php
require_once '../config.php';

$message = '';
$error = '';

// Create uploads directory if it doesn't exist
$upload_dir = '../uploads/logos/';
if (!file_exists($upload_dir)) {
    mkdir($upload_dir, 0777, true);
}

// Fetch existing settings (assuming only one row)
$stmt = $pdo->query("SELECT * FROM school_settings LIMIT 1");
$settings = $stmt->fetch();

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $school_name = trim($_POST['school_name']);
    $short_name = trim($_POST['short_name']);
    $address = trim($_POST['address']);
    $phone = trim($_POST['phone']);
    $email = trim($_POST['email']);
    $website = trim($_POST['website']);
    $logo_path = $settings['logo_path'] ?? '';

    // Handle logo upload
    if (isset($_FILES['logo']) && $_FILES['logo']['error'] == 0) {
        $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
        $max_size = 2 * 1024 * 1024; // 2MB

        if (!in_array($_FILES['logo']['type'], $allowed_types)) {
            $error = 'Invalid file type. Only JPG, PNG, and GIF are allowed.';
        } elseif ($_FILES['logo']['size'] > $max_size) {
            $error = 'File size too large. Maximum size is 2MB.';
        } else {
            $file_extension = pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
            $new_filename = 'school_logo_' . time() . '.' . $file_extension;
            $upload_path = $upload_dir . $new_filename;

            if (move_uploaded_file($_FILES['logo']['tmp_name'], $upload_path)) {
                // Delete old logo if exists
                if ($logo_path && file_exists('../' . $logo_path)) {
                    unlink('../' . $logo_path);
                }
                $logo_path = 'uploads/logos/' . $new_filename;
            } else {
                $error = 'Failed to upload logo.';
            }
        }
    }

    if (!$error) {
        try {
            if ($settings) {
                $stmt = $pdo->prepare("UPDATE school_settings SET school_name = ?, short_name = ?, address = ?, phone = ?, email = ?, website = ?, logo_path = ? WHERE id = ?");
                $stmt->execute([$school_name, $short_name, $address, $phone, $email, $website, $logo_path, $settings['id']]);
            } else {
                $stmt = $pdo->prepare("INSERT INTO school_settings (school_name, short_name, address, phone, email, website, logo_path) VALUES (?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$school_name, $short_name, $address, $phone, $email, $website, $logo_path]);
            }
            $message = 'Settings saved successfully.';
            // Refresh settings
            $stmt = $pdo->query("SELECT * FROM school_settings LIMIT 1");
            $settings = $stmt->fetch();
        } catch (PDOException $e) {
            $error = 'Error saving settings: ' . $e->getMessage();
        }
    }
}

include '../includes/admin_header.php';
?>

<style>
.settings-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
}

.settings-header {
    text-align: center;
    margin-bottom: 40px;
    padding: 30px 0;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 15px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
}

.settings-header h1 {
    margin: 0 0 10px 0;
    font-size: 2.5em;
    font-weight: 700;
}

.settings-header p {
    margin: 0;
    font-size: 1.1em;
    opacity: 0.9;
}

.settings-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: 30px;
    margin-bottom: 30px;
}

.settings-card {
    background: white;
    border-radius: 15px;
    padding: 30px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
    border: 1px solid #f0f0f0;
    transition: transform 0.3s ease, box-shadow 0.3s ease;
}

.settings-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 15px 35px rgba(0,0,0,0.15);
}

.card-header {
    display: flex;
    align-items: center;
    margin-bottom: 25px;
    padding-bottom: 15px;
    border-bottom: 2px solid #f0f0f0;
}

.card-icon {
    font-size: 2.5em;
    margin-right: 15px;
    color: #667eea;
}

.card-title {
    margin: 0;
    font-size: 1.4em;
    font-weight: 600;
    color: #2c3e50;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #333;
    font-size: 0.95em;
}

.form-group input[type="text"],
.form-group input[type="email"],
.form-group input[type="url"],
.form-group textarea {
    width: 100%;
    padding: 12px 15px;
    border: 2px solid #e1e8ed;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.3s ease, box-shadow 0.3s ease;
}

.form-group input[type="text"]:focus,
.form-group input[type="email"]:focus,
.form-group input[type="url"]:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.form-group textarea {
    resize: vertical;
    min-height: 100px;
}

.form-group small {
    display: block;
    margin-top: 5px;
    color: #666;
    font-size: 0.85em;
}

.logo-section {
    background: #f8f9fa;
    border-radius: 10px;
    padding: 20px;
    margin-bottom: 20px;
}

.logo-upload-area {
    border: 2px dashed #667eea;
    border-radius: 10px;
    padding: 30px;
    text-align: center;
    transition: border-color 0.3s ease;
    cursor: pointer;
    position: relative;
}

.logo-upload-area:hover {
    border-color: #764ba2;
}

.logo-upload-area input[type="file"] {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    opacity: 0;
    cursor: pointer;
}

.upload-icon {
    font-size: 3em;
    color: #667eea;
    margin-bottom: 10px;
}

.upload-text {
    color: #666;
    margin-bottom: 5px;
}

.upload-hint {
    color: #999;
    font-size: 0.9em;
}

.current-logo-display {
    margin-top: 20px;
    text-align: center;
}

.current-logo-display img {
    max-width: 150px;
    max-height: 100px;
    border-radius: 8px;
    border: 2px solid #e1e8ed;
    padding: 5px;
}

.btn {
    display: inline-block;
    padding: 12px 30px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    text-decoration: none;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-size: 16px;
    font-weight: 600;
    transition: transform 0.3s ease, box-shadow 0.3s ease;
}

.btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
}

.message {
    background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
    color: #155724;
    padding: 15px 20px;
    border-radius: 8px;
    margin-bottom: 20px;
    border-left: 4px solid #28a745;
    display: flex;
    align-items: center;
}

.message::before {
    content: "✓";
    font-size: 1.5em;
    margin-right: 10px;
}

.error-message {
    background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
    color: #721c24;
    padding: 15px 20px;
    border-radius: 8px;
    margin-bottom: 20px;
    border-left: 4px solid #dc3545;
    display: flex;
    align-items: center;
}

.error-message::before {
    content: "⚠";
    font-size: 1.5em;
    margin-right: 10px;
}

.save-section {
    text-align: center;
    margin-top: 40px;
    padding: 30px;
    background: white;
    border-radius: 15px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
}

.save-section h3 {
    margin: 0 0 15px 0;
    color: #2c3e50;
    font-size: 1.5em;
}

.save-section p {
    margin: 0 0 25px 0;
    color: #7f8c8d;
}

@media (max-width: 768px) {
    .settings-grid {
        grid-template-columns: 1fr;
        gap: 20px;
    }

    .settings-card {
        padding: 20px;
    }

    .settings-header h1 {
        font-size: 2em;
    }
}
</style>

<div class="settings-container">
    <div class="settings-header">
        <h1>⚙️ School Settings</h1>
        <p>Configure your school's information and branding</p>
    </div>

    <?php if ($message): ?>
        <div class="message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <form method="post" action="settings.php" enctype="multipart/form-data">
        <div class="settings-grid">
            <!-- Basic Information -->
            <div class="settings-card">
                <div class="card-header">
                    <div class="card-icon">🏫</div>
                    <h2 class="card-title">Basic Information</h2>
                </div>

                <div class="form-group">
                    <label for="school_name">School Name</label>
                    <input type="text" id="school_name" name="school_name" value="<?php echo htmlspecialchars($settings['school_name'] ?? ''); ?>" required>
                </div>

                <div class="form-group">
                    <label for="short_name">Short Name</label>
                    <input type="text" id="short_name" name="short_name" value="<?php echo htmlspecialchars($settings['short_name'] ?? ''); ?>" placeholder="e.g., SMS, SIS">
                    <small>Abbreviated name for the school (optional)</small>
                </div>

                <div class="form-group">
                    <label for="address">Address</label>
                    <textarea id="address" name="address"><?php echo htmlspecialchars($settings['address'] ?? ''); ?></textarea>
                </div>
            </div>

            <!-- Contact Information -->
            <div class="settings-card">
                <div class="card-header">
                    <div class="card-icon">📞</div>
                    <h2 class="card-title">Contact Information</h2>
                </div>

                <div class="form-group">
                    <label for="phone">Phone Number</label>
                    <input type="text" id="phone" name="phone" value="<?php echo htmlspecialchars($settings['phone'] ?? ''); ?>">
                </div>

                <div class="form-group">
                    <label for="email">Email Address</label>
                    <input type="email" id="email" name="email" value="<?php echo htmlspecialchars($settings['email'] ?? ''); ?>">
                </div>

                <div class="form-group">
                    <label for="website">Website</label>
                    <input type="url" id="website" name="website" value="<?php echo htmlspecialchars($settings['website'] ?? ''); ?>" placeholder="https://www.example.com">
                </div>
            </div>

            <!-- Logo Upload -->
            <div class="settings-card">
                <div class="card-header">
                    <div class="card-icon">🖼️</div>
                    <h2 class="card-title">School Logo</h2>
                </div>

                <div class="logo-section">
                    <div class="logo-upload-area">
                        <input type="file" id="logo" name="logo" accept="image/*" onchange="previewLogo(event)">
                        <div class="upload-icon">📤</div>
                        <div class="upload-text">Click to upload school logo</div>
                        <div class="upload-hint">JPG, PNG, GIF up to 2MB</div>
                    </div>

                    <?php if (isset($settings['logo_path']) && $settings['logo_path']): ?>
                        <div class="current-logo-display">
                            <p><strong>Current Logo:</strong></p>
                            <img id="currentLogoPreview" src="../<?php echo htmlspecialchars($settings['logo_path']); ?>" alt="Current Logo">
                        </div>
                    <?php endif; ?>

                    <div class="logo-preview" id="logoPreview" style="display: none; margin-top: 20px;">
                        <p><strong>Preview:</strong></p>
                        <img id="previewImage" src="" alt="Logo Preview" style="max-width: 150px; max-height: 100px; border-radius: 8px; border: 2px solid #e1e8ed; padding: 5px;">
                    </div>
                </div>
            </div>
        </div>

        <!-- Save Button -->
        <div class="save-section">
            <h3>💾 Save Changes</h3>
            <p>Click the button below to save your school settings</p>
            <button type="submit" class="btn btn-primary">Save Settings</button>
        </div>
    </form>
</div>

<script>
function previewLogo(event) {
    const file = event.target.files[0];
    const preview = document.getElementById('logoPreview');
    const previewImage = document.getElementById('previewImage');

    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            previewImage.src = e.target.result;
            preview.style.display = 'block';
        };
        reader.readAsDataURL(file);
    } else {
        preview.style.display = 'none';
    }
}
</script>

<?php
include '../includes/admin_footer.php';
?>
