 <?php
require_once '../config.php';

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['record_payment'])) {
        $student_fee_id = $_POST['student_fee_id'];
        $payment_amount = floatval($_POST['payment_amount']);
        $payment_method = $_POST['payment_method'];
        $payment_date = $_POST['payment_date'];
        $receipt_number = 'RCP-' . date('Ymd') . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);

        try {
            // Get student fee details
            $stmt = $pdo->prepare("SELECT sf.*, s.name as student_name, f.name as fee_name, f.amount as fee_amount FROM student_fees sf JOIN students s ON sf.student_id = s.id JOIN fees f ON sf.fee_id = f.id WHERE sf.id = ?");
            $stmt->execute([$student_fee_id]);
            $student_fee = $stmt->fetch();

            if ($student_fee) {
                $total_fee_amount = floatval($student_fee['fee_amount']);
                $current_paid_amount = floatval($student_fee['paid_amount'] ?? 0);
                $new_paid_amount = $current_paid_amount + $payment_amount;

                // Determine payment status
                if ($new_paid_amount >= $total_fee_amount) {
                    $paid_status = TRUE;
                    $remaining_balance = 0;
                    $status_text = 'Fully Paid';
                } else {
                    $paid_status = FALSE;
                    $remaining_balance = $total_fee_amount - $new_paid_amount;
                    $status_text = 'Partially Paid';
                }

                // Update payment status and amounts
                $stmt = $pdo->prepare("UPDATE student_fees SET paid = ?, paid_at = ?, payment_method = ?, receipt_number = ?, paid_amount = ? WHERE id = ?");
                $stmt->execute([$paid_status, $payment_date, $payment_method, $receipt_number, $new_paid_amount, $student_fee_id]);

                $message = "Payment recorded successfully! Receipt: $receipt_number - Status: $status_text";

                if ($remaining_balance > 0) {
                    $message .= " - Remaining Balance: GHC " . number_format($remaining_balance, 2);
                }

                // Store receipt data in session for printing
                $_SESSION['receipt_data'] = [
                    'receipt_number' => $receipt_number,
                    'student_name' => $student_fee['student_name'],
                    'fee_name' => $student_fee['fee_name'],
                    'amount' => $payment_amount,
                    'total_paid' => $new_paid_amount,
                    'remaining_balance' => $remaining_balance,
                    'payment_date' => $payment_date,
                    'payment_method' => $payment_method,
                    'status' => $status_text
                ];
            } else {
                $error = 'Student fee record not found.';
            }
        } catch (PDOException $e) {
            $error = 'Error recording payment: ' . $e->getMessage();
        }
    }
}

$filter_search = $_GET['search'] ?? '';
$filter_class = $_GET['class'] ?? '';
$filter_academic_year = $_GET['academic_year'] ?? '';
$filter_fee_type = $_GET['fee_type'] ?? '';
$filter_status = $_GET['status'] ?? 'unpaid';

$query = "SELECT sf.*, s.name as student_name, s.class, s.academic_year, f.name as fee_name, f.amount as fee_amount FROM student_fees sf JOIN students s ON sf.student_id = s.id JOIN fees f ON sf.fee_id = f.id WHERE 1=1";
$params = [];

if ($filter_search) {
    $query .= " AND s.name LIKE ?";
    $params[] = '%' . $filter_search . '%';
}
if ($filter_class) {
    $query .= " AND s.class = ?";
    $params[] = $filter_class;
}
if ($filter_academic_year) {
    $query .= " AND s.academic_year = ?";
    $params[] = $filter_academic_year;
}
if ($filter_fee_type) {
    $query .= " AND f.name = ?";
    $params[] = $filter_fee_type;
}

if ($filter_status === 'paid') {
    $query .= " AND sf.paid = TRUE";
} elseif ($filter_status === 'unpaid') {
    $query .= " AND sf.paid = FALSE";
} elseif ($filter_status === 'partial') {
    $query .= " AND sf.paid = FALSE AND sf.paid_amount > 0";
}

$query .= " ORDER BY sf.due_date ASC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$student_fees = $stmt->fetchAll();

$classes_stmt = $pdo->query("SELECT name FROM classes ORDER BY name");
$classes = $classes_stmt->fetchAll(PDO::FETCH_COLUMN);

$years_stmt = $pdo->query("SELECT year FROM academic_years ORDER BY year DESC");
$academic_years = $years_stmt->fetchAll(PDO::FETCH_COLUMN);

include '../includes/admin_header.php';
?>

<link rel="stylesheet" href="../css/status.css">

<h2>Record Fee Payments</h2>

<?php if ($message): ?>
    <div class="message"><?php echo htmlspecialchars($message); ?>
        <?php if (isset($_SESSION['receipt_data'])): ?>
            <a href="print_receipt.php" target="_blank" class="btn btn-small">Print Receipt</a>
        <?php endif; ?>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<form method="get" action="record_payments.php" class="filter-form">
    <label for="search">Search Student:</label>
    <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($_GET['search'] ?? ''); ?>" placeholder="Enter student name">

    <label for="class">Filter by Class:</label>
    <select id="class" name="class">
        <option value="">All Classes</option>
        <?php foreach ($classes as $class): ?>
            <option value="<?php echo htmlspecialchars($class); ?>" <?php if ($class == $filter_class) echo 'selected'; ?>>
                <?php echo htmlspecialchars($class); ?>
            </option>
        <?php endforeach; ?>
    </select>

    <label for="academic_year">Filter by Academic Year:</label>
    <select id="academic_year" name="academic_year">
        <option value="">All Years</option>
        <?php foreach ($academic_years as $year): ?>
            <option value="<?php echo htmlspecialchars($year); ?>" <?php if ($year == $filter_academic_year) echo 'selected'; ?>>
                <?php echo htmlspecialchars($year); ?>
            </option>
        <?php endforeach; ?>
    </select>

    <label for="fee_type">Filter by Fee Type:</label>
    <select id="fee_type" name="fee_type">
        <option value="">All Fee Types</option>
        <?php
        $fee_types_stmt = $pdo->query("SELECT DISTINCT name FROM fees ORDER BY name");
        $fee_types = $fee_types_stmt->fetchAll(PDO::FETCH_COLUMN);
        $filter_fee_type = $_GET['fee_type'] ?? '';
        foreach ($fee_types as $fee_type):
        ?>
            <option value="<?php echo htmlspecialchars($fee_type); ?>" <?php if ($fee_type == $filter_fee_type) echo 'selected'; ?>>
                <?php echo htmlspecialchars($fee_type); ?>
            </option>
        <?php endforeach; ?>
    </select>

    <label for="status">Payment Status:</label>
    <select id="status" name="status">
        <option value="unpaid" <?php if ($filter_status == 'unpaid') echo 'selected'; ?>>Unpaid</option>
        <option value="partial" <?php if ($filter_status == 'partial') echo 'selected'; ?>>Partially Paid</option>
        <option value="paid" <?php if ($filter_status == 'paid') echo 'selected'; ?>>Paid</option>
        <option value="all" <?php if ($filter_status == 'all') echo 'selected'; ?>>All</option>
    </select>

    <button type="submit" class="btn btn-secondary">Filter</button>
    <a href="record_payments.php" class="btn">Clear Filters</a>
</form>

<style>
    .payments-table {
    background: white;
    border-radius: 8px;
    overflow: hidden;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.payments-table table {
    width: 100%;
    border-collapse: collapse;
}

.payments-table th,
.payments-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.payments-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.payments-table tr:hover {
    background: #f8f9fa;
}
</style>

<div class="payments-table">
    <table class="payments-table">
        <thead>
            <tr>
                <th>Student Name</th>
                <th>Class</th>
                <th>Academic Year</th>
                <th>Fee Name</th>
                <th>Amount</th>
                <th>Balance</th>
                <th>Due Date</th>
                <th>Status</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($student_fees as $fee): ?>
                <?php
                    $remaining_balance = $fee['fee_amount'] - ($fee['paid_amount'] ?? 0);
                ?>
                <tr>
                    <td><?php echo htmlspecialchars($fee['student_name']); ?></td>
                    <td><?php echo htmlspecialchars($fee['class']); ?></td>
                    <td><?php echo htmlspecialchars($fee['academic_year']); ?></td>
                    <td><?php echo htmlspecialchars($fee['fee_name']); ?></td>
                    <td>GHC <?php echo number_format($fee['fee_amount'], 2); ?></td>
                    <td>GHC <?php echo number_format($remaining_balance, 2); ?></td>
                    <td><?php echo htmlspecialchars($fee['due_date']); ?></td>
                    <td>
                        <?php
                            if ($fee['paid']) {
                                $status_class = 'paid';
                                $status_text = 'Paid';
                            } elseif ($fee['paid_amount'] > 0) {
                                $status_class = 'partial';
                                $status_text = 'Partially Paid';
                            } else {
                                $status_class = 'unpaid';
                                $status_text = 'Unpaid';
                            }
                        ?>
                        <span class="status <?php echo $status_class; ?>">
                            <?php echo $status_text; ?>
                        </span>
                    </td>
                    <td>
                        <?php if (!$fee['paid']): ?>
                            <button type="button" class="btn btn-small btn-primary" onclick="openPaymentModal(<?php echo $fee['id']; ?>, '<?php echo htmlspecialchars($fee['student_name']); ?>', '<?php echo htmlspecialchars($fee['fee_name']); ?>', <?php echo $remaining_balance; ?>)">
                                Record Payment
                            </button>
                        <?php else: ?>
                            <span class="text-muted">Paid on <?php echo htmlspecialchars($fee['paid_at']); ?></span>
                            <?php if ($fee['receipt_number']): ?>
                                <br><small>Receipt: <?php echo htmlspecialchars($fee['receipt_number']); ?></small>
                            <?php endif; ?>
                        <?php endif; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>

<!-- Payment Modal -->
<div id="paymentModal" class="modal">
    <div class="modal-content">
        <span class="close" onclick="closePaymentModal()">&times;</span>
        <h3>Record Payment</h3>
        <form method="post" action="record_payments.php" id="paymentForm">
            <input type="hidden" name="student_fee_id" id="student_fee_id">

            <div class="form-group">
                <label>Student: <span id="modal_student_name"></span></label>
            </div>

            <div class="form-group">
                <label>Fee: <span id="modal_fee_name"></span></label>
            </div>

            <div class="form-group">
                <label for="payment_amount">Payment Amount (GHC):</label>
                <input type="number" id="payment_amount" name="payment_amount" step="0.01" required>
            </div>

            <div class="form-group">
                <label for="payment_method">Payment Method:</label>
                <select id="payment_method" name="payment_method" required>
                    <option value="">Select Method</option>
                    <option value="Cash">Cash</option>
                    <option value="Bank Transfer">Bank Transfer</option>
                    <option value="Mobile Money">Mobile Money</option>
                    <option value="Cheque">Cheque</option>
                </select>
            </div>

            <div class="form-group">
                <label for="payment_date">Payment Date:</label>
                <input type="date" id="payment_date" name="payment_date" value="<?php echo date('Y-m-d'); ?>" required>
            </div>

            <button type="submit" name="record_payment" class="btn btn-primary">Record Payment</button>
        </form>
    </div>
</div>

<script>
function openPaymentModal(feeId, studentName, feeName, amount) {
    document.getElementById('student_fee_id').value = feeId;
    document.getElementById('modal_student_name').textContent = studentName;
    document.getElementById('modal_fee_name').textContent = feeName;
    document.getElementById('payment_amount').value = amount;
    document.getElementById('paymentModal').style.display = 'block';
}

function closePaymentModal() {
    document.getElementById('paymentModal').style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('paymentModal');
    if (event.target == modal) {
        modal.style.display = 'none';
    }
}
</script>

<style>
.filter-form {
    background: #f8f9fa;
    padding: 20px;
    border-radius: 8px;
    margin-bottom: 20px;
    display: flex;
    flex-wrap: wrap;
    gap: 15px;
    align-items: end;
}

.filter-form label {
    display: block;
    margin-bottom: 5px;
    font-weight: 600;
    color: #495057;
}

.filter-form input[type="text"],
.filter-form select {
    padding: 8px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
    width: 150px;
}

.filter-form input[type="text"]:focus,
.filter-form select:focus {
    outline: none;
    border-color: #007bff;
    box-shadow: 0 0 0 2px rgba(0,123,255,0.25);
}

.payments-table {
    margin-top: 20px;
}

.status {
    padding: 4px 8px;
    border-radius: 4px;
    font-weight: bold;
}

.status.paid {
    background: #d4edda;
    color: #155724;
}

.status.partial {
    background: #fff3cd;
    color: #856404;
}

.status.unpaid {
    background: #f8d7da;
    color: #721c24;
}

.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.4);
}

.modal-content {
    background-color: #fefefe;
    margin: 15% auto;
    padding: 20px;
    border: 1px solid #888;
    width: 90%;
    max-width: 500px;
    border-radius: 8px;
}

.close {
    color: #aaa;
    float: right;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
}

.close:hover {
    color: black;
}

.btn-small {
    padding: 5px 10px;
    font-size: 0.9em;
}

.text-muted {
    color: #6c757d;
    font-size: 0.9em;
}
</style>

<?php
include '../includes/admin_footer.php';
?>
