<?php
require_once '../config.php';

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

// Handle form submissions
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_route'])) {
        // Add new route
        $route_name = trim($_POST['route_name']);
        $route_number = trim($_POST['route_number']);
        $description = trim($_POST['description']);
        $start_point = trim($_POST['start_point']);
        $end_point = trim($_POST['end_point']);
        $distance = floatval($_POST['distance']);
        $duration = intval($_POST['duration']);
        $capacity = intval($_POST['capacity']);

        try {
            $stmt = $pdo->prepare("INSERT INTO transportation_routes (route_name, route_number, description, start_point, end_point, distance_km, estimated_duration, bus_capacity, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$route_name, $route_number, $description, $start_point, $end_point, $distance, $duration, $capacity, $_SESSION['user_id']]);
            $message = "Route added successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error adding route: " . $e->getMessage();
            $message_type = "error";
        }
    } elseif (isset($_POST['add_stop'])) {
        // Add stop to route
        $route_id = intval($_POST['route_id']);
        $stop_name = trim($_POST['stop_name']);
        $stop_order = intval($_POST['stop_order']);
        $pickup_time = $_POST['pickup_time'] ?: null;
        $dropoff_time = $_POST['dropoff_time'] ?: null;

        try {
            $stmt = $pdo->prepare("INSERT INTO transportation_stops (route_id, stop_name, stop_order, pickup_time, dropoff_time) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$route_id, $stop_name, $stop_order, $pickup_time, $dropoff_time]);
            $message = "Stop added successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error adding stop: " . $e->getMessage();
            $message_type = "error";
        }
    } elseif (isset($_POST['assign_student'])) {
        // Assign student to route
        $student_id = intval($_POST['student_id']);
        $route_id = intval($_POST['route_id']);
        $stop_id = intval($_POST['stop_id']);
        $assignment_type = $_POST['assignment_type'];
        $start_date = $_POST['start_date'];
        $end_date = $_POST['end_date'];
        $emergency_contact = trim($_POST['emergency_contact']);
        $medical_conditions = trim($_POST['medical_conditions']);

        try {
            $stmt = $pdo->prepare("INSERT INTO transportation_assignments (student_id, route_id, stop_id, assignment_type, start_date, end_date, emergency_contact, medical_conditions, assigned_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$student_id, $route_id, $stop_id, $assignment_type, $start_date, $end_date, $emergency_contact, $medical_conditions, $_SESSION['user_id']]);
            $message = "Student assigned to route successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error assigning student: " . $e->getMessage();
            $message_type = "error";
        }
    }
}

// Get all routes
try {
    $routes_stmt = $pdo->query("SELECT * FROM transportation_routes ORDER BY route_name");
    $routes = $routes_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $routes = [];
}

// Get all students for assignment
try {
    $students_stmt = $pdo->query("SELECT s.id, CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as name, s.class, u.username FROM students s JOIN users u ON s.user_id = u.id ORDER BY s.first_name, s.last_name");
    $students = $students_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $students = [];
}

// Get assignments with details
try {
    $assignments_stmt = $pdo->prepare("
        SELECT ta.*, s.name as student_name, s.class, tr.route_name, tr.route_number, ts.stop_name
        FROM transportation_assignments ta
        JOIN students s ON ta.student_id = s.id
        JOIN transportation_routes tr ON ta.route_id = tr.id
        LEFT JOIN transportation_stops ts ON ta.stop_id = ts.id
        ORDER BY ta.assigned_at DESC
    ");
    $assignments_stmt->execute();
    $assignments = $assignments_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $assignments = [];
}

$page_title = "Transportation Management";
include '../includes/admin_header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2><i class="fas fa-bus"></i> Transportation Management</h2>
                <div>
                    <button class="btn btn-primary" data-toggle="modal" data-target="#addRouteModal">
                        <i class="fas fa-plus"></i> Add Route
                    </button>
                    <button class="btn btn-success" data-toggle="modal" data-target="#assignStudentModal">
                        <i class="fas fa-user-plus"></i> Assign Student
                    </button>
                </div>
            </div>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="close" data-dismiss="alert">
                        <span>&times;</span>
                    </button>
                </div>
            <?php endif; ?>

            <!-- Routes Section -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5><i class="fas fa-route"></i> Bus Routes</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead class="thead-dark">
                                <tr>
                                    <th>Route Number</th>
                                    <th>Route Name</th>
                                    <th>Start Point</th>
                                    <th>End Point</th>
                                    <th>Distance (km)</th>
                                    <th>Capacity</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($routes as $route): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($route['route_number']); ?></td>
                                        <td><?php echo htmlspecialchars($route['route_name']); ?></td>
                                        <td><?php echo htmlspecialchars($route['start_point']); ?></td>
                                        <td><?php echo htmlspecialchars($route['end_point']); ?></td>
                                        <td><?php echo htmlspecialchars($route['distance_km']); ?></td>
                                        <td><?php echo htmlspecialchars($route['bus_capacity']); ?></td>
                                        <td>
                                            <span class="badge badge-<?php echo $route['status'] === 'active' ? 'success' : 'secondary'; ?>">
                                                <?php echo ucfirst($route['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <button class="btn btn-sm btn-info" onclick="viewRoute(<?php echo $route['id']; ?>)">
                                                <i class="fas fa-eye"></i> View
                                            </button>
                                            <button class="btn btn-sm btn-warning" onclick="editRoute(<?php echo $route['id']; ?>)">
                                                <i class="fas fa-edit"></i> Edit
                                            </button>
                                            <button class="btn btn-sm btn-success" onclick="addStop(<?php echo $route['id']; ?>)">
                                                <i class="fas fa-plus"></i> Add Stop
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                <?php if (empty($routes)): ?>
                                    <tr>
                                        <td colspan="8" class="text-center">No routes found. Add your first route!</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Student Assignments Section -->
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-users"></i> Student Transportation Assignments</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead class="thead-dark">
                                <tr>
                                    <th>Student Name</th>
                                    <th>Class</th>
                                    <th>Route</th>
                                    <th>Stop</th>
                                    <th>Assignment Type</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($assignments as $assignment): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($assignment['student_name']); ?></td>
                                        <td><?php echo htmlspecialchars($assignment['class']); ?></td>
                                        <td><?php echo htmlspecialchars($assignment['route_name']); ?> (<?php echo htmlspecialchars($assignment['route_number']); ?>)</td>
                                        <td><?php echo htmlspecialchars($assignment['stop_name'] ?? 'N/A'); ?></td>
                                        <td><?php echo ucfirst($assignment['assignment_type']); ?></td>
                                        <td>
                                            <span class="badge badge-<?php echo $assignment['status'] === 'active' ? 'success' : 'secondary'; ?>">
                                                <?php echo ucfirst($assignment['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <button class="btn btn-sm btn-warning" onclick="editAssignment(<?php echo $assignment['id']; ?>)">
                                                <i class="fas fa-edit"></i> Edit
                                            </button>
                                            <button class="btn btn-sm btn-danger" onclick="deleteAssignment(<?php echo $assignment['id']; ?>)">
                                                <i class="fas fa-trash"></i> Delete
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                <?php if (empty($assignments)): ?>
                                    <tr>
                                        <td colspan="7" class="text-center">No student assignments found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Route Modal -->
<div class="modal fade" id="addRouteModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add New Route</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="route_name">Route Name *</label>
                                <input type="text" class="form-control" id="route_name" name="route_name" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="route_number">Route Number *</label>
                                <input type="text" class="form-control" id="route_number" name="route_number" required>
                            </div>
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="description">Description</label>
                        <textarea class="form-control" id="description" name="description" rows="2"></textarea>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="start_point">Start Point *</label>
                                <input type="text" class="form-control" id="start_point" name="start_point" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="end_point">End Point *</label>
                                <input type="text" class="form-control" id="end_point" name="end_point" required>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="distance">Distance (km)</label>
                                <input type="number" class="form-control" id="distance" name="distance" step="0.01">
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="duration">Duration (minutes)</label>
                                <input type="number" class="form-control" id="duration" name="duration">
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="capacity">Bus Capacity</label>
                                <input type="number" class="form-control" id="capacity" name="capacity" value="50">
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="add_route" class="btn btn-primary">Add Route</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Assign Student Modal -->
<div class="modal fade" id="assignStudentModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Assign Student to Route</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="student_id">Student *</label>
                                <select class="form-control" id="student_id" name="student_id" required>
                                    <option value="">Select Student</option>
                                    <?php foreach ($students as $student): ?>
                                        <option value="<?php echo $student['id']; ?>">
                                            <?php echo htmlspecialchars($student['name']); ?> (<?php echo htmlspecialchars($student['class']); ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="route_id">Route *</label>
                                <select class="form-control" id="route_id" name="route_id" required onchange="loadRouteStops(this.value)">
                                    <option value="">Select Route</option>
                                    <?php foreach ($routes as $route): ?>
                                        <option value="<?php echo $route['id']; ?>">
                                            <?php echo htmlspecialchars($route['route_name']); ?> (<?php echo htmlspecialchars($route['route_number']); ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="stop_id">Pickup/Dropoff Stop</label>
                                <select class="form-control" id="stop_id" name="stop_id">
                                    <option value="">Select Stop (Optional)</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="assignment_type">Assignment Type *</label>
                                <select class="form-control" id="assignment_type" name="assignment_type" required>
                                    <option value="both">Pickup & Dropoff</option>
                                    <option value="pickup">Pickup Only</option>
                                    <option value="dropoff">Dropoff Only</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="start_date">Start Date</label>
                                <input type="date" class="form-control" id="start_date" name="start_date">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="end_date">End Date</label>
                                <input type="date" class="form-control" id="end_date" name="end_date">
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="emergency_contact">Emergency Contact</label>
                                <input type="tel" class="form-control" id="emergency_contact" name="emergency_contact">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="medical_conditions">Medical Conditions</label>
                                <textarea class="form-control" id="medical_conditions" name="medical_conditions" rows="2" placeholder="Any medical conditions or special needs"></textarea>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="assign_student" class="btn btn-success">Assign Student</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Add Stop Modal -->
<div class="modal fade" id="addStopModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add Stop to Route</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" id="stop_route_id" name="route_id">
                    <div class="form-group">
                        <label for="stop_name">Stop Name *</label>
                        <input type="text" class="form-control" id="stop_name" name="stop_name" required>
                    </div>
                    <div class="form-group">
                        <label for="stop_order">Stop Order *</label>
                        <input type="number" class="form-control" id="stop_order" name="stop_order" required min="1">
                        <small class="form-text text-muted">Order in which this stop appears on the route</small>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="pickup_time">Pickup Time</label>
                                <input type="time" class="form-control" id="pickup_time" name="pickup_time">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="dropoff_time">Dropoff Time</label>
                                <input type="time" class="form-control" id="dropoff_time" name="dropoff_time">
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="add_stop" class="btn btn-success">Add Stop</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function viewRoute(routeId) {
    // TODO: Implement view route details with stops
    alert('View route functionality to be implemented');
}

function editRoute(routeId) {
    // TODO: Implement edit route functionality
    alert('Edit route functionality to be implemented');
}

function addStop(routeId) {
    document.getElementById('stop_route_id').value = routeId;
    $('#addStopModal').modal('show');
}

function editAssignment(assignmentId) {
    // TODO: Implement edit assignment functionality
    alert('Edit assignment functionality to be implemented');
}

function deleteAssignment(assignmentId) {
    if (confirm('Are you sure you want to delete this assignment?')) {
        // TODO: Implement delete assignment functionality
        alert('Delete assignment functionality to be implemented');
    }
}

function loadRouteStops(routeId) {
    if (!routeId) {
        document.getElementById('stop_id').innerHTML = '<option value="">Select Stop (Optional)</option>';
        return;
    }

    // TODO: Load stops for selected route via AJAX
    // For now, just show a placeholder
    document.getElementById('stop_id').innerHTML = '<option value="">Loading stops...</option>';
}
</script>

<?php include '../includes/admin_footer.php'; ?>
