<?php
require_once '../config.php';
require_once '../includes/access_control.php';

$message = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['add_term'])) {
        $name = trim($_POST['name']);
        // Remove academic_year_id usage since terms are reusable
        $start_date = $_POST['start_date'];
        $end_date = $_POST['end_date'];
        $description = trim($_POST['description']);
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        try {
            // If setting as active, deactivate others first
            if ($is_active) {
                $pdo->prepare("UPDATE terms SET is_active = FALSE")->execute();
            }

            $stmt = $pdo->prepare("INSERT INTO terms (name, start_date, end_date, description, is_active) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$name, $start_date, $end_date, $description, $is_active]);
            $message = 'Term added successfully.';
        } catch (PDOException $e) {
            $message = 'Error adding term: ' . $e->getMessage();
        }
    } elseif (isset($_POST['edit_term'])) {
        $id = $_POST['id'];
        $name = trim($_POST['name']);
        // Remove academic_year_id usage since terms are reusable
        $start_date = $_POST['start_date'];
        $end_date = $_POST['end_date'];
        $description = trim($_POST['description']);
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        try {
            // If setting as active, deactivate others first
            if ($is_active) {
                $pdo->prepare("UPDATE terms SET is_active = FALSE WHERE id != ?")->execute([$id]);
            }

            $stmt = $pdo->prepare("UPDATE terms SET name = ?, start_date = ?, end_date = ?, description = ?, is_active = ? WHERE id = ?");
            $stmt->execute([$name, $start_date, $end_date, $description, $is_active, $id]);
            $message = 'Term updated successfully.';
        } catch (PDOException $e) {
            $message = 'Error updating term: ' . $e->getMessage();
        }
    } elseif (isset($_POST['delete_term'])) {
        $id = $_POST['id'];

        try {
            $stmt = $pdo->prepare("DELETE FROM terms WHERE id = ?");
            $stmt->execute([$id]);
            $message = 'Term deleted successfully.';
        } catch (PDOException $e) {
            $message = 'Error deleting term: ' . $e->getMessage();
        }
    }
}

 // Get all terms without academic year names since terms are reusable
$stmt = $pdo->query("SELECT * FROM terms ORDER BY created_at DESC");
$terms = $stmt->fetchAll();

// No need to get academic years for dropdown anymore
$academic_years = [];

// Get term for edit
$edit_term = null;
if (isset($_GET['edit'])) {
    $id = $_GET['edit'];
    $stmt = $pdo->prepare("SELECT * FROM terms WHERE id = ?");
    $stmt->execute([$id]);
    $edit_term = $stmt->fetch();
}

include '../includes/admin_header.php';
?>

<h2>Manage Terms/Semesters</h2>

<?php if ($message): ?>
    <div class="message"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>

<?php if (!$edit_term): ?>
    <button type="button" class="btn btn-primary" id="toggleTermForm" style="margin-bottom: 20px;">
        <i class="fas fa-plus-circle"></i> Add New Term
    </button>
<?php endif; ?>

<div id="termFormContainer" style="display: <?php echo $edit_term ? 'block' : 'none'; ?>;">
    <h3><?php echo $edit_term ? 'Edit Term' : 'Add New Term'; ?></h3>

    <form method="post" action="manage_terms.php" id="termForm" style="background: #f8f9fa; padding: 25px; border-radius: 10px; border: 1px solid #e9ecef; margin-bottom: 30px;">
        <?php if ($edit_term): ?>
            <input type="hidden" name="id" value="<?php echo $edit_term['id']; ?>">
        <?php endif; ?>

        <div class="row">
            <div class="col-md-6">
                <div class="form-group">
                    <label for="name" style="font-weight: 600; color: #495057;">
                        <i class="fas fa-tag"></i> Term Name:
                    </label>
                    <input type="text" id="name" name="name" value="<?php echo htmlspecialchars($edit_term['name'] ?? ''); ?>"
                           placeholder="e.g., Fall, Spring" required
                           style="width: 100%; padding: 12px; border: 2px solid #e9ecef; border-radius: 8px; font-size: 14px;">
                </div>
            </div>
            <div class="col-md-3">
                <div class="form-group">
                    <label for="start_date" style="font-weight: 600; color: #495057;">
                        <i class="fas fa-calendar-plus"></i> Start Date:
                    </label>
                    <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($edit_term['start_date'] ?? ''); ?>" required
                           style="width: 100%; padding: 12px; border: 2px solid #e9ecef; border-radius: 8px; font-size: 14px;">
                </div>
            </div>
            <div class="col-md-3">
                <div class="form-group">
                    <label for="end_date" style="font-weight: 600; color: #495057;">
                        <i class="fas fa-calendar-minus"></i> End Date:
                    </label>
                    <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($edit_term['end_date'] ?? ''); ?>" required
                           style="width: 100%; padding: 12px; border: 2px solid #e9ecef; border-radius: 8px; font-size: 14px;">
                </div>
            </div>
        </div>

        <div class="form-group">
            <label for="description" style="font-weight: 600; color: #495057;">
                <i class="fas fa-align-left"></i> Description:
            </label>
            <textarea id="description" name="description" rows="3"
                      style="width: 100%; padding: 12px; border: 2px solid #e9ecef; border-radius: 8px; font-size: 14px; resize: vertical;"><?php echo htmlspecialchars($edit_term['description'] ?? ''); ?></textarea>
        </div>

        <div class="form-group">
            <label for="is_active" style="font-weight: 600; color: #495057; cursor: pointer;">
                <input type="checkbox" id="is_active" name="is_active" <?php echo ($edit_term['is_active'] ?? false) ? 'checked' : ''; ?>
                       style="margin-right: 8px; transform: scale(1.2);">
                <i class="fas fa-check-circle"></i> Set as Active Term
            </label>
        </div>

        <div style="margin-top: 20px;">
            <button type="submit" name="<?php echo $edit_term ? 'edit_term' : 'add_term'; ?>" class="btn btn-primary"
                    style="border-radius: 8px; padding: 12px 24px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border: none; font-weight: 600;">
                <i class="fas fa-<?php echo $edit_term ? 'edit' : 'plus'; ?>"></i> <?php echo $edit_term ? 'Update Term' : 'Add Term'; ?>
            </button>

            <?php if ($edit_term): ?>
                <a href="manage_terms.php" class="btn" style="border-radius: 8px; padding: 12px 24px; margin-left: 10px;">Cancel</a>
            <?php else: ?>
                <button type="button" class="btn btn-secondary" id="cancelForm" style="border-radius: 8px; padding: 12px 24px; margin-left: 10px;">
                    <i class="fas fa-times"></i> Cancel
                </button>
            <?php endif; ?>
        </div>
    </form>
</div>

<h3>All Terms/Semesters</h3>

<table>
    <thead>
        <tr>
            <th>ID</th>
            <th>Term Name</th>
            <!-- Removed Academic Year column -->
            <th>Start Date</th>
            <th>End Date</th>
            <th>Description</th>
            <th>Status</th>
            <th>Actions</th>
        </tr>
    </thead>
    <tbody>
        <?php foreach ($terms as $term): ?>
            <tr>
                <td><?php echo $term['id']; ?></td>
                <td><?php echo htmlspecialchars($term['name']); ?></td>
                <!-- Removed Academic Year display -->
                <td><?php echo htmlspecialchars($term['start_date']); ?></td>
                <td><?php echo htmlspecialchars($term['end_date']); ?></td>
                <td><?php echo htmlspecialchars($term['description'] ?? ''); ?></td>
                <td><?php echo $term['is_active'] ? 'Active' : 'Inactive'; ?></td>
                <td>
                    <a href="?edit=<?php echo $term['id']; ?>" class="btn btn-small">Edit</a>
                    <form method="post" action="manage_terms.php" style="display:inline;">
                        <input type="hidden" name="id" value="<?php echo $term['id']; ?>">
                        <button type="submit" name="delete_term" class="btn btn-small btn-danger" onclick="return confirm('Are you sure?')">Delete</button>
                    </form>
                </td>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<script>
// Toggle form visibility
document.getElementById('toggleTermForm').addEventListener('click', function() {
    const formContainer = document.getElementById('termFormContainer');
    const isVisible = formContainer.style.display !== 'none';

    if (isVisible) {
        formContainer.style.display = 'none';
        this.innerHTML = '<i class="fas fa-plus-circle"></i> Add New Term';
    } else {
        formContainer.style.display = 'block';
        this.innerHTML = '<i class="fas fa-minus-circle"></i> Hide Form';
    }
});

// Cancel form (hide it)
document.getElementById('cancelForm').addEventListener('click', function() {
    const formContainer = document.getElementById('termFormContainer');
    formContainer.style.display = 'none';
    document.getElementById('toggleTermForm').innerHTML = '<i class="fas fa-plus-circle"></i> Add New Term';

    // Reset form
    document.getElementById('termForm').reset();
});

// Form validation
document.getElementById('termForm').addEventListener('submit', function(e) {
    const name = document.getElementById('name').value.trim();
    const startDate = document.getElementById('start_date').value;
    const endDate = document.getElementById('end_date').value;

    if (!name) {
        e.preventDefault();
        alert('Please enter a term name.');
        return;
    }

    if (!startDate || !endDate) {
        e.preventDefault();
        alert('Please select both start and end dates.');
        return;
    }

    if (new Date(startDate) >= new Date(endDate)) {
        e.preventDefault();
        alert('End date must be after start date.');
        return;
    }
});
</script>

<?php
include '../includes/admin_footer.php';
?>
