<?php
require_once '../config.php';

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['update_stock'])) {
        $item_id = intval($_POST['item_id']);
        $quantity_change = intval($_POST['quantity_change']);
        $reason = trim($_POST['reason']);

        try {
            // Get current stock
            $current_stmt = $pdo->prepare("SELECT quantity FROM inventory_stock WHERE item_id = ?");
            $current_stmt->execute([$item_id]);
            $current_stock = $current_stmt->fetch(PDO::FETCH_ASSOC);

            $previous_stock = $current_stock ? $current_stock['quantity'] : 0;
            $new_stock = $previous_stock + $quantity_change;

            // Update or insert stock
            if ($current_stock) {
                $update_stmt = $pdo->prepare("UPDATE inventory_stock SET quantity = ?, last_updated = NOW(), updated_by = ? WHERE item_id = ?");
                $update_stmt->execute([$new_stock, $_SESSION['user_id'], $item_id]);
            } else {
                $insert_stmt = $pdo->prepare("INSERT INTO inventory_stock (item_id, quantity, updated_by) VALUES (?, ?, ?)");
                $insert_stmt->execute([$item_id, $new_stock, $_SESSION['user_id']]);
            }

            // Log transaction
            $transaction_stmt = $pdo->prepare("INSERT INTO inventory_transactions (item_id, transaction_type, quantity, previous_stock, new_stock, reason, performed_by) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $transaction_type = $quantity_change > 0 ? 'in' : 'out';
            $transaction_stmt->execute([$item_id, $transaction_type, abs($quantity_change), $previous_stock, $new_stock, $reason, $_SESSION['user_id']]);

            $message = "Stock updated successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error updating stock: " . $e->getMessage();
            $message_type = "error";
        }
    }

    if (isset($_POST['add_inventory_category'])) {
        $category_name = trim($_POST['category_name']);

        try {
            $stmt = $pdo->prepare("INSERT INTO inventory_categories (name) VALUES (?)");
            $stmt->execute([$category_name]);
            $message = "Category added successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error adding category: " . $e->getMessage();
            $message_type = "error";
        }
    }
}

// Get all items with current stock
try {
    $items_stmt = $pdo->prepare("
        SELECT ii.*, ic.name as category_name, COALESCE(ist.quantity, 0) as current_stock
        FROM inventory_items ii
        LEFT JOIN inventory_categories ic ON ii.category_id = ic.id
        LEFT JOIN inventory_stock ist ON ii.id = ist.item_id
        ORDER BY ii.name
    ");
    $items_stmt->execute();
    $items = $items_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $items = [];
}

// Get recent transactions
try {
    $transactions_stmt = $pdo->prepare("
        SELECT it.*, ii.name as item_name, u.username as performed_by_name
        FROM inventory_transactions it
        JOIN inventory_items ii ON it.item_id = ii.id
        JOIN users u ON it.performed_by = u.id
        ORDER BY it.transaction_date DESC
        LIMIT 20
    ");
    $transactions_stmt->execute();
    $transactions = $transactions_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $transactions = [];
}

// Get inventory categories
try {
    $categories_stmt = $pdo->query("SELECT * FROM inventory_categories ORDER BY name");
    $categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $categories = [];
}

$page_title = "Inventory Management - Stock";
include '../includes/admin_header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2><i class="fas fa-boxes"></i> Inventory Management - Stock</h2>
                <div>
                <button class="btn btn-facilities" data-toggle="modal" data-target="#updateStockModal">
                    <i class="fas fa-plus-minus"></i> Update Stock
                </button>
                <button class="btn btn-facilities" onclick="window.location.href='manage_items.php'">
                    <i class="fas fa-cog"></i> Manage Items
                </button>
                <button class="btn btn-facilities" data-toggle="modal" data-target="#addCategoryModal">
                    <i class="fas fa-plus"></i> Add Category
                </button>
                <button class="btn btn-facilities" data-toggle="modal" data-target="#viewCategoriesModal">
                    <i class="fas fa-list"></i> View Categories
                </button>
            </div>
            </div>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="close" data-dismiss="alert">
                        <span>&times;</span>
                    </button>
                </div>
            <?php endif; ?>

            <!-- Stock Levels -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5><i class="fas fa-chart-bar"></i> Current Stock Levels</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover facilities-table">
                            <thead class="thead-dark">
                                <tr>
                                    <th>Item Name</th>
                                    <th>Category</th>
                                    <th>Current Stock</th>
                                    <th>Min Stock</th>
                                    <th>Max Stock</th>
                                    <th>Status</th>
                                    <th>Location</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($items as $item): ?>
                                    <tr class="<?php
                                        if ($item['current_stock'] <= $item['minimum_stock']) echo 'table-danger';
                                        elseif ($item['current_stock'] >= $item['maximum_stock']) echo 'table-warning';
                                        elseif ($item['current_stock'] < $item['minimum_stock'] * 1.2) echo 'table-info';
                                    ?>">
                                        <td><?php echo htmlspecialchars($item['name']); ?></td>
                                        <td><?php echo htmlspecialchars($item['category_name']); ?></td>
                                        <td><?php echo htmlspecialchars($item['current_stock']); ?> <?php echo htmlspecialchars($item['unit']); ?></td>
                                        <td><?php echo htmlspecialchars($item['minimum_stock']); ?></td>
                                        <td><?php echo htmlspecialchars($item['maximum_stock']); ?></td>
                                        <td>
                                            <?php if ($item['current_stock'] <= $item['minimum_stock']): ?>
                                                <span class="badge badge-danger">Low Stock</span>
                                            <?php elseif ($item['current_stock'] >= $item['maximum_stock']): ?>
                                                <span class="badge badge-warning">Over Stock</span>
                                            <?php else: ?>
                                                <span class="badge badge-success">Normal</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($item['location']); ?></td>
                                        <td>
                                            <button class="btn btn-sm btn-primary" onclick="updateStock(<?php echo $item['id']; ?>, '<?php echo htmlspecialchars($item['name']); ?>')">
                                                <i class="fas fa-edit"></i> Update
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                <?php if (empty($items)): ?>
                                    <tr>
                                        <td colspan="8" class="text-center">No inventory items found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Recent Transactions -->
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-history"></i> Recent Transactions</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover facilities-table">
                            <thead class="thead-dark">
                                <tr>
                                    <th>Item</th>
                                    <th>Type</th>
                                    <th>Quantity</th>
                                    <th>Previous Stock</th>
                                    <th>New Stock</th>
                                    <th>Reason</th>
                                    <th>Performed By</th>
                                    <th>Date</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($transactions as $transaction): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($transaction['item_name']); ?></td>
                                        <td>
                                            <span class="badge badge-<?php echo $transaction['transaction_type'] === 'in' ? 'success' : 'danger'; ?>">
                                                <?php echo ucfirst($transaction['transaction_type']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo htmlspecialchars($transaction['quantity']); ?></td>
                                        <td><?php echo htmlspecialchars($transaction['previous_stock']); ?></td>
                                        <td><?php echo htmlspecialchars($transaction['new_stock']); ?></td>
                                        <td><?php echo htmlspecialchars($transaction['reason']); ?></td>
                                        <td><?php echo htmlspecialchars($transaction['performed_by_name']); ?></td>
                                        <td><?php echo date('M d, Y H:i', strtotime($transaction['transaction_date'])); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                                <?php if (empty($transactions)): ?>
                                    <tr>
                                        <td colspan="8" class="text-center">No transactions found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Category Modal -->
<div class="modal fade" id="addCategoryModal" tabindex="-1" role="dialog" aria-labelledby="addCategoryModalLabel" aria-hidden="true">
  <div class="modal-dialog" role="document">
    <form method="POST" id="addCategoryForm">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title" id="addCategoryModalLabel">Add Inventory Category</h5>
          <button type="button" class="close" data-dismiss="modal" aria-label="Close">
            <span aria-hidden="true">&times;</span>
          </button>
        </div>
        <div class="modal-body">
          <div class="form-group">
            <label for="category_name">Category Name *</label>
            <input type="text" class="form-control" id="category_name" name="category_name" required>
          </div>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
          <button type="submit" name="add_inventory_category" class="btn btn-primary">Add Category</button>
        </div>
      </div>
    </form>
  </div>
</div>

<!-- View Categories Modal -->
<div class="modal fade" id="viewCategoriesModal" tabindex="-1" role="dialog" aria-labelledby="viewCategoriesModalLabel" aria-hidden="true">
  <div class="modal-dialog" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="viewCategoriesModalLabel">Inventory Categories</h5>
        <button type="button" class="close" data-dismiss="modal" aria-label="Close">
          <span aria-hidden="true">&times;</span>
        </button>
      </div>
      <div class="modal-body">
        <ul class="list-group" id="categoryList">
          <?php foreach ($categories as $category): ?>
            <li class="list-group-item"><?php echo htmlspecialchars($category['name']); ?></li>
          <?php endforeach; ?>
          <?php if (empty($categories)): ?>
            <li class="list-group-item text-center">No categories found.</li>
          <?php endif; ?>
        </ul>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>

<!-- Update Stock Modal -->
<div class="modal fade" id="updateStockModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Update Stock Level</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" id="update_item_id" name="item_id">
                    <div class="form-group">
                        <label for="selected_item"><i class="fas fa-box"></i> Selected Item</label>
                        <input type="text" class="form-control" id="selected_item" readonly>
                    </div>
                    <div class="form-group">
                        <label for="quantity_change"><i class="fas fa-plus-minus"></i> Quantity Change *</label>
                        <input type="number" class="form-control" id="quantity_change" name="quantity_change" required>
                        <small class="form-text text-muted">
                            Use positive numbers to add stock, negative numbers to reduce stock
                        </small>
                    </div>
                    <div class="form-group">
                        <label for="reason"><i class="fas fa-question-circle"></i> Reason *</label>
                        <select class="form-control" id="reason" name="reason" required>
                            <option value="">Select Reason</option>
                            <option value="Purchase">Purchase</option>
                            <option value="Return">Return</option>
                            <option value="Usage">Usage</option>
                            <option value="Damage">Damage</option>
                            <option value="Loss">Loss</option>
                            <option value="Transfer">Transfer</option>
                            <option value="Adjustment">Adjustment</option>
                            <option value="Other">Other</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="update_stock" class="btn btn-primary">Update Stock</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function updateStock(itemId, itemName) {
    document.getElementById('update_item_id').value = itemId;
    document.getElementById('selected_item').value = itemName;
    document.getElementById('quantity_change').value = '';
    document.getElementById('reason').value = '';
    $('#updateStockModal').modal('show');
}

$(document).ready(function() {
    $('#addCategoryForm').submit(function(e) {
        e.preventDefault();
        var categoryName = $('#category_name').val().trim();
        if (categoryName === '') {
            alert('Category name is required.');
            return;
        }
        $.ajax({
            url: 'manage_stock.php',
            type: 'POST',
            data: { add_inventory_category: true, category_name: categoryName },
            success: function(response) {
                location.reload();
            },
            error: function() {
                alert('Error adding category.');
            }
        });
    });
});
</script>

<?php include '../includes/admin_footer.php'; ?>
