<?php
require_once '../config.php';

$message = '';
$error = '';

if (!in_array($_SESSION['role'], ['admin'])) {
    header('Location: ../dashboard.php');
    exit;
}

// Fetch all levels
$levels = $pdo->query("SELECT id, name FROM levels ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);

// Fetch grading systems for each level
$grading_systems = [];
foreach ($levels as $level) {
    $stmt = $pdo->prepare("SELECT * FROM grading_systems WHERE level_id = ? ORDER BY min_score DESC");
    $stmt->execute([$level['id']]);
    $grading_systems[$level['id']] = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $level_id = $_POST['level_id'] ?? '';
    $grades = $_POST['grades'] ?? [];
    $min_scores = $_POST['min_scores'] ?? [];
    $max_scores = $_POST['max_scores'] ?? [];
    $descriptions = $_POST['descriptions'] ?? [];

    // Validate level_id exists
    $valid_level_ids = array_column($levels, 'id');
    if (!$level_id || !in_array($level_id, $valid_level_ids)) {
        $error = 'Invalid level selected.';
    } elseif (empty($grades)) {
        $error = 'No grades provided.';
    } else {
        try {
            $pdo->beginTransaction();

            // Delete existing grades for this level
            $stmt = $pdo->prepare("DELETE FROM grading_systems WHERE level_id = ?");
            $stmt->execute([$level_id]);

            // Insert new grades
            foreach ($grades as $index => $grade) {
                if (!empty($grade)) {
                    $min_score = $min_scores[$index] ?? 0;
                    $max_score = $max_scores[$index] ?? 0;
                    $description = $descriptions[$index] ?? '';

                    $stmt = $pdo->prepare("INSERT INTO grading_systems (level_id, grade, min_score, max_score, description) VALUES (?, ?, ?, ?, ?)");
                    $stmt->execute([$level_id, $grade, $min_score, $max_score, $description]);
                }
            }

            $pdo->commit();

            // Get level name for message
            $level_name = '';
            foreach ($levels as $level) {
                if ($level['id'] == $level_id) {
                    $level_name = $level['name'];
                    break;
                }
            }
            $message = $level_name . ' grading system updated successfully.';

            // Refresh data
            $stmt = $pdo->prepare("SELECT * FROM grading_systems WHERE level_id = ? ORDER BY min_score DESC");
            $stmt->execute([$level_id]);
            $grading_systems[$level_id] = $stmt->fetchAll(PDO::FETCH_ASSOC);

        } catch (PDOException $e) {
            $pdo->rollBack();
            $error = 'Error updating grading system: ' . $e->getMessage();
        }
    }
}

include '../includes/admin_header.php';
?>

<style>
.grading-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
}

.page-header {
    text-align: center;
    margin-bottom: 40px;
    padding: 30px 0;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 15px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
}

.page-header h1 {
    margin: 0 0 10px 0;
    font-size: 2.5em;
    font-weight: 700;
}

.page-header p {
    margin: 0;
    font-size: 1.1em;
    opacity: 0.9;
}

.level-tabs {
    display: flex;
    margin-bottom: 30px;
    background: white;
    border-radius: 10px;
    overflow: hidden;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    flex-wrap: wrap;
}

.level-tab {
    flex: 1;
    min-width: 200px;
    padding: 15px 20px;
    text-align: center;
    cursor: pointer;
    background: #f8f9fa;
    border: none;
    font-size: 1.1em;
    font-weight: 600;
    color: #666;
    transition: all 0.3s ease;
}

.level-tab.active {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.grading-card {
    background: white;
    border-radius: 15px;
    padding: 30px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
    margin-bottom: 30px;
}

.card-header {
    display: flex;
    align-items: center;
    margin-bottom: 25px;
    padding-bottom: 15px;
    border-bottom: 2px solid #f0f0f0;
}

.card-icon {
    font-size: 2.5em;
    margin-right: 15px;
    color: #667eea;
}

.card-title {
    margin: 0;
    font-size: 1.4em;
    font-weight: 600;
    color: #2c3e50;
}

.grades-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
}

.grades-table th,
.grades-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #ddd;
}

.grades-table th {
    background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
    color: white;
    font-weight: 600;
    font-size: 0.9em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.grades-table tr:nth-child(even) {
    background-color: #f8f9fa;
}

.grades-table tr:hover {
    background-color: #e8f5e8;
}

.grades-table input,
.grades-table textarea {
    width: 100%;
    padding: 8px 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 0.9em;
    transition: border-color 0.3s ease;
    box-sizing: border-box;
}

.grades-table input:focus,
.grades-table textarea:focus {
    outline: none;
    border-color: #27ae60;
    box-shadow: 0 0 0 2px rgba(39, 174, 96, 0.1);
}

.grades-table textarea {
    resize: vertical;
    min-height: 60px;
}

.btn {
    display: inline-block;
    padding: 12px 30px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    text-decoration: none;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-size: 16px;
    font-weight: 600;
    transition: transform 0.3s ease, box-shadow 0.3s ease;
}

.btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
}

.btn-success {
    background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
}

.btn-add {
    background: linear-gradient(135deg, #f39c12 0%, #e67e22 100%);
    margin-bottom: 20px;
}

.message {
    background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
    color: #155724;
    padding: 15px 20px;
    border-radius: 8px;
    margin-bottom: 20px;
    border-left: 4px solid #28a745;
    display: flex;
    align-items: center;
}

.message::before {
    content: "✓";
    font-size: 1.5em;
    margin-right: 10px;
}

.error-message {
    background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
    color: #721c24;
    padding: 15px 20px;
    border-radius: 8px;
    margin-bottom: 20px;
    border-left: 4px solid #dc3545;
    display: flex;
    align-items: center;
}

.error-message::before {
    content: "⚠";
    font-size: 1.5em;
    margin-right: 10px;
}

.form-actions {
    text-align: center;
    margin-top: 30px;
    padding: 20px;
    background: #f8f9fa;
    border-radius: 10px;
}

@media (max-width: 768px) {
    .level-tabs {
        flex-direction: column;
    }

    .grades-table {
        font-size: 0.8em;
    }

    .grades-table th,
    .grades-table td {
        padding: 8px 10px;
    }
}
</style>

<div class="grading-container">
    <div class="page-header">
        <h1>📊 Grading System Management</h1>
        <p>Configure grading thresholds for all education levels</p>
    </div>

    <?php if ($message): ?>
        <div class="message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="level-tabs">
        <?php foreach ($levels as $index => $level): ?>
            <button class="level-tab <?php echo $index === 0 ? 'active' : ''; ?>" onclick="showLevel('<?php echo htmlspecialchars($level['id']); ?>')">
                <?php echo htmlspecialchars($level['name']); ?>
            </button>
        <?php endforeach; ?>
    </div>

    <?php foreach ($levels as $index => $level): ?>
        <!-- <?php echo htmlspecialchars($level['name']); ?> Level Form -->
        <div id="level-<?php echo htmlspecialchars($level['id']); ?>" class="grading-card" style="display: <?php echo $index === 0 ? 'block' : 'none'; ?>;">
            <div class="card-header">
                <div class="card-icon">🏫</div>
                <h2 class="card-title"><?php echo htmlspecialchars($level['name']); ?> Grading System</h2>
            </div>

            <form method="post" action="manage_grading_system.php">
                <input type="hidden" name="level_id" value="<?php echo htmlspecialchars($level['id']); ?>">

                <table class="grades-table">
                    <thead>
                        <tr>
                            <th>Grade</th>
                            <th>Min Score</th>
                            <th>Max Score</th>
                            <th>Description</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody id="grades-body-<?php echo htmlspecialchars($level['id']); ?>">
                        <?php foreach ($grading_systems[$level['id']] as $grade): ?>
                            <tr>
                                <td><input type="text" name="grades[]" value="<?php echo htmlspecialchars($grade['grade']); ?>" required></td>
                                <td><input type="number" name="min_scores[]" step="0.01" min="0" max="100" value="<?php echo htmlspecialchars($grade['min_score']); ?>" required></td>
                                <td><input type="number" name="max_scores[]" step="0.01" min="0" max="100" value="<?php echo htmlspecialchars($grade['max_score']); ?>" required></td>
                                <td><textarea name="descriptions[]"><?php echo htmlspecialchars($grade['description'] ?? ''); ?></textarea></td>
                                <td><button type="button" class="btn btn-add" onclick="removeGrade(this)">Remove</button></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <div class="form-actions">
                    <button type="button" class="btn btn-add" onclick="addGrade('<?php echo htmlspecialchars($level['id']); ?>')">➕ Add Grade</button>
                    <button type="submit" class="btn btn-success">💾 Save <?php echo htmlspecialchars($level['name']); ?> Grading System</button>
                </div>
            </form>
        </div>
    <?php endforeach; ?>
</div>

<script>
function showLevel(levelId) {
    // Hide all levels
    document.querySelectorAll('.grading-card').forEach(card => card.style.display = 'none');

    // Remove active class from all tabs
    document.querySelectorAll('.level-tab').forEach(tab => tab.classList.remove('active'));

    // Show selected level and activate tab
    document.getElementById('level-' + levelId).style.display = 'block';
    event.target.classList.add('active');
}

function addGrade(levelId) {
    const tbody = document.getElementById('grades-body-' + levelId);
    const row = document.createElement('tr');

    row.innerHTML = `
        <td><input type="text" name="grades[]" required></td>
        <td><input type="number" name="min_scores[]" step="0.01" min="0" max="100" required></td>
        <td><input type="number" name="max_scores[]" step="0.01" min="0" max="100" required></td>
        <td><textarea name="descriptions[]"></textarea></td>
        <td><button type="button" class="btn btn-add" onclick="removeGrade(this)">Remove</button></td>
    `;

    tbody.appendChild(row);
}

function removeGrade(button) {
    const row = button.closest('tr');
    const tbody = row.parentElement;

    // Only remove if there are more than one rows
    if (tbody.children.length > 1) {
        row.remove();
    } else {
        alert('At least one grade is required.');
    }
}
</script>

<?php
include '../includes/admin_footer.php';
?>
