<?php
require_once '../config.php';
require_once '../includes/access_control.php';

// requireFeatureAccess('facilities_management'); // Temporarily disabled

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['approve_booking'])) {
        $booking_id = intval($_POST['booking_id']);

        try {
            $stmt = $pdo->prepare("UPDATE facility_bookings SET status = 'approved', approved_by = ?, approved_at = NOW() WHERE id = ?");
            $stmt->execute([$_SESSION['user_id'], $booking_id]);
            $message = "Booking approved successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error approving booking: " . $e->getMessage();
            $message_type = "error";
        }
    } elseif (isset($_POST['reject_booking'])) {
        $booking_id = intval($_POST['booking_id']);
        $reason = trim($_POST['rejection_reason']);

        try {
            $stmt = $pdo->prepare("UPDATE facility_bookings SET status = 'rejected', notes = CONCAT(COALESCE(notes, ''), '\nRejection reason: ', ?) WHERE id = ?");
            $stmt->execute([$reason, $booking_id]);
            $message = "Booking rejected successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error rejecting booking: " . $e->getMessage();
            $message_type = "error";
        }
    }
}

// Get all bookings with facility and user details
try {
    $bookings_stmt = $pdo->prepare("
        SELECT fb.*, f.name as facility_name, f.location, fc.name as category_name,
               u.username as booked_by_username,
               CASE
                   WHEN u.role = 'student' THEN s.name
                   WHEN u.role = 'teacher' THEN t.name
                   ELSE u.username
               END as booked_by_name,
               CASE
                   WHEN u.role = 'student' THEN s.class
                   ELSE ''
               END as student_class,
               au.username as approved_by_username
        FROM facility_bookings fb
        JOIN facilities f ON fb.facility_id = f.id
        JOIN facility_categories fc ON f.category_id = fc.id
        JOIN users u ON fb.booked_by = u.id
        LEFT JOIN students s ON (u.role = 'student' AND s.user_id = u.id)
        LEFT JOIN teachers t ON (u.role = 'teacher' AND t.user_id = u.id)
        LEFT JOIN users au ON fb.approved_by = au.id
        ORDER BY fb.booking_date DESC, fb.start_time DESC
    ");
    $bookings_stmt->execute();
    $bookings = $bookings_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $bookings = [];
}

$page_title = "Facility Bookings Management";
include '../includes/admin_header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2><i class="fas fa-calendar-check"></i> Facility Bookings Management</h2>
                <div>
                    <button class="btn btn-primary" onclick="window.location.href='manage_facilities.php'">
                        <i class="fas fa-cog"></i> Manage Facilities
                    </button>
                    <button class="btn btn-info" onclick="window.location.href='book_facility.php'">
                        <i class="fas fa-plus"></i> Book Facility
                    </button>
                </div>
            </div>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="close" data-dismiss="alert">
                        <span>&times;</span>
                    </button>
                </div>
            <?php endif; ?>

            <!-- Bookings Table -->
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-list"></i> All Bookings</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover facilities-table">
                            <thead class="thead-dark">
                                <tr>
                                    <th>Facility</th>
                                    <th>Booked By</th>
                                    <th>Event</th>
                                    <th>Date</th>
                                    <th>Time</th>
                                    <th>Attendees</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($bookings as $booking): ?>
                                    <tr class="<?php
                                        if ($booking['status'] === 'pending') echo 'table-warning';
                                        elseif ($booking['status'] === 'approved') echo 'table-success';
                                        elseif ($booking['status'] === 'rejected') echo 'table-danger';
                                    ?>">
                                        <td>
                                            <strong><?php echo htmlspecialchars($booking['facility_name']); ?></strong><br>
                                            <small class="text-muted"><?php echo htmlspecialchars($booking['category_name']); ?> - <?php echo htmlspecialchars($booking['location']); ?></small>
                                        </td>
                                        <td>
                                            <?php echo htmlspecialchars($booking['booked_by_name']); ?>
                                            <?php if ($booking['student_class']): ?>
                                                <br><small class="text-muted">Class: <?php echo htmlspecialchars($booking['student_class']); ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($booking['event_name']); ?></strong>
                                            <?php if ($booking['event_description']): ?>
                                                <br><small class="text-muted"><?php echo htmlspecialchars(substr($booking['event_description'], 0, 50)); ?>...</small>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo date('M d, Y', strtotime($booking['booking_date'])); ?></td>
                                        <td><?php echo date('H:i', strtotime($booking['start_time'])) . ' - ' . date('H:i', strtotime($booking['end_time'])); ?></td>
                                        <td><?php echo htmlspecialchars($booking['expected_attendees']); ?></td>
                                        <td>
                                            <span class="badge badge-<?php
                                                echo $booking['status'] === 'approved' ? 'success' :
                                                     ($booking['status'] === 'pending' ? 'warning' :
                                                      ($booking['status'] === 'rejected' ? 'danger' : 'secondary'));
                                            ?>">
                                                <?php echo ucfirst($booking['status']); ?>
                                            </span>
                                            <?php if ($booking['approved_by_username']): ?>
                                                <br><small class="text-muted">by <?php echo htmlspecialchars($booking['approved_by_username']); ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($booking['status'] === 'pending'): ?>
                                                <button class="btn btn-sm btn-success" onclick="approveBooking(<?php echo $booking['id']; ?>)">
                                                    <i class="fas fa-check"></i> Approve
                                                </button>
                                                <button class="btn btn-sm btn-danger" onclick="rejectBooking(<?php echo $booking['id']; ?>)">
                                                    <i class="fas fa-times"></i> Reject
                                                </button>
                                            <?php endif; ?>
                                            <button class="btn btn-sm btn-info" onclick="viewBookingDetails(<?php echo $booking['id']; ?>)">
                                                <i class="fas fa-eye"></i> View
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                <?php if (empty($bookings)): ?>
                                    <tr>
                                        <td colspan="8" class="text-center">No bookings found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Approve Booking Modal -->
<div class="modal fade" id="approveBookingModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Approve Booking</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <p>Are you sure you want to approve this booking?</p>
                    <input type="hidden" id="approve_booking_id" name="booking_id">
                    <div id="approve-booking-details"></div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="approve_booking" class="btn btn-success">Approve Booking</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Reject Booking Modal -->
<div class="modal fade" id="rejectBookingModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Reject Booking</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <p>Please provide a reason for rejecting this booking:</p>
                    <input type="hidden" id="reject_booking_id" name="booking_id">
                    <div class="form-group">
                        <label for="rejection_reason">Rejection Reason *</label>
                        <textarea class="form-control" id="rejection_reason" name="rejection_reason" rows="3" required></textarea>
                    </div>
                    <div id="reject-booking-details"></div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="reject_booking" class="btn btn-danger">Reject Booking</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function approveBooking(bookingId) {
    document.getElementById('approve_booking_id').value = bookingId;
    // You could add AJAX call here to get booking details
    $('#approveBookingModal').modal('show');
}

function rejectBooking(bookingId) {
    document.getElementById('reject_booking_id').value = bookingId;
    document.getElementById('rejection_reason').value = '';
    // You could add AJAX call here to get booking details
    $('#rejectBookingModal').modal('show');
}

function viewBookingDetails(bookingId) {
    // TODO: Implement view booking details functionality
    alert('View booking details functionality to be implemented');
}
</script>

<?php include '../includes/admin_footer.php'; ?>
