<?php
require_once '../config.php';
require_once '../includes/enhanced_access_control_fixed.php';

// Require permission to access this page - only super_admin can manage admins
requirePermission('user_management', '../login.php');

$message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_admin'])) {
        $username = trim($_POST['username']);
        $password = $_POST['password'];
        $confirm_password = $_POST['confirm_password'];
        $first_name = trim($_POST['first_name']);
        $last_name = trim($_POST['last_name']);
        $email = trim($_POST['email']);
        $phone = trim($_POST['phone']);
        $admin_level = $_POST['admin_level'];

        $errors = [];

        if (empty($username) || empty($password) || empty($confirm_password) || empty($first_name) || empty($last_name)) {
            $errors[] = "Username, password, first name, and last name are required.";
        }

        if ($password !== $confirm_password) {
            $errors[] = "Passwords do not match.";
        }

        // Check if username already exists
        $stmt = $pdo->prepare("SELECT id FROM admins WHERE username = ?");
        $stmt->execute([$username]);
        if ($stmt->fetch()) {
            $errors[] = "Username already exists.";
        }

        if (empty($errors)) {
            // Insert into unified admins table
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            $pdo->beginTransaction();
            try {
                $stmt = $pdo->prepare("INSERT INTO admins (username, password, role, admin_level, first_name, last_name, email, phone, is_active, created_at) VALUES (?, ?, 'admin', ?, ?, ?, ?, ?, 1, NOW())");
                $stmt->execute([$username, $hashed_password, $admin_level, $first_name, $last_name, $email, $phone]);

                $pdo->commit();
                $message = 'Admin added successfully.';
            } catch (Exception $e) {
                $pdo->rollBack();
                $message = 'Error adding admin: ' . $e->getMessage();
            }
        } else {
            $message = implode('<br>', $errors);
        }
    } elseif (isset($_POST['edit_admin'])) {
        $admin_id = $_POST['admin_id'];
        $first_name = trim($_POST['first_name']);
        $last_name = trim($_POST['last_name']);
        $email = trim($_POST['email']);
        $phone = trim($_POST['phone']);
        $admin_level = $_POST['admin_level'];
        $is_active = isset($_POST['is_active']) ? 1 : 0;
        $new_password = trim($_POST['new_password']);
        $confirm_new_password = trim($_POST['confirm_new_password']);

        $errors = [];

        if (!empty($new_password) && $new_password !== $confirm_new_password) {
            $errors[] = "New passwords do not match.";
        }

        if (empty($errors)) {
            $pdo->beginTransaction();
            try {
                // Update admin details
                $stmt = $pdo->prepare("UPDATE admins SET first_name = ?, last_name = ?, email = ?, phone = ?, admin_level = ?, is_active = ?, updated_at = NOW() WHERE id = ?");
                $stmt->execute([$first_name, $last_name, $email, $phone, $admin_level, $is_active, $admin_id]);

                // Update password if provided
                if (!empty($new_password)) {
                    $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                    $stmt = $pdo->prepare("UPDATE admins SET password = ? WHERE id = ?");
                    $stmt->execute([$hashed_password, $admin_id]);
                }

                $pdo->commit();
                $message = 'Admin updated successfully.';
            } catch (Exception $e) {
                $pdo->rollBack();
                $message = 'Error updating admin: ' . $e->getMessage();
            }
        } else {
            $message = implode('<br>', $errors);
        }
    } elseif (isset($_POST['delete_admin'])) {
        $admin_id = $_POST['admin_id'];

        // Delete admin record
        $stmt = $pdo->prepare("DELETE FROM admins WHERE id = ?");
        $stmt->execute([$admin_id]);

        $message = 'Admin deleted successfully.';
    } elseif (isset($_POST['toggle_admin_status'])) {
        $admin_id = $_POST['admin_id'];

        // Get current status
        $stmt = $pdo->prepare("SELECT is_active FROM admins WHERE id = ?");
        $stmt->execute([$admin_id]);
        $admin = $stmt->fetch();

        if ($admin) {
            $new_status = $admin['is_active'] ? 0 : 1;
            $action = $new_status ? 'activated' : 'deactivated';

            $stmt = $pdo->prepare("UPDATE admins SET is_active = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$new_status, $admin_id]);

            $message = 'Admin ' . $action . ' successfully.';
        }
    }
}

// Get all admins
$stmt = $pdo->query("
    SELECT *
    FROM admins
    WHERE admin_level IN ('admin', 'super_admin')
    ORDER BY created_at DESC
");
$admins = $stmt->fetchAll();

// Get admin for edit
$edit_admin = null;
if (isset($_GET['edit'])) {
    $admin_id = $_GET['edit'];
    $stmt = $pdo->prepare("
        SELECT *
        FROM admins
        WHERE id = ?
    ");
    $stmt->execute([$admin_id]);
    $edit_admin = $stmt->fetch();
}

include '../includes/admin_header.php';
?>

<style>
.form-row {
    display: flex;
    gap: 20px;
    margin-bottom: 15px;
}
.form-row .form-group {
    flex: 1;
}
.form-row .form-group:last-child {
    margin-bottom: 0;
}
fieldset {
    margin-bottom: 20px;
    border: 1px solid #ddd;
    border-radius: 5px;
    padding: 15px;
}
legend {
    font-weight: bold;
    color: #333;
}

.admins-table {
    background: white;
    border-radius: 8px;
    overflow: hidden;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.admins-table table {
    width: 100%;
    border-collapse: collapse;
}

.admins-table th,
.admins-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.admins-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.admins-table tr:hover {
    background: #f8f9fa;
}

.status-active {
    color: #28a745;
    font-weight: bold;
}

.status-inactive {
    color: #dc3545;
    font-weight: bold;
}
</style>

<h2>Manage Admins</h2>

<?php if ($message): ?>
    <div class="message"><?php echo $message; ?></div>
<?php endif; ?>

<!-- Button to trigger add modal -->
<button type="button" class="btn btn-primary" data-toggle="modal" data-target="#addAdminModal">Add New Admin</button>

<!-- Add Admin Modal -->
<div class="modal fade" id="addAdminModal" tabindex="-1" role="dialog" aria-labelledby="addAdminModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addAdminModalLabel">Add New Admin</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form method="post" action="manage_admins.php" id="addAdminForm">
                    <fieldset>
                        <legend>Login Credentials</legend>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="username">👤 Username:</label>
                                <input type="text" id="username" name="username" required>
                            </div>
                            <div class="form-group">
                                <label for="password">🔒 Password:</label>
                                <input type="password" id="password" name="password" required>
                            </div>
                            <div class="form-group">
                                <label for="confirm_password">🔒 Confirm Password:</label>
                                <input type="password" id="confirm_password" name="confirm_password" required>
                            </div>
                        </div>
                    </fieldset>

                    <fieldset>
                        <legend>Personal Information</legend>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="first_name">👤 First Name:</label>
                                <input type="text" id="first_name" name="first_name" required>
                            </div>
                            <div class="form-group">
                                <label for="last_name">👤 Last Name:</label>
                                <input type="text" id="last_name" name="last_name" required>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="email">📧 Email:</label>
                                <input type="email" id="email" name="email">
                            </div>
                            <div class="form-group">
                                <label for="phone">📞 Phone:</label>
                                <input type="text" id="phone" name="phone">
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="admin_level">Admin Level:</label>
                                <select id="admin_level" name="admin_level" required>
                                    <option value="admin">Admin</option>
                                    <option value="super_admin">Super Admin</option>
                                </select>
                            </div>
                        </div>
                    </fieldset>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="submit" form="addAdminForm" name="add_admin" class="btn btn-primary">Add Admin</button>
            </div>
        </div>
    </div>
</div>

<?php if ($edit_admin): ?>
<h3>Edit Admin</h3>

<form method="post" action="manage_admins.php">
    <input type="hidden" name="admin_id" value="<?php echo $edit_admin['id']; ?>">

    <fieldset>
        <legend>Login Credentials</legend>
        <div class="form-row">
            <div class="form-group">
                <label for="username">👤 Username:</label>
                <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($edit_admin['username'] ?? ''); ?>" readonly>
                <small class="text-muted">Username cannot be changed</small>
            </div>
        </div>
        <div class="form-row">
            <div class="form-group">
                <label for="new_password">🔒 New Password (leave blank to keep current):</label>
                <input type="password" id="new_password" name="new_password">
            </div>
            <div class="form-group">
                <label for="confirm_new_password">🔒 Confirm New Password:</label>
                <input type="password" id="confirm_new_password" name="confirm_new_password">
            </div>
        </div>
    </fieldset>

    <fieldset>
        <legend>Personal Information</legend>
        <div class="form-row">
            <div class="form-group">
                <label for="first_name">👤 First Name:</label>
                <input type="text" id="first_name" name="first_name" value="<?php echo htmlspecialchars($edit_admin['first_name'] ?? ''); ?>" required>
            </div>
            <div class="form-group">
                <label for="last_name">👤 Last Name:</label>
                <input type="text" id="last_name" name="last_name" value="<?php echo htmlspecialchars($edit_admin['last_name'] ?? ''); ?>" required>
            </div>
        </div>
        <div class="form-row">
            <div class="form-group">
                <label for="email">📧 Email:</label>
                <input type="email" id="email" name="email" value="<?php echo htmlspecialchars($edit_admin['email'] ?? ''); ?>">
            </div>
            <div class="form-group">
                <label for="phone">📞 Phone:</label>
                <input type="text" id="phone" name="phone" value="<?php echo htmlspecialchars($edit_admin['phone'] ?? ''); ?>">
            </div>
        </div>
        <div class="form-row">
            <div class="form-group">
                <label for="admin_level">Admin Level:</label>
                <select id="admin_level" name="admin_level" required>
                    <option value="admin" <?php echo ($edit_admin['admin_level'] ?? '') == 'admin' ? 'selected' : ''; ?>>Admin</option>
                    <option value="super_admin" <?php echo ($edit_admin['admin_level'] ?? '') == 'super_admin' ? 'selected' : ''; ?>>Super Admin</option>
                </select>
            </div>
            <div class="form-group">
                <label for="is_active">
                    <input type="checkbox" id="is_active" name="is_active" <?php echo ($edit_admin['is_active'] ?? 0) ? 'checked' : ''; ?>> Active Account
                </label>
            </div>
        </div>
    </fieldset>

    <button type="submit" name="edit_admin" class="btn btn-primary">Update Admin</button>
    <a href="manage_admins.php" class="btn">Cancel</a>
</form>
<?php endif; ?>

<h3>All Admins</h3>

<table class="admins-table">
    <thead>
        <tr>
            <th>Username</th>
            <th>Name</th>
            <th>Email</th>
            <th>Phone</th>
            <th>Admin Level</th>
            <th>Status</th>
            <th>Created</th>
            <th>Actions</th>
        </tr>
    </thead>
    <tbody>
        <?php foreach ($admins as $admin): ?>
            <tr>
                <td><?php echo htmlspecialchars($admin['username'] ?? 'N/A'); ?></td>
                <td><?php echo htmlspecialchars(trim(($admin['first_name'] ?? '') . ' ' . ($admin['last_name'] ?? ''))); ?></td>
                <td><?php echo htmlspecialchars($admin['email'] ?? ''); ?></td>
                <td><?php echo htmlspecialchars($admin['phone'] ?? ''); ?></td>
                <td><?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $admin['admin_level']))); ?></td>
                <td>
                    <span class="status-<?php echo ($admin['is_active'] ?? 0) ? 'active' : 'inactive'; ?>">
                        <?php echo ($admin['is_active'] ?? 0) ? 'Active' : 'Inactive'; ?>
                    </span>
                </td>
                <td><?php echo htmlspecialchars($admin['created_at'] ?? ''); ?></td>
                <td>
                    <a href="?edit=<?php echo $admin['id']; ?>" class="btn btn-small" style="background-color: #28a745; color: white;">Edit</a>
                    <form method="post" action="manage_admins.php" style="display:inline;">
                        <input type="hidden" name="admin_id" value="<?php echo $admin['id']; ?>">
                        <button type="submit" name="toggle_admin_status" class="btn btn-small" style="background-color: #ffc107; color: black;" onclick="return confirm('Are you sure you want to <?php echo ($admin['is_active'] ?? 0) ? 'deactivate' : 'activate'; ?> this admin account?')"><?php echo ($admin['is_active'] ?? 0) ? 'Deactivate' : 'Activate'; ?></button>
                    </form>
                    <form method="post" action="manage_admins.php" style="display:inline;">
                        <input type="hidden" name="admin_id" value="<?php echo $admin['id']; ?>">
                        <button type="submit" name="delete_admin" class="btn btn-small btn-danger" onclick="return confirm('Are you sure you want to delete this admin?')">Delete</button>
                    </form>
                </td>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<?php
include '../includes/admin_footer.php';
?>
