<?php
require_once '../config.php';
require_once '../includes/access_control.php';

$message = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['add_academic_year'])) {
        $year = trim($_POST['year']);
        $start_date = $_POST['start_date'];
        $end_date = $_POST['end_date'];
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        try {
            // If setting as active, deactivate others first
            if ($is_active) {
                $pdo->prepare("UPDATE academic_years SET is_active = FALSE")->execute();
                // Deactivate all terms since terms are now reusable across academic years
                $pdo->prepare("UPDATE terms SET is_active = FALSE")->execute();
            }

            $stmt = $pdo->prepare("INSERT INTO academic_years (year, start_date, end_date, is_active) VALUES (?, ?, ?, ?)");
            $stmt->execute([$year, $start_date, $end_date, $is_active]);
            $message = 'Academic year added successfully.';

            // Get the inserted academic year ID
            $academic_year_id = $pdo->lastInsertId();

            // Do NOT create new courses when a new academic year is created
            // This is to prevent duplication of courses for each academic year
            // So no code here to create courses automatically

        } catch (PDOException $e) {
            if ($e->getCode() == 23000) { // Duplicate entry
                $message = 'Academic year already exists.';
            } else {
                $message = 'Error adding academic year: ' . $e->getMessage();
            }
        }
    } elseif (isset($_POST['edit_academic_year'])) {
        $id = $_POST['id'];
        $year = trim($_POST['year']);
        $start_date = $_POST['start_date'];
        $end_date = $_POST['end_date'];
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        try {
            // If setting as active, deactivate others first
            if ($is_active) {
                $pdo->prepare("UPDATE academic_years SET is_active = FALSE WHERE id != ?")->execute([$id]);
                // Deactivate all terms since terms are now reusable across academic years
                $pdo->prepare("UPDATE terms SET is_active = FALSE")->execute();
                // Note: No need to activate a specific term since terms are reusable
            }

            $stmt = $pdo->prepare("UPDATE academic_years SET year = ?, start_date = ?, end_date = ?, is_active = ? WHERE id = ?");
            $stmt->execute([$year, $start_date, $end_date, $is_active, $id]);
            $message = 'Academic year updated successfully.';
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) { // Duplicate entry
                $message = 'Academic year already exists.';
            } else {
                $message = 'Error updating academic year: ' . $e->getMessage();
            }
        }
    } elseif (isset($_POST['delete_academic_year'])) {
        $id = $_POST['id'];

        try {
            $stmt = $pdo->prepare("DELETE FROM academic_years WHERE id = ?");
            $stmt->execute([$id]);
            $message = 'Academic year deleted successfully.';
        } catch (PDOException $e) {
            $message = 'Error deleting academic year: ' . $e->getMessage();
        }
    }
}

// Get all academic years
$stmt = $pdo->query("SELECT * FROM academic_years ORDER BY year DESC");
$academic_years = $stmt->fetchAll();

// Get academic year for edit
$edit_academic_year = null;
if (isset($_GET['edit'])) {
    $id = $_GET['edit'];
    $stmt = $pdo->prepare("SELECT * FROM academic_years WHERE id = ?");
    $stmt->execute([$id]);
    $edit_academic_year = $stmt->fetch();
}

include '../includes/admin_header.php';
?>

<h2>Manage Academic Years</h2>

<?php if ($message): ?>
    <div class="message"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>

<h3><?php echo $edit_academic_year ? 'Edit Academic Year' : 'Add New Academic Year'; ?></h3>

<form method="post" action="manage_academic_years.php">
    <?php if ($edit_academic_year): ?>
        <input type="hidden" name="id" value="<?php echo $edit_academic_year['id']; ?>">
    <?php endif; ?>

    <div class="form-group">
        <label for="year">Academic Year:</label>
        <input type="text" id="year" name="year" value="<?php echo htmlspecialchars($edit_academic_year['year'] ?? ''); ?>" placeholder="e.g., 2023-2024" required>
    </div>

    <div class="form-group">
        <label for="start_date">Start Date:</label>
        <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($edit_academic_year['start_date'] ?? ''); ?>" required>
    </div>

    <div class="form-group">
        <label for="end_date">End Date:</label>
        <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($edit_academic_year['end_date'] ?? ''); ?>" required>
    </div>

    <div class="form-group">
        <label for="is_active">
            <input type="checkbox" id="is_active" name="is_active" <?php echo ($edit_academic_year['is_active'] ?? false) ? 'checked' : ''; ?>>
            Set as Active Academic Year
        </label>
    </div>

    <button type="submit" name="<?php echo $edit_academic_year ? 'edit_academic_year' : 'add_academic_year'; ?>" class="btn btn-primary">
        <?php echo $edit_academic_year ? 'Update Academic Year' : 'Add Academic Year'; ?>
    </button>

    <?php if ($edit_academic_year): ?>
        <a href="manage_academic_years.php" class="btn">Cancel</a>
    <?php endif; ?>
</form>

<h3>All Academic Years</h3>

<table>
    <thead>
        <tr>
            <th>ID</th>
            <th>Academic Year</th>
            <th>Start Date</th>
            <th>End Date</th>
            <th>Status</th>
            <th>Actions</th>
        </tr>
    </thead>
    <tbody>
        <?php foreach ($academic_years as $ay): ?>
            <tr>
                <td><?php echo $ay['id']; ?></td>
                <td><?php echo htmlspecialchars($ay['year']); ?></td>
                <td><?php echo htmlspecialchars($ay['start_date']); ?></td>
                <td><?php echo htmlspecialchars($ay['end_date']); ?></td>
                <td><?php echo $ay['is_active'] ? 'Active' : 'Inactive'; ?></td>
                <td>
                    <a href="?edit=<?php echo $ay['id']; ?>" class="btn btn-small">Edit</a>
                    <form method="post" action="manage_academic_years.php" style="display:inline;">
                        <input type="hidden" name="id" value="<?php echo $ay['id']; ?>">
                        <button type="submit" name="delete_academic_year" class="btn btn-small btn-danger" onclick="return confirm('Are you sure?')">Delete</button>
                    </form>
                </td>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<?php
include '../includes/admin_footer.php';
?>
