<?php
require_once '../config.php';
require_once '../includes/access_control.php';

// Check if user is accounts
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'accounts') {
    header('Location: ../index.php');
    exit;
}

// Get store revenue report data
$report_stmt = $pdo->prepare("
    SELECT
        ii.id,
        ii.name as item_name,
        SUM(ss.quantity) as total_quantity,
        SUM(ss.quantity) * ii.selling_price as total_expected,
        SUM(ss.actual_received) as total_received,
        SUM(ss.outstanding) as total_outstanding,
        COUNT(DISTINCT ss.sale_id) as total_sales
    FROM student_sales ss
    JOIN inventory_items ii ON ss.item_id = ii.id
    GROUP BY ii.id, ii.name
    ORDER BY total_expected DESC
");
$report_stmt->execute();
$report_data = $report_stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate overall totals
$overall_expected = array_sum(array_column($report_data, 'total_expected'));
$overall_received = array_sum(array_column($report_data, 'total_received'));
$overall_outstanding = array_sum(array_column($report_data, 'total_outstanding'));

// Get inventory revenue potential (total value of all inventory items)
$inventory_stmt = $pdo->prepare("
    SELECT SUM(quantity * selling_price) as total_potential_revenue
    FROM inventory_items
");
$inventory_stmt->execute();
$inventory_data = $inventory_stmt->fetch(PDO::FETCH_ASSOC);
$total_potential_revenue = $inventory_data['total_potential_revenue'] ?? 0;

$page_title = "Store Revenue Report";
include 'includes/accounting_header.php';
?>

<style>
/* Custom styling to match accounting sidebar colors */
.btn-facilities {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    border: none;
    color: white;
    padding: 10px 20px;
    border-radius: 5px;
    transition: all 0.3s ease;
}

.btn-facilities:hover {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    color: white;
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.facilities-table {
    background: white;
    border-radius: 8px;
    overflow: hidden;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.facilities-table thead th {
    background: #f8f9fa;
    color: #495057;
    border: none;
    font-weight: 600;
    padding: 15px;
}

.facilities-table tbody tr:hover {
    background-color: rgba(46, 125, 50, 0.05);
}

/* Card styling */
.card {
    border: none;
    border-radius: 10px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.1);
}

.card-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    border-radius: 10px 10px 0 0 !important;
    border: none;
    padding: 15px 20px;
}

.card-header h5 {
    margin: 0;
    font-weight: 600;
}

/* Summary cards */
.summary-card {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    border-radius: 10px;
    padding: 20px;
    margin-bottom: 20px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.summary-card .card-title {
    font-size: 0.9em;
    margin-bottom: 5px;
    opacity: 0.9;
}

.summary-card .card-value {
    font-size: 2em;
    font-weight: bold;
    margin: 0;
}

/* Outstanding amount styling */
.text-danger {
    color: #dc3545 !important;
}

.text-success {
    color: #28a745 !important;
}

.text-warning {
    color: #ffc107 !important;
}
</style>

<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2><i class="fas fa-chart-line"></i> Store Revenue Report</h2>
                <button class="btn btn-facilities" onclick="window.print()">
                    <i class="fas fa-print"></i> Print Report
                </button>
            </div>

            <!-- Summary Cards -->
            <div class="row mb-4">
                <div class="col-md-2">
                    <div class="summary-card">
                        <div class="card-title">Inventory Revenue Potential</div>
                        <div class="card-value">₵<?php echo number_format($total_potential_revenue, 2); ?></div>
                    </div>
                </div>
                <div class="col-md-2">
                    <div class="summary-card">
                        <div class="card-title">Total Expected Revenue</div>
                        <div class="card-value">₵<?php echo number_format($overall_expected, 2); ?></div>
                    </div>
                </div>
                <div class="col-md-2">
                    <div class="summary-card">
                        <div class="card-title">Total Revenue Received</div>
                        <div class="card-value">₵<?php echo number_format($overall_received, 2); ?></div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="summary-card">
                        <div class="card-title">Outstanding Revenue</div>
                        <div class="card-value">₵<?php echo number_format($overall_outstanding, 2); ?></div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="summary-card">
                        <div class="card-title">Collection Rate</div>
                        <div class="card-value"><?php echo $overall_expected > 0 ? round(($overall_received / $overall_expected) * 100, 1) : 0; ?>%</div>
                    </div>
                </div>
            </div>

            <!-- Revenue Report Table -->
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-list"></i> Revenue by Item</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover facilities-table">
                            <thead class="thead-dark">
                                <tr>
                                    <th>Item Name</th>
                                    <th>Number Sold</th>
                                    <th>Expected Revenue</th>
                                    <th>Revenue Received</th>
                                    <th>Outstanding Revenue</th>
                                    <th>Collection Rate</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($report_data as $item): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($item['item_name']); ?></td>
                                        <td><?php echo number_format($item['total_quantity']); ?></td>
                                        <td>₵<?php echo number_format($item['total_expected'], 2); ?></td>
                                        <td class="text-success">₵<?php echo number_format($item['total_received'], 2); ?></td>
                                        <td class="<?php echo $item['total_outstanding'] > 0 ? 'text-danger' : 'text-success'; ?>">
                                            ₵<?php echo number_format($item['total_outstanding'], 2); ?>
                                        </td>
                                        <td>
                                            <?php
                                            $rate = $item['total_expected'] > 0 ? round(($item['total_received'] / $item['total_expected']) * 100, 1) : 0;
                                            $rate_class = $rate >= 90 ? 'text-success' : ($rate >= 70 ? 'text-warning' : 'text-danger');
                                            ?>
                                            <span class="<?php echo $rate_class; ?>"><?php echo $rate; ?>%</span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                <?php if (empty($report_data)): ?>
                                    <tr>
                                        <td colspan="6" class="text-center">No sales data found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                            <tfoot>
                                <tr style="background-color: #f8f9fa; font-weight: bold;">
                                    <td><strong>GRAND TOTAL</strong></td>
                                    <td><?php echo number_format(array_sum(array_column($report_data, 'total_quantity'))); ?></td>
                                    <td>₵<?php echo number_format($overall_expected, 2); ?></td>
                                    <td class="text-success">₵<?php echo number_format($overall_received, 2); ?></td>
                                    <td class="<?php echo $overall_outstanding > 0 ? 'text-danger' : 'text-success'; ?>">
                                        ₵<?php echo number_format($overall_outstanding, 2); ?>
                                    </td>
                                    <td>
                                        <?php
                                        $overall_rate = $overall_expected > 0 ? round(($overall_received / $overall_expected) * 100, 1) : 0;
                                        $overall_rate_class = $overall_rate >= 90 ? 'text-success' : ($overall_rate >= 70 ? 'text-warning' : 'text-danger');
                                        ?>
                                        <span class="<?php echo $overall_rate_class; ?>"><?php echo $overall_rate; ?>%</span>
                                    </td>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
@media print {
    .btn, .card-header, .summary-card {
        display: none !important;
    }
    .container-fluid {
        margin: 0;
        padding: 0;
    }
    .table {
        font-size: 12px;
    }
    .card {
        box-shadow: none;
        border: 1px solid #ddd;
    }
}
</style>

<?php include '../includes/admin_footer.php'; ?>
