<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';

// Check permissions - only admins and accounts users can manage purchase orders
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    header('Location: ../login.php');
    exit;
}

$message = '';
$error = '';

if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    unset($_SESSION['message']);
}

if (isset($_SESSION['error'])) {
    $error = $_SESSION['error'];
    unset($_SESSION['error']);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['submit_for_approval'])) {
        // Submit PO for approval
        $po_id = $_POST['po_id'];

        try {
            // Update status to pending
            $stmt = $pdo->prepare("UPDATE purchase_orders SET status = 'pending', updated_at = NOW() WHERE id = ? AND (status = 'draft' OR status IS NULL OR status = '')");
            $stmt->execute([$po_id]);

            if ($stmt->rowCount() > 0) {
                // Log activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'purchase_order_submitted',
                    "Submitted purchase order ID $po_id for approval"
                );

                $_SESSION['message'] = "Purchase order submitted for approval successfully!";
            } else {
                $error = "Purchase order not found or already submitted.";
            }
        } catch (PDOException $e) {
            $error = "Error submitting purchase order: " . $e->getMessage();
        }

        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    } elseif (isset($_POST['approve_po'])) {
        // Approve pending PO
        $po_id = $_POST['po_id'];

        try {
            $stmt = $pdo->prepare("UPDATE purchase_orders SET status = 'approved', updated_at = NOW() WHERE id = ? AND status = 'pending'");
            $stmt->execute([$po_id]);

            if ($stmt->rowCount() > 0) {
                // Log activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'purchase_order_approved',
                    "Approved purchase order ID $po_id"
                );

                $_SESSION['message'] = "Purchase order approved successfully!";
            } else {
                $error = "Purchase order not found or not in pending status.";
            }
        } catch (PDOException $e) {
            $error = "Error approving purchase order: " . $e->getMessage();
        }

        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    } elseif (isset($_POST['reject_po'])) {
        // Reject pending PO (send back to draft)
        $po_id = $_POST['po_id'];

        try {
            $stmt = $pdo->prepare("UPDATE purchase_orders SET status = 'draft', updated_at = NOW() WHERE id = ? AND status = 'pending'");
            $stmt->execute([$po_id]);

            if ($stmt->rowCount() > 0) {
                // Log activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'purchase_order_rejected',
                    "Rejected purchase order ID $po_id - sent back to draft"
                );

                $_SESSION['message'] = "Purchase order rejected and sent back to draft status.";
            } else {
                $error = "Purchase order not found or not in pending status.";
            }
        } catch (PDOException $e) {
            $error = "Error rejecting purchase order: " . $e->getMessage();
        }

        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    } elseif (isset($_POST['add_purchase_order'])) {
        // Add new purchase order
        $supplier_id = $_POST['supplier_id'];
        $order_date = $_POST['order_date'];
        $expected_delivery_date = $_POST['expected_delivery_date'];
        $total_amount = floatval($_POST['total_amount']);
        $notes = trim($_POST['notes']);
        $status = 'draft'; // New POs always start as draft

        // Get item data
        $item_names = $_POST['item_name'] ?? [];
        $category_ids = $_POST['category_id'] ?? [];
        $quantities = $_POST['quantity'] ?? [];
        $units = $_POST['unit'] ?? [];
        $unit_prices = $_POST['unit_price'] ?? [];
        $total_prices = $_POST['total_price'] ?? [];

        if (empty($supplier_id) || empty($order_date)) {
            $error = "Supplier and Date are required.";
        } elseif (empty($item_names) || count(array_filter($item_names)) === 0) {
            $error = "At least one item is required.";
        } else {
            try {
                // Generate order number
                $year = date('Y');
                $stmt = $pdo->prepare("SELECT MAX(CAST(SUBSTRING(po_number, 9) AS UNSIGNED)) as max_num FROM purchase_orders WHERE po_number LIKE ?");
                $stmt->execute(["PO-$year-%"]);
                $result = $stmt->fetch();
                $next_num = ($result['max_num'] ?? 0) + 1;
                $po_number = "PO-$year-" . str_pad($next_num, 4, '0', STR_PAD_LEFT);

                // Start transaction
                $pdo->beginTransaction();

                // Insert purchase order
                $stmt = $pdo->prepare("
                    INSERT INTO purchase_orders (po_number, supplier_id, order_date, expected_delivery_date, total_amount, notes, status)
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$po_number, $supplier_id, $order_date, $expected_delivery_date, $total_amount, $notes, $status]);
                $po_id = $pdo->lastInsertId();

                // Insert items
                $stmt = $pdo->prepare("
                    INSERT INTO purchase_order_items (purchase_order_id, item_name, category_id, quantity_ordered, unit, unit_price, total_price)
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                ");

                foreach ($item_names as $index => $item_name) {
                    if (!empty($item_name)) {
                        $category_id = intval($category_ids[$index] ?? 0);
                        $quantity = floatval($quantities[$index] ?? 0);
                        $unit = trim($units[$index] ?? 'pieces');
                        $unit_price = floatval($unit_prices[$index] ?? 0);
                        $total_price = floatval($total_prices[$index] ?? 0);

                        $stmt->execute([$po_id, $item_name, $category_id, $quantity, $unit, $unit_price, $total_price]);
                    }
                }

                // Commit transaction
                $pdo->commit();

                // Log activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'purchase_order_created',
                    "Created new purchase order: $po_number with " . count(array_filter($item_names)) . " items"
                );

                $_SESSION['message'] = "Purchase order added successfully!";
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            } catch (PDOException $e) {
                // Rollback on error
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $error = "Error adding purchase order: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['edit_purchase_order'])) {
        // Edit existing purchase order
        $id = $_POST['po_id'];
        $po_number = trim($_POST['po_number']);
        $supplier_id = $_POST['supplier_id'];
        $order_date = $_POST['order_date'];
        $expected_delivery_date = $_POST['expected_delivery_date'];
        $total_amount = floatval($_POST['total_amount']);
        $notes = trim($_POST['notes']);

        // Get item data from edit form
        $item_names = $_POST['edit_item_name'] ?? [];
        $category_ids = $_POST['edit_category_id'] ?? [];
        $quantities = $_POST['edit_quantity'] ?? [];
        $units = $_POST['edit_unit'] ?? [];
        $unit_prices = $_POST['edit_unit_price'] ?? [];
        $total_prices = $_POST['edit_total_price'] ?? [];

        if (empty($po_number) || empty($supplier_id) || empty($order_date)) {
            $error = "Order Number, Supplier, and Date are required.";
        } elseif (!preg_match('/^PO-\d{4}-\d{4}$/', $po_number)) {
            $error = "Order Number must be in format PO-YYYY-NNNN.";
        } elseif (empty($item_names) || count(array_filter($item_names)) === 0) {
            $error = "At least one item is required.";
        } else {
            try {
                // Check if another PO with this number exists
                $stmt = $pdo->prepare("SELECT id FROM purchase_orders WHERE po_number = ? AND id != ?");
                $stmt->execute([$po_number, $id]);
                if ($stmt->fetch()) {
                    $error = "Another purchase order with this number already exists.";
                } else {
                    // Start transaction
                    $pdo->beginTransaction();

                    // Update purchase order
                    $stmt = $pdo->prepare("
                        UPDATE purchase_orders
                        SET po_number = ?, supplier_id = ?, order_date = ?, expected_delivery_date = ?, total_amount = ?, notes = ?, updated_at = NOW()
                        WHERE id = ?
                    ");
                    $stmt->execute([$po_number, $supplier_id, $order_date, $expected_delivery_date, $total_amount, $notes, $id]);

                    // Delete existing items
                    $stmt = $pdo->prepare("DELETE FROM purchase_order_items WHERE purchase_order_id = ?");
                    $stmt->execute([$id]);

                    // Insert updated items
                    $stmt = $pdo->prepare("
                        INSERT INTO purchase_order_items (purchase_order_id, item_name, category_id, quantity_ordered, unit, unit_price, total_price)
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                    ");

                    foreach ($item_names as $index => $item_name) {
                        if (!empty($item_name)) {
                            $category_id = intval($category_ids[$index] ?? 0);
                            $quantity = floatval($quantities[$index] ?? 0);
                            $unit = trim($units[$index] ?? 'pieces');
                            $unit_price = floatval($unit_prices[$index] ?? 0);
                            $total_price = floatval($total_prices[$index] ?? 0);

                            $stmt->execute([$id, $item_name, $category_id, $quantity, $unit, $unit_price, $total_price]);
                        }
                    }

                    // Commit transaction
                    $pdo->commit();

                    // Log activity
                    AccountantActivityLogger::logActivity(
                        $_SESSION['user_id'],
                        'purchase_order_updated',
                        "Updated purchase order: $po_number with " . count(array_filter($item_names)) . " items"
                    );

                    $message = "Purchase order updated successfully!";
                }
            } catch (PDOException $e) {
                // Rollback on error
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $error = "Error updating purchase order: " . $e->getMessage();
            }
        }

    }

}

// Get all suppliers for dropdown
try {
    $stmt = $pdo->query("SELECT id, name AS supplier_name FROM suppliers s WHERE s.status = 'active' ORDER BY s.name");
    $suppliers = $stmt->fetchAll();
} catch (PDOException $e) {
    $suppliers = [];
}

// Get all expense categories for dropdown
try {
    $stmt = $pdo->query("SELECT id, name FROM budget_categories WHERE type = 'expense' ORDER BY name");
    $expense_categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $expense_categories = [];
}

// Get all draft and pending purchase orders for display
try {
    $stmt = $pdo->query("
        SELECT po.id, po.po_number AS po_number, po.supplier_id, po.order_date, po.expected_delivery_date, po.total_amount, po.notes, COALESCE(NULLIF(po.status, ''), 'draft') as status, po.created_at, po.updated_at, s.name AS supplier_name,
               (
                   SELECT COUNT(*) FROM purchase_order_items poi WHERE poi.purchase_order_id = po.id
               ) as item_count
        FROM purchase_orders po
        LEFT JOIN suppliers s ON po.supplier_id = s.id
        WHERE po.status IN ('draft', 'pending', '') OR po.status IS NULL
        ORDER BY po.order_date DESC, po_number DESC
    ");
    $purchase_orders = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading purchase orders: " . $e->getMessage();
    $purchase_orders = [];
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .settings-container {
        max-width: 1400px;
        margin: 0 auto;
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .settings-header {
        background: white;
        padding: 25px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .add-po-section {
        display: none;
        background: white;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        margin-bottom: 30px;
    }

    .add-po-section.show {
        display: block;
    }

    .settings-content {
        display: block;
    }

    .pos-list-section {
        background: white;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .section-title {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
    }

    .form-group {
        margin-bottom: 20px;
    }

    .form-group label {
        display: block;
        font-weight: 600;
        margin-bottom: 5px;
        color: #495057;
    }

    .form-group input,
    .form-group select,
    .form-group textarea {
        width: 100%;
        padding: 10px 12px;
        border: 1px solid #ced4da;
        border-radius: 5px;
        font-size: 14px;
    }

    .form-group textarea {
        resize: vertical;
        min-height: 80px;
    }

    .form-row {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px;
    }

    .btn {
        padding: 10px 20px;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        display: inline-block;
        margin-right: 10px;
    }

    .btn-primary {
        background: #007bff;
        color: white;
    }

    .btn-success {
        background: #28a745;
        color: white;
    }

    .btn-danger {
        background: #dc3545;
        color: white;
    }

    .btn-secondary {
        background: #6c757d;
        color: white;
    }

    .btn-warning {
        background: #ffc107;
        color: #212529;
    }

    .message {
        padding: 15px;
        border-radius: 5px;
        margin-bottom: 20px;
    }

    .success-message {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .error-message {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .pos-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    .pos-table th,
    .pos-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .pos-table th {
        background: #f8f9fa;
        font-weight: 600;
        color: #495057;
    }

    .po-number {
        font-weight: bold;
        color: #007bff;
    }

    .status-badge {
        display: inline-block;
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: 600;
    }

    .status-draft { background: #fff3cd; color: #856404; }
    .status-pending_payment { background: #d1ecf1; color: #0c5460; }
    .status-approved { background: #d4edda; color: #155724; }
    .status-received { background: #cce5ff; color: #004085; }
    .status-paid { background: #28a745; color: #ffffff; }

    .actions-column {
        width: 200px;
    }

    .modal {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        z-index: 1000;
    }

    .modal-content {
        background: white;
        margin: 5% auto;
        padding: 30px;
        border-radius: 10px;
        width: 90%;
        max-width: 600px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        max-height: 90vh;
        overflow-y: auto;
    }

    .modal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
        padding-bottom: 10px;
        border-bottom: 1px solid #e9ecef;
    }

    .modal-title {
        margin: 0;
        color: #2c3e50;
    }

    .close {
        font-size: 24px;
        cursor: pointer;
        color: #6c757d;
    }

    /* Items Section Styling */
    .items-section {
        margin-top: 20px;
    }

    .items-header {
        display: grid;
        grid-template-columns: 3fr 2fr 2fr 2fr 2fr 2fr 1fr;
        gap: 10px;
        font-weight: 600;
        color: #495057;
        margin-bottom: 10px;
        padding: 10px 0;
        border-bottom: 2px solid #e9ecef;
    }

    .item-row {
        display: grid;
        grid-template-columns: 3fr 2fr 2fr 2fr 2fr 2fr 1fr;
        gap: 10px;
        align-items: center;
        padding: 15px 0;
        border-bottom: 1px solid #e9ecef;
        transition: background-color 0.2s;
    }

    .item-row:hover {
        background-color: #f8f9fa;
    }

    .item-row .form-group {
        margin: 0;
    }

    .item-row input {
        width: 100%;
        padding: 8px 10px;
        border: 1px solid #ced4da;
        border-radius: 4px;
        font-size: 14px;
    }

    .item-row input:focus {
        outline: none;
        border-color: #007bff;
        box-shadow: 0 0 0 2px rgba(0,123,255,0.25);
    }

    .item-row .remove-item {
        width: 30px;
        height: 30px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 16px;
        font-weight: bold;
        margin: 0 auto;
    }

    .add-item-btn {
        margin-top: 15px;
        background: #28a745;
        color: white;
        border: none;
        padding: 8px 16px;
        border-radius: 4px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        display: inline-flex;
        align-items: center;
        gap: 5px;
    }

    .add-item-btn:hover {
        background: #218838;
    }

    @media (max-width: 768px) {
        .settings-content {
            grid-template-columns: 1fr;
        }

        .form-row {
            grid-template-columns: 1fr;
        }

        .items-header {
            display: none; /* Hide header on mobile */
        }

        .item-row {
            grid-template-columns: 1fr;
            gap: 5px;
            padding: 10px;
            border: 1px solid #e9ecef;
            border-radius: 5px;
            margin-bottom: 10px;
        }

        .item-row .form-group {
            display: flex;
            flex-direction: column;
        }

        .item-row label {
            font-size: 12px;
            font-weight: 600;
            color: #495057;
            margin-bottom: 3px;
        }

        .item-row input {
            padding: 6px 8px;
        }

        .item-row .remove-item {
            align-self: flex-end;
            margin-top: 5px;
        }
    }
</style>

<div class="settings-container">
    <div class="settings-header">
        <div>
            <h1>📄 Purchase Orders Management</h1>
            <p>Manage your school's purchase orders</p>
        </div>
        <button class="btn btn-primary" onclick="toggleAddPO()">➕ Create New PO</button>
    </div>

    <?php if ($message): ?>
        <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="settings-content">
        <div class="add-po-section">
            <h2 class="section-title">➕ Create New Purchase Order</h2>
            <form method="post">
                <div class="form-row">
                    <div class="form-group">
                        <label for="supplier_id">Supplier *</label>
                        <select name="supplier_id" id="supplier_id" required>
                            <option value="">Select Supplier</option>
                            <?php foreach ($suppliers as $supplier): ?>
                                <option value="<?php echo $supplier['id']; ?>"><?php echo htmlspecialchars($supplier['supplier_name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="order_date">Order Date *</label>
                        <input type="date" name="order_date" id="order_date" required>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="expected_delivery_date">Expected Delivery Date</label>
                        <input type="date" name="expected_delivery_date" id="expected_delivery_date">
                    </div>

                    <div class="form-group">
                        <label for="total_amount">Total Amount (GHS)</label>
                        <input type="number" name="total_amount" id="total_amount" step="0.01" min="0" placeholder="0.00" readonly>
                        <small class="text-muted">Calculated from items below</small>
                    </div>
                </div>

                <!-- Purchase Order Items Section -->
                <div class="form-group items-section">
                    <label>Items</label>
                    <div class="items-header">
                        <div>Item Name</div>
                        <div>Category</div>
                        <div>Quantity</div>
                        <div>Unit</div>
                        <div>Unit Price</div>
                        <div>Total</div>
                        <div>Action</div>
                    </div>
                    <div id="items-container">
                        <div class="item-row" data-row="1">
                            <input type="text" name="item_name[]" placeholder="Item Name" required>
                            <select name="category_id[]" required>
                                <option value="">Select Category</option>
                                <?php foreach ($expense_categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <input type="number" name="quantity[]" placeholder="Qty" min="1" step="0.01" required>
                            <input type="text" name="unit[]" placeholder="Unit" value="pieces">
                            <input type="number" name="unit_price[]" placeholder="Unit Price" step="0.01" min="0" required>
                            <input type="number" name="total_price[]" placeholder="Total" step="0.01" min="0" readonly>
                            <button type="button" class="btn btn-danger btn-sm remove-item" onclick="removeItem(this)">×</button>
                        </div>
                    </div>
                    <button type="button" class="btn btn-secondary btn-sm add-item-btn" onclick="addItem()">+ Add Item</button>
                </div>

                <div class="form-group">
                    <label for="notes">Notes</label>
                    <textarea name="notes" id="notes" placeholder="Details about the purchase order..."></textarea>
                </div>

                <button type="submit" name="add_purchase_order" class="btn btn-primary">Create Purchase Order</button>
            </form>
        </div>

        <div class="pos-list-section">
            <h2 class="section-title">📋 All Purchase Orders</h2>
            <table class="pos-table">
                <thead>
                    <tr>
                        <th>PO Number</th>
                        <th>Supplier</th>
                        <th>Date</th>
                        <th>Total (GHS)</th>
                        <th>Status</th>
                        <th>Items</th>
                        <th class="actions-column">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($purchase_orders as $po): ?>
                        <?php
                        // Fetch items for this PO
                        $items = [];
                        try {
                            $stmt = $pdo->prepare("SELECT item_name, category_id, quantity, unit, unit_price, total_price FROM purchase_order_items WHERE purchase_order_id = ?");
                            $stmt->execute([$po['id']]);
                            $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
                        } catch (PDOException $e) {
                            $items = [];
                        }
                        ?>
                        <tr>
                            <td class="po-number"><?php echo htmlspecialchars($po['po_number']); ?></td>
                            <td><?php echo htmlspecialchars($po['supplier_name'] ?? 'N/A'); ?></td>
                            <td><?php echo date('M j, Y', strtotime($po['order_date'])); ?></td>
                            <td><?php echo number_format($po['total_amount'], 2); ?></td>
                            <td>
                                <span class="status-badge status-<?php echo strtolower($po['status']); ?>">
                                    <?php echo ucfirst($po['status']); ?>
                                </span>
                            </td>
                            <td><?php echo $po['item_count']; ?> items</td>
                            <td class="actions-column">
                                <button class="btn btn-info btn-sm" onclick="viewPO(this, <?php echo $po['id']; ?>, '<?php echo addslashes($po['po_number']); ?>', '<?php echo addslashes($po['supplier_name'] ?? 'N/A'); ?>', '<?php echo $po['order_date']; ?>', '<?php echo $po['expected_delivery_date'] ?? ''; ?>', <?php echo $po['total_amount']; ?>, '<?php echo addslashes($po['status']); ?>', '<?php echo addslashes($po['notes'] ?? ''); ?>')" data-items='<?php echo htmlspecialchars(json_encode($items)); ?>'>👁️ View</button>
                                <?php if ($po['status'] === 'draft'): ?>
                                    <button class="btn btn-warning btn-sm" data-po-id="<?php echo $po['id']; ?>" data-po-number="<?php echo htmlspecialchars($po['po_number']); ?>" onclick="submitForApproval(this)">📤 Submit for Approval</button>
                                <?php endif; ?>
                                <button class="btn btn-secondary btn-sm" onclick="editPO(this, <?php echo $po['id']; ?>, '<?php echo addslashes($po['po_number']); ?>', <?php echo $po['supplier_id']; ?>, '<?php echo $po['order_date']; ?>', '<?php echo $po['expected_delivery_date'] ?? ''; ?>', <?php echo $po['total_amount']; ?>, '<?php echo addslashes($po['status']); ?>', '<?php echo addslashes($po['notes'] ?? ''); ?>')" data-items='<?php echo htmlspecialchars(json_encode($items)); ?>'>Edit</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Edit Purchase Order Modal -->
<div id="editModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Edit Purchase Order</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <form method="post">
            <input type="hidden" name="po_id" id="edit_po_id">
            <div class="form-group">
                <label for="edit_po_number">Order Number *</label>
                <input type="text" name="po_number" id="edit_po_number" required pattern="PO-\d{4}-\d{4}" title="Format: PO-YYYY-NNNN">
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_supplier_id">Supplier *</label>
                    <select name="supplier_id" id="edit_supplier_id" required>
                        <option value="">Select Supplier</option>
                        <?php foreach ($suppliers as $supplier): ?>
                            <option value="<?php echo $supplier['id']; ?>"><?php echo htmlspecialchars($supplier['supplier_name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label for="edit_order_date">Order Date *</label>
                    <input type="date" name="order_date" id="edit_order_date" required>
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_expected_delivery_date">Expected Delivery Date</label>
                    <input type="date" name="expected_delivery_date" id="edit_expected_delivery_date">
                </div>
                <div class="form-group">
                    <label for="edit_total_amount">Total Amount (GHS)</label>
                    <input type="number" name="total_amount" id="edit_total_amount" step="0.01" min="0" readonly>
                    <small class="text-muted">Calculated from items below</small>
                </div>
            </div>

            <!-- Purchase Order Items Section -->
            <div class="form-group items-section">
                <label>Items</label>
                <div class="items-header">
                    <div>Item Name</div>
                    <div>Category</div>
                    <div>Quantity</div>
                    <div>Unit</div>
                    <div>Unit Price</div>
                    <div>Total</div>
                    <div>Action</div>
                </div>
                <div id="edit-items-container">
                    <!-- Items will be populated by JavaScript -->
                </div>
                <button type="button" class="btn btn-secondary btn-sm add-item-btn" onclick="addEditItem()">+ Add Item</button>
            </div>

            <div class="form-group">
                <label for="edit_notes">Notes</label>
                <textarea name="notes" id="edit_notes"></textarea>
            </div>
            <button type="submit" name="edit_purchase_order" class="btn btn-success">Update Purchase Order</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<!-- View Purchase Order Modal -->
<div id="viewModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Purchase Order Details</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <div class="po-details">
            <div class="detail-row">
                <strong>PO Number:</strong> <span id="view_po_number"></span>
            </div>
            <div class="detail-row">
                <strong>Supplier:</strong> <span id="view_supplier_name"></span>
            </div>
            <div class="detail-row">
                <strong>Order Date:</strong> <span id="view_order_date"></span>
            </div>
            <div class="detail-row">
                <strong>Expected Delivery:</strong> <span id="view_expected_delivery"></span>
            </div>
            <div class="detail-row">
                <strong>Status:</strong> <span id="view_status" class="status-badge"></span>
            </div>
            <div class="detail-row">
                <strong>Total Amount:</strong> <span id="view_total_amount"></span>
            </div>
            <div class="detail-row">
                <strong>Notes:</strong> <span id="view_notes"></span>
            </div>
            <div class="items-section">
                <h3>Items</h3>
                <div id="view-items-list"></div>
            </div>
        </div>
        <div style="text-align: right; margin-top: 20px;">
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Close</button>
        </div>
    </div>
</div>

<!-- Status Toggle Modal -->
<div id="statusModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Change Purchase Order Status</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <p>Are you sure you want to <span id="status_action"></span> this purchase order?</p>
        <p style="color: #856404; font-weight: 600;">This will affect whether the PO appears in reports and workflows.</p>
        <form method="post">
            <input type="hidden" name="po_id" id="status_po_id">
            <input type="hidden" name="current_status" id="current_status">
            <button type="submit" name="toggle_active" class="btn btn-warning">Yes, Change Status</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<script>
window.expenseCategories = <?php echo json_encode($expense_categories); ?>;

function editPO(button, id, orderNumber, supplierId, orderDate, deliveryDate, totalAmount, notes) {
    document.getElementById('edit_po_id').value = id;
    document.getElementById('edit_po_number').value = orderNumber;
    document.getElementById('edit_supplier_id').value = supplierId;
    document.getElementById('edit_order_date').value = orderDate;
    document.getElementById('edit_expected_delivery_date').value = deliveryDate;
    document.getElementById('edit_total_amount').value = totalAmount;
    document.getElementById('edit_notes').value = notes;

    // Populate items
    const itemsContainer = document.getElementById('edit-items-container');
    itemsContainer.innerHTML = ''; // Clear existing items

    const itemsData = JSON.parse(button.getAttribute('data-items') || '[]');

    if (itemsData.length === 0) {
        // Add at least one empty row if no items
        addEditItem();
    } else {
        itemsData.forEach(item => {
            addEditItem(item.item_name, item.quantity, item.unit, item.unit_price, item.total_price);
        });
    }

    updateEditTotalAmount();
    document.getElementById('editModal').style.display = 'block';
}

function toggleStatus(id, isActive) {
    document.getElementById('status_po_id').value = id;
    document.getElementById('current_status').value = isActive ? '1' : '0';
    document.getElementById('status_action').textContent = isActive ? 'deactivate' : 'activate';
    document.getElementById('statusModal').style.display = 'block';
}

function viewPO(button, id, poNumber, supplierName, orderDate, expectedDelivery, totalAmount, status, notes) {
    document.getElementById('view_po_number').textContent = poNumber;
    document.getElementById('view_supplier_name').textContent = supplierName;
    document.getElementById('view_order_date').textContent = new Date(orderDate).toLocaleDateString();
    document.getElementById('view_expected_delivery').textContent = expectedDelivery ? new Date(expectedDelivery).toLocaleDateString() : 'N/A';
    document.getElementById('view_total_amount').textContent = 'GHS ' + parseFloat(totalAmount).toFixed(2);
    document.getElementById('view_status').textContent = status.charAt(0).toUpperCase() + status.slice(1);
    document.getElementById('view_status').className = 'status-badge status-' + status.toLowerCase();
    document.getElementById('view_notes').textContent = notes || 'N/A';

    // Populate items
    const itemsList = document.getElementById('view-items-list');
    itemsList.innerHTML = '';

    const itemsData = JSON.parse(button.getAttribute('data-items') || '[]');

    if (itemsData.length === 0) {
        itemsList.innerHTML = '<p>No items found.</p>';
    } else {
        const itemsTable = document.createElement('table');
        itemsTable.className = 'pos-table';
        itemsTable.innerHTML = `
            <thead>
                <tr>
                    <th>Item Name</th>
                    <th>Category</th>
                    <th>Quantity</th>
                    <th>Unit</th>
                    <th>Unit Price</th>
                    <th>Total</th>
                </tr>
            </thead>
            <tbody>
                ${itemsData.map(item => `
                    <tr>
                        <td>${item.item_name}</td>
                        <td>${window.expenseCategories.find(cat => cat.id == item.category_id)?.name || 'N/A'}</td>
                        <td>${item.quantity}</td>
                        <td>${item.unit}</td>
                        <td>GHS ${parseFloat(item.unit_price).toFixed(2)}</td>
                        <td>GHS ${parseFloat(item.total_price).toFixed(2)}</td>
                    </tr>
                `).join('')}
            </tbody>
        `;
        itemsList.appendChild(itemsTable);
    }

    document.getElementById('viewModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('editModal').style.display = 'none';
    document.getElementById('viewModal').style.display = 'none';
    document.getElementById('statusModal').style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    if (event.target.className === 'modal') {
        event.target.style.display = 'none';
    }
}

// Function to add new item row
function addItem() {
    const container = document.getElementById('items-container');
    const rowCount = container.children.length + 1;

    const itemRow = document.createElement('div');
    itemRow.className = 'item-row';
    itemRow.setAttribute('data-row', rowCount);

    itemRow.innerHTML = `
        <input type="text" name="item_name[]" placeholder="Item Name" required>
        <select name="category_id[]" required>
            <option value="">Select Category</option>
            ${window.expenseCategories.map(cat => `<option value="${cat.id}">${cat.name}</option>`).join('')}
        </select>
        <input type="number" name="quantity[]" placeholder="Qty" min="1" step="0.01" required>
        <input type="text" name="unit[]" placeholder="Unit" value="pieces">
        <input type="number" name="unit_price[]" placeholder="Unit Price" step="0.01" min="0" required>
        <input type="number" name="total_price[]" placeholder="Total" step="0.01" min="0" readonly>
        <button type="button" class="btn btn-danger btn-sm remove-item" onclick="removeItem(this)">×</button>
    `;

    container.appendChild(itemRow);

    // Add event listeners to new inputs
    const newRow = container.lastElementChild;
    const quantityInput = newRow.querySelector('input[name="quantity[]"]');
    const unitPriceInput = newRow.querySelector('input[name="unit_price[]"]');

    quantityInput.addEventListener('input', updateTotalAmount);
    unitPriceInput.addEventListener('input', updateTotalAmount);
}

// Function to remove item row
function removeItem(button) {
    const itemRow = button.closest('.item-row');
    itemRow.remove();
    updateTotalAmount();
}

// Function to calculate and update total amount
function updateTotalAmount() {
    const totalInputs = document.querySelectorAll('input[name="total_price[]"]');
    const quantityInputs = document.querySelectorAll('input[name="quantity[]"]');
    const unitPriceInputs = document.querySelectorAll('input[name="unit_price[]"]');

    let grandTotal = 0;

    totalInputs.forEach((totalInput, index) => {
        const quantity = parseFloat(quantityInputs[index]?.value) || 0;
        const unitPrice = parseFloat(unitPriceInputs[index]?.value) || 0;
        const total = quantity * unitPrice;

        totalInput.value = total.toFixed(2);
        grandTotal += total;
    });

    document.getElementById('total_amount').value = grandTotal.toFixed(2);
}

// Function to add new item row for edit modal
function addEditItem(itemName = '', categoryId = '', quantity = '', unit = 'pieces', unitPrice = '', totalPrice = '') {
    const container = document.getElementById('edit-items-container');
    const rowCount = container.children.length + 1;

    const itemRow = document.createElement('div');
    itemRow.className = 'item-row';
    itemRow.setAttribute('data-row', rowCount);

    itemRow.innerHTML = `
        <input type="text" name="edit_item_name[]" placeholder="Item Name" value="${itemName}" required>
        <select name="edit_category_id[]" required>
            <option value="">Select Category</option>
            ${window.expenseCategories.map(cat => `<option value="${cat.id}" ${cat.id == categoryId ? 'selected' : ''}>${cat.name}</option>`).join('')}
        </select>
        <input type="number" name="edit_quantity[]" placeholder="Qty" min="1" step="0.01" value="${quantity}" required>
        <input type="text" name="edit_unit[]" placeholder="Unit" value="${unit}">
        <input type="number" name="edit_unit_price[]" placeholder="Unit Price" step="0.01" min="0" value="${unitPrice}" required>
        <input type="number" name="edit_total_price[]" placeholder="Total" step="0.01" min="0" value="${totalPrice}" readonly>
        <button type="button" class="btn btn-danger btn-sm remove-item" onclick="removeEditItem(this)">×</button>
    `;

    container.appendChild(itemRow);

    // Add event listeners to new inputs
    const newRow = container.lastElementChild;
    const quantityInput = newRow.querySelector('input[name="edit_quantity[]"]');
    const unitPriceInput = newRow.querySelector('input[name="edit_unit_price[]"]');

    quantityInput.addEventListener('input', updateEditTotalAmount);
    unitPriceInput.addEventListener('input', updateEditTotalAmount);
}

// Function to remove item row from edit modal
function removeEditItem(button) {
    const itemRow = button.closest('.item-row');
    itemRow.remove();
    updateEditTotalAmount();
}

// Function to calculate and update total amount for edit modal
function updateEditTotalAmount() {
    const totalInputs = document.querySelectorAll('input[name="edit_total_price[]"]');
    const quantityInputs = document.querySelectorAll('input[name="edit_quantity[]"]');
    const unitPriceInputs = document.querySelectorAll('input[name="edit_unit_price[]"]');

    let grandTotal = 0;

    totalInputs.forEach((totalInput, index) => {
        const quantity = parseFloat(quantityInputs[index]?.value) || 0;
        const unitPrice = parseFloat(unitPriceInputs[index]?.value) || 0;
        const total = quantity * unitPrice;

        totalInput.value = total.toFixed(2);
        grandTotal += total;
    });

    document.getElementById('edit_total_amount').value = grandTotal.toFixed(2);
}

// Function to toggle the add PO section
function submitForApproval(button) {
    const poId = button.getAttribute('data-po-id');
    const poNumber = button.getAttribute('data-po-number');
    if (confirm(`Are you sure you want to submit purchase order ${poNumber} for approval?`)) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="submit_for_approval" value="1">
            <input type="hidden" name="po_id" value="${poId}">
        `;
        document.body.appendChild(form);
        form.submit();
    }
}

function toggleAddPO() {
    const addSection = document.querySelector('.add-po-section');
    addSection.classList.toggle('show');
}

// Set today's date as default for order date and initialize calculations
document.addEventListener('DOMContentLoaded', function() {
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('order_date').value = today;
    document.getElementById('edit_order_date').value = today;

    // Initialize item calculation
    updateTotalAmount();

    // Add event listeners to existing inputs
    const quantityInputs = document.querySelectorAll('input[name="quantity[]"]');
    const unitPriceInputs = document.querySelectorAll('input[name="unit_price[]"]');

    quantityInputs.forEach(input => input.addEventListener('input', updateTotalAmount));
    unitPriceInputs.forEach(input => input.addEventListener('input', updateTotalAmount));
});
</script>

<?php include '../includes/admin_footer.php'; ?>
