<?php
require_once '../config.php';

// Get filter parameters
$filter_class = $_GET['class'] ?? '';
$filter_academic_year = $_GET['academic_year'] ?? '';
$filter_status = $_GET['status'] ?? 'all';

// Build query for student-wise fee summary
$query = "SELECT 
    s.id as student_id,
    s.name as student_name,
    s.class,
    s.academic_year,
    sf.fee_id,
    f.name as fee_name,
    f.amount as fee_amount,
    sf.amount as student_fee_amount,
    sf.paid,
    sf.paid_amount,
    sf.due_date,
    sf.paid_at,
    sf.payment_method,
    sf.receipt_number
FROM students s
JOIN student_fees sf ON s.id = sf.student_id
JOIN fees f ON sf.fee_id = f.id
WHERE 1=1";

$params = [];

if ($filter_class) {
    $query .= " AND s.class = ?";
    $params[] = $filter_class;
}

if ($filter_academic_year) {
    $query .= " AND s.academic_year = ?";
    $params[] = $filter_academic_year;
}

if ($filter_status === 'paid') {
    $query .= " AND sf.paid = TRUE";
} elseif ($filter_status === 'partial') {
    $query .= " AND sf.paid_amount > 0 AND sf.paid_amount < sf.amount AND sf.paid = FALSE";
} elseif ($filter_status === 'unpaid') {
    $query .= " AND sf.paid = FALSE AND (sf.paid_amount IS NULL OR sf.paid_amount = 0)";
}

$query .= " ORDER BY CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name), sf.due_date ASC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$all_fees = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Group fees by student
$students_fees = [];
foreach ($all_fees as $fee) {
    $student_id = $fee['student_id'];
    if (!isset($students_fees[$student_id])) {
        $students_fees[$student_id] = [
            'name' => $fee['student_name'],
            'class' => $fee['class'],
            'academic_year' => $fee['academic_year'],
            'fees' => [],
            'total_expected' => 0,
            'total_paid' => 0,
            'total_outstanding' => 0
        ];
    }
    
    $students_fees[$student_id]['fees'][] = $fee;
    $students_fees[$student_id]['total_expected'] += $fee['student_fee_amount'];
    $students_fees[$student_id]['total_paid'] += $fee['paid_amount'] ?? 0;
    $students_fees[$student_id]['total_outstanding'] += ($fee['student_fee_amount'] - ($fee['paid_amount'] ?? 0));
}

// School details (hardcoded or from config)
$school_name = "School Name";
$school_address = "School Address, City, Ghana";
$report_date = date('Y-m-d');
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Detailed Student-wise Fee Summary Report</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 20px;
            line-height: 1.6;
            color: #333;
            background-color: #f9f9f9;
        }
        .header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #333;
            padding-bottom: 20px;
        }
        .header h1 {
            margin: 0;
            font-size: 24px;
            color: #333;
        }
        .header p {
            margin: 5px 0;
            font-size: 14px;
        }
        .filters {
            background: #f0f0f0;
            padding: 10px;
            margin-bottom: 20px;
            border-radius: 5px;
            font-size: 12px;
        }
        .student-section {
            margin-bottom: 40px;
            page-break-inside: avoid;
        }
        .student-header {
            background: #e0e0e0;
            padding: 10px;
            border-radius: 5px 5px 0 0;
            font-weight: bold;
            font-size: 16px;
            border-bottom: 1px solid #ccc;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 10px;
        }
        th, td {
            border: 1px solid #ccc;
            padding: 8px;
            text-align: left;
        }
        th {
            background: #f0f0f0;
            font-weight: bold;
        }
        .total-row {
            background: #f9f9f9;
            font-weight: bold;
        }
        .status-paid { color: green; }
        .status-unpaid { color: red; }
        .status-partial { color: orange; }
        .footer {
            margin-top: 50px;
            text-align: center;
            font-size: 12px;
            color: #666;
            border-top: 1px solid #ccc;
            padding-top: 10px;
        }
        @media print {
            body { background: white; }
            .no-print { display: none; }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1><?php echo htmlspecialchars($school_name); ?></h1>
        <p><?php echo htmlspecialchars($school_address); ?></p>
        <h2>Detailed Student-wise Fee Summary Report</h2>
        <p>Generated on: <?php echo $report_date; ?></p>
    </div>

    <div class="filters">
        <strong>Filters Applied:</strong><br>
        <?php if ($filter_class): ?>Class: <?php echo htmlspecialchars($filter_class); ?><br><?php endif; ?>
        <?php if ($filter_academic_year): ?>Academic Year: <?php echo htmlspecialchars($filter_academic_year); ?><br><?php endif; ?>
        <?php if ($filter_status !== 'all'): ?>Status: <?php echo ucfirst($filter_status); ?><br><?php endif; ?>
    </div>

    <?php foreach ($students_fees as $student_id => $student): ?>
        <div class="student-section">
            <div class="student-header">
                Student: <?php echo htmlspecialchars($student['name']); ?> | 
                Class: <?php echo htmlspecialchars($student['class']); ?> | 
                Academic Year: <?php echo htmlspecialchars($student['academic_year']); ?>
            </div>
            
            <table>
                <thead>
                    <tr>
                        <th>Fee Name</th>
                        <th>Amount (GHC)</th>
                        <th>Due Date</th>
                        <th>Paid Amount (GHC)</th>
                        <th>Payment Date</th>
                        <th>Payment Method</th>
                        <th>Receipt No.</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($student['fees'] as $fee): 
                        $status = $fee['paid'] ? 'Paid' : ($fee['paid_amount'] > 0 ? 'Partial' : 'Unpaid');
                        $status_class = strtolower($status);
                    ?>
                        <tr>
                            <td><?php echo htmlspecialchars($fee['fee_name']); ?></td>
                            <td><?php echo number_format($fee['student_fee_amount'], 2); ?></td>
                            <td><?php echo htmlspecialchars($fee['due_date'] ?? 'N/A'); ?></td>
                            <td><?php echo number_format($fee['paid_amount'] ?? 0, 2); ?></td>
                            <td><?php echo $fee['paid_at'] ? htmlspecialchars($fee['paid_at']) : 'N/A'; ?></td>
                            <td><?php echo htmlspecialchars($fee['payment_method'] ?? 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($fee['receipt_number'] ?? 'N/A'); ?></td>
                            <td class="status-<?php echo $status_class; ?>"><?php echo $status; ?></td>
                        </tr>
                    <?php endforeach; ?>
                    <tr class="total-row">
                        <td colspan="2"><strong>Total Expected: GHC <?php echo number_format($student['total_expected'], 2); ?></strong></td>
                        <td colspan="2"><strong>Total Paid: GHC <?php echo number_format($student['total_paid'], 2); ?></strong></td>
                        <td colspan="4"><strong>Outstanding: GHC <?php echo number_format($student['total_outstanding'], 2); ?></strong></td>
                    </tr>
                </tbody>
            </table>
        </div>
    <?php endforeach; ?>

    <?php if (empty($students_fees)): ?>
        <p style="text-align: center; font-size: 18px; color: #666;">No records found matching the selected filters.</p>
    <?php endif; ?>

    <div class="footer">
        <p>This is a computer-generated report. For official purposes, please contact the school administration.</p>
        <p>Page <span id="pageNum"></span> of <span id="totalPages"></span></p>
    </div>

    <script>
        // Auto-print when page loads
        window.onload = function() {
            window.print();
        };
    </script>
</body>
</html>
