<?php
session_start();

// Check permissions - only admins and accounts users can view PO prints
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    header('Location: ../login.php');
    exit;
}

if (!isset($_GET['po_id']) || empty($_GET['po_id'])) {
    echo "Invalid PO ID.";
    exit;
}

$po_id = intval($_GET['po_id']);

require_once '../config.php';

// Fetch PO details
try {
    $stmt = $pdo->prepare("
        SELECT po.*, s.name AS supplier_name, s.contact_person, s.phone, s.email, s.address AS supplier_address,
               u.username AS created_by_name
        FROM purchase_orders po
        JOIN suppliers s ON po.supplier_id = s.id
        LEFT JOIN users u ON po.created_by = u.id
        WHERE po.id = ?
    ");
    $stmt->execute([$po_id]);
    $po = $stmt->fetch();

    if (!$po) {
        echo "Purchase Order not found.";
        exit;
    }
} catch (PDOException $e) {
    echo "Error fetching PO details: " . $e->getMessage();
    exit;
}

// Fetch PO items
try {
    $stmt = $pdo->prepare("
        SELECT poi.*, COALESCE(poi.quantity_received, 0) as quantity_received,
               COALESCE(poi.quantity_paid, 0) as quantity_paid
        FROM purchase_order_items poi
        WHERE poi.purchase_order_id = ?
        ORDER BY poi.id
    ");
    $stmt->execute([$po_id]);
    $po_items = $stmt->fetchAll();
} catch (PDOException $e) {
    $po_items = [];
}

// Fetch payment history for this PO
try {
    $stmt = $pdo->prepare("
        SELECT pop.*, u.username AS processed_by_name
        FROM purchase_order_payments pop
        LEFT JOIN users u ON pop.processed_by = u.id
        WHERE pop.purchase_order_id = ?
        ORDER BY pop.payment_date DESC
    ");
    $stmt->execute([$po_id]);
    $payments = $stmt->fetchAll();
} catch (PDOException $e) {
    $payments = [];
}

// Calculate totals
$total_paid = array_sum(array_column($payments, 'amount'));
$remaining_balance = $po['total_amount'] - $total_paid;

// Fetch school settings
$school_name = "School Name";
$school_address = "School Address, City, Ghana";

try {
    $stmt = $pdo->prepare("SELECT school_name, address FROM school_settings LIMIT 1");
    $stmt->execute();
    $school_info = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($school_info) {
        $school_name = htmlspecialchars($school_info['school_name']);
        $school_address = htmlspecialchars($school_info['address']);
    }
} catch (Exception $e) {
    // Use default values if database query fails
}

$report_date = date('Y-m-d');
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PO Details - <?php echo htmlspecialchars($po['po_number']); ?></title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 15px 20px;
            line-height: 1.4;
            color: #222;
            background-color: #fff;
            max-width: 800px;
            margin-left: auto;
            margin-right: auto;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
            font-size: 12px;
        }

        .po-container {
            max-width: 100%;
        }

        /* Header Section */
        .header-section {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 30px;
            border-bottom: 3px solid #007BFF;
            padding-bottom: 15px;
        }

        .school-info {
            flex: 1;
        }

        .school-info h1 {
            margin: 0 0 8px 0;
            font-size: 20px;
            color: #007BFF;
            font-weight: 700;
            letter-spacing: 1px;
        }

        .school-info p {
            margin: 0;
            font-size: 12px;
            color: #555;
        }

        .po-info {
            text-align: right;
            flex: 1;
        }

        .po-info h2 {
            margin: 0 0 10px 0;
            font-size: 18px;
            color: #007BFF;
            font-weight: 700;
        }

        .po-details {
            font-size: 12px;
        }

        .po-details p {
            margin: 3px 0;
        }

        /* Supplier Information */
        .supplier-section {
            margin-bottom: 20px;
            page-break-inside: avoid;
        }

        .supplier-section h3 {
            margin: 0 0 10px 0;
            font-size: 14px;
            color: #007BFF;
            font-weight: 700;
            border-bottom: 2px solid #e9ecef;
            padding-bottom: 3px;
        }

        .supplier-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10px;
        }

        .supplier-row {
            display: flex;
            justify-content: space-between;
            padding: 5px 0;
        }

        .supplier-row label {
            font-weight: 600;
            color: #555;
        }

        .supplier-row span {
            font-weight: 500;
            color: #333;
        }

        /* Items Table */
        .items-section {
            margin-bottom: 20px;
        }

        .items-section h3 {
            margin: 0 0 10px 0;
            font-size: 14px;
            color: #007BFF;
            font-weight: 700;
            border-bottom: 2px solid #e9ecef;
            padding-bottom: 3px;
        }

        .items-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 11px;
        }

        .items-table th,
        .items-table td {
            padding: 8px 5px;
            text-align: left;
            border-bottom: 1px solid #e9ecef;
        }

        .items-table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #495057;
            font-size: 11px;
        }

        .items-table .amount {
            text-align: right;
        }

        .items-table .quantity {
            text-align: center;
        }

        /* Payment History */
        .payments-section {
            margin-bottom: 20px;
        }

        .payments-section h3 {
            margin: 0 0 10px 0;
            font-size: 14px;
            color: #007BFF;
            font-weight: 700;
            border-bottom: 2px solid #e9ecef;
            padding-bottom: 3px;
        }

        .payments-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 11px;
        }

        .payments-table th,
        .payments-table td {
            padding: 6px 5px;
            text-align: left;
            border-bottom: 1px solid #e9ecef;
        }

        .payments-table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #495057;
            font-size: 11px;
        }

        .payments-table .amount {
            text-align: right;
        }

        /* Summary Section */
        .summary-section {
            margin-bottom: 20px;
            page-break-inside: avoid;
        }

        .summary-section h3 {
            margin: 0 0 10px 0;
            font-size: 14px;
            color: #007BFF;
            font-weight: 700;
            border-bottom: 2px solid #e9ecef;
            padding-bottom: 3px;
        }

        .summary-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }

        .summary-box {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 5px;
            padding: 15px;
            text-align: center;
        }

        .summary-box .label {
            font-size: 12px;
            font-weight: 600;
            color: #007BFF;
            margin-bottom: 5px;
        }

        .summary-box .value {
            font-size: 16px;
            font-weight: 700;
            color: #0056b3;
        }

        .summary-box .value.positive {
            color: #28a745;
        }

        .summary-box .value.negative {
            color: #dc3545;
        }

        /* Footer */
        .footer-section {
            margin-top: 30px;
            text-align: center;
            border-top: 1px solid #ddd;
            padding-top: 15px;
            page-break-inside: avoid;
        }

        .footer-section p {
            margin: 5px 0;
            font-size: 11px;
            color: #666;
        }

        .status-badge {
            display: inline-block;
            padding: 3px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 600;
            text-transform: uppercase;
        }

        .status-paid {
            background: #d4edda;
            color: #155724;
        }

        .status-partial {
            background: #fff3cd;
            color: #856404;
        }

        .status-pending {
            background: #f8d7da;
            color: #721c24;
        }

        @media print {
            body {
                background: white;
                box-shadow: none;
                max-width: 100%;
                padding: 20px;
                font-size: 11px;
            }
            .no-print {
                display: none;
            }
            .header-section {
                border-bottom: 1px solid #000;
                margin-bottom: 20px;
                padding-bottom: 15px;
            }
            .school-info h1 {
                color: #000;
            }
            .po-info h2 {
                color: #000;
            }
            .supplier-section h3, .items-section h3, .payments-section h3, .summary-section h3 {
                color: #000;
                border-bottom: 1px solid #000;
            }
            .summary-box {
                border: 1px solid #000;
                background: white;
            }
            .summary-box .label, .summary-box .value {
                color: #000;
            }
            .status-badge {
                border: 1px solid #000;
            }
        }
    </style>
</head>
<body>
    <div class="po-container">
        <!-- Header Section -->
        <div class="header-section">
            <div class="school-info">
                <h1><?php echo htmlspecialchars($school_name); ?></h1>
                <p><?php echo htmlspecialchars($school_address); ?></p>
            </div>
            <div class="po-info">
                <h2>PURCHASE ORDER</h2>
                <div class="po-details">
                    <p><strong>PO Number:</strong> <?php echo htmlspecialchars($po['po_number']); ?></p>
                    <p><strong>Date:</strong> <?php echo date('M j, Y', strtotime($po['order_date'])); ?></p>
                    <p><strong>Status:</strong>
                        <span class="status-badge status-<?php
                            $status_class = 'pending';
                            if ($po['status'] === 'paid') $status_class = 'paid';
                            elseif (strpos($po['status'], 'paid') !== false || strpos($po['status'], 'partial') !== false) $status_class = 'partial';
                            echo $status_class;
                        ?>">
                            <?php echo ucfirst(str_replace('_', ' ', $po['status'])); ?>
                        </span>
                    </p>
                </div>
            </div>
        </div>

        <!-- Supplier Information -->
        <div class="supplier-section">
            <h3>Supplier Information</h3>
            <div class="supplier-grid">
                <div class="supplier-row">
                    <label>Supplier Name:</label>
                    <span><?php echo htmlspecialchars($po['supplier_name']); ?></span>
                </div>
                <div class="supplier-row">
                    <label>Contact Person:</label>
                    <span><?php echo htmlspecialchars($po['contact_person'] ?: 'N/A'); ?></span>
                </div>
                <div class="supplier-row">
                    <label>Phone:</label>
                    <span><?php echo htmlspecialchars($po['phone'] ?: 'N/A'); ?></span>
                </div>
                <div class="supplier-row">
                    <label>Email:</label>
                    <span><?php echo htmlspecialchars($po['email'] ?: 'N/A'); ?></span>
                </div>
                <div class="supplier-row" style="grid-column: span 2;">
                    <label>Address:</label>
                    <span><?php echo htmlspecialchars($po['supplier_address'] ?: 'N/A'); ?></span>
                </div>
            </div>
        </div>

        <!-- Items Section -->
        <div class="items-section">
            <h3>Purchase Order Items</h3>
            <table class="items-table">
                <thead>
                    <tr>
                        <th style="width: 40%;">Item Description</th>
                        <th style="width: 15%;" class="quantity">Quantity</th>
                        <th style="width: 15%;" class="quantity">Received</th>
                        <th style="width: 15%;" class="amount">Unit Price</th>
                        <th style="width: 15%;" class="amount">Total</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($po_items as $item): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($item['description']); ?></td>
                            <td class="quantity"><?php echo number_format($item['quantity_ordered'], 2); ?></td>
                            <td class="quantity"><?php echo number_format($item['quantity_received'], 2); ?></td>
                            <td class="amount">GHC <?php echo number_format($item['unit_price'], 2); ?></td>
                            <td class="amount">GHC <?php echo number_format($item['quantity_ordered'] * $item['unit_price'], 2); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Payment History -->
        <?php if (!empty($payments)): ?>
        <div class="payments-section">
            <h3>Payment History</h3>
            <table class="payments-table">
                <thead>
                    <tr>
                        <th style="width: 20%;">Date</th>
                        <th style="width: 25%;">Reference</th>
                        <th style="width: 20%;">Method</th>
                        <th style="width: 20%;" class="amount">Amount</th>
                        <th style="width: 15%;">Processed By</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($payments as $payment): ?>
                        <tr>
                            <td><?php echo date('M j, Y', strtotime($payment['payment_date'])); ?></td>
                            <td><?php echo htmlspecialchars($payment['payment_reference']); ?></td>
                            <td><?php echo ucfirst(str_replace('_', ' ', $payment['payment_method'])); ?></td>
                            <td class="amount">GHC <?php echo number_format($payment['amount'], 2); ?></td>
                            <td><?php echo htmlspecialchars($payment['processed_by_name'] ?: 'Unknown'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>

        <!-- Summary Section -->
        <div class="summary-section">
            <h3>Financial Summary</h3>
            <div class="summary-grid">
                <div class="summary-box">
                    <div class="label">Total PO Amount</div>
                    <div class="value">GHC <?php echo number_format($po['total_amount'], 2); ?></div>
                </div>
                <div class="summary-box">
                    <div class="label">Total Paid</div>
                    <div class="value positive">GHC <?php echo number_format($total_paid, 2); ?></div>
                </div>
                <div class="summary-box">
                    <div class="label">Remaining Balance</div>
                    <div class="value <?php echo $remaining_balance > 0 ? 'negative' : 'positive'; ?>">
                        GHC <?php echo number_format($remaining_balance, 2); ?>
                    </div>
                </div>
                <div class="summary-box">
                    <div class="label">Payment Status</div>
                    <div class="value">
                        <span class="status-badge status-<?php
                            $status_class = 'pending';
                            if ($remaining_balance <= 0) $status_class = 'paid';
                            elseif ($total_paid > 0) $status_class = 'partial';
                            echo $status_class;
                        ?>">
                            <?php
                            if ($remaining_balance <= 0) echo 'Fully Paid';
                            elseif ($total_paid > 0) echo 'Partially Paid';
                            else echo 'Unpaid';
                            ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Footer -->
        <div class="footer-section">
            <p><strong>Created By:</strong> <?php echo htmlspecialchars($po['created_by_name'] ?: 'Unknown'); ?> |
               <strong>Created Date:</strong> <?php echo date('M j, Y', strtotime($po['created_at'])); ?></p>
            <p>This is a computer-generated document. Printed on <?php echo date('F j, Y \a\t g:i A'); ?></p>
        </div>
    </div>

    <script>
        // Auto-print when page loads
        window.onload = function() {
            setTimeout(function() {
                window.print();
            }, 100);
        };

        // Close window after printing (for popup windows)
        window.onafterprint = function() {
            if (window.opener) {
                window.close();
            }
        };
    </script>
</body>
</html>
