<?php
require_once '../config.php';

// Get filter parameters
$filter_loan_id = $_GET['loan_id'] ?? '';
$filter_date_from = $_GET['date_from'] ?? '';
$filter_date_to = $_GET['date_to'] ?? '';
$filter_payment_method = $_GET['payment_method'] ?? '';
$search = $_GET['search'] ?? '';

// Build query with remaining balance calculation
$query = "
    SELECT lp.*,
           l.name as loan_name,
           l.original_amount as loan_amount,
           u.username as recorded_by_name,
           (SELECT SUM(lp2.payment_amount)
            FROM loan_payments lp2
            WHERE lp2.loan_id = lp.loan_id
            AND lp2.payment_date <= lp.payment_date
            AND lp2.id <= lp.id) as cumulative_paid,
           (l.original_amount - (SELECT SUM(lp2.payment_amount)
                                FROM loan_payments lp2
                                WHERE lp2.loan_id = lp.loan_id
                                AND lp2.payment_date <= lp.payment_date
                                AND lp2.id <= lp.id)) as remaining_balance
    FROM loan_payments lp
    JOIN liabilities l ON lp.loan_id = l.id
    JOIN users u ON lp.recorded_by = u.id
    WHERE 1=1
";

$params = [];

if ($filter_loan_id) {
    $query .= " AND lp.loan_id = ?";
    $params[] = $filter_loan_id;
}

if ($filter_date_from) {
    $query .= " AND lp.payment_date >= ?";
    $params[] = $filter_date_from;
}

if ($filter_date_to) {
    $query .= " AND lp.payment_date <= ?";
    $params[] = $filter_date_to;
}

if ($filter_payment_method) {
    $query .= " AND lp.payment_method = ?";
    $params[] = $filter_payment_method;
}

if ($search) {
    $query .= " AND (l.name LIKE ? OR lp.payment_amount LIKE ? OR lp.payment_method LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$query .= " ORDER BY lp.payment_date DESC, lp.created_at DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$payments = $stmt->fetchAll();

// Calculate totals
$total_payments = count($payments);
$total_amount = array_sum(array_column($payments, 'payment_amount'));

// Get school settings
$school_stmt = $pdo->query("SELECT school_name, address, phone, email, logo_path FROM school_settings LIMIT 1");
$school_settings = $school_stmt->fetch(PDO::FETCH_ASSOC);

$school_name = $school_settings['school_name'] ?? "School Name";
$school_address = $school_settings['address'] ?? "School Address, City, Ghana";
$school_phone = $school_settings['phone'] ?? "";
$school_email = $school_settings['email'] ?? "";
$school_logo = $school_settings['logo_path'] ?? "";

$report_date = date('Y-m-d H:i:s');

// Get filter descriptions
$filter_descriptions = [];
if ($filter_loan_id) {
    $loan_stmt = $pdo->prepare("SELECT name FROM liabilities WHERE id = ?");
    $loan_stmt->execute([$filter_loan_id]);
    $loan = $loan_stmt->fetch();
    $filter_descriptions[] = "Loan: " . ($loan ? $loan['name'] : 'Unknown');
}
if ($filter_date_from) $filter_descriptions[] = "From Date: " . $filter_date_from;
if ($filter_date_to) $filter_descriptions[] = "To Date: " . $filter_date_to;
if ($filter_payment_method) $filter_descriptions[] = "Payment Method: " . ucfirst(str_replace('_', ' ', $filter_payment_method));
if ($search) $filter_descriptions[] = "Search: " . $search;
$filters_text = empty($filter_descriptions) ? "All Payments" : implode(", ", $filter_descriptions);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Loan Payment History Report</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 20px;
            line-height: 1.6;
            color: #333;
            background-color: #fff;
        }
        .header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #333;
            padding-bottom: 20px;
        }
        .header h1 {
            margin: 0;
            font-size: 24px;
            color: #333;
        }
        .header p {
            margin: 5px 0;
            font-size: 14px;
        }
        .filters {
            background: #f0f0f0;
            padding: 10px;
            margin-bottom: 20px;
            border-radius: 5px;
            font-size: 12px;
        }
        .summary {
            background: #e8f4f8;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            display: flex;
            justify-content: space-around;
            align-items: center;
        }
        .summary-item {
            text-align: center;
        }
        .summary-item h3 {
            margin: 0 0 5px 0;
            color: #2c3e50;
            font-size: 18px;
        }
        .summary-item p {
            margin: 0;
            color: #7f8c8d;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        th, td {
            border: 1px solid #ccc;
            padding: 8px;
            text-align: left;
        }
        th {
            background: #f0f0f0;
            font-weight: bold;
        }
        .amount {
            font-weight: bold;
            color: #2c3e50;
        }
        .payment-method {
            padding: 2px 6px;
            border-radius: 8px;
            font-size: 0.8em;
            font-weight: bold;
            text-transform: uppercase;
        }
        .payment-method.cash {
            background: #28a745;
            color: white;
        }
        .payment-method.bank_transfer {
            background: #007bff;
            color: white;
        }
        .payment-method.cheque {
            background: #ffc107;
            color: #212529;
        }
        .payment-method.mobile_money {
            background: #6f42c1;
            color: white;
        }
        .no-payments {
            text-align: center;
            color: #7f8c8d;
            font-style: italic;
            padding: 40px;
        }
        .footer {
            margin-top: 50px;
            text-align: center;
            font-size: 12px;
            color: #666;
            border-top: 1px solid #ccc;
            padding-top: 10px;
        }
        @media print {
            body { background: white; }
            .no-print { display: none; }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1><?php echo htmlspecialchars($school_name); ?></h1>
        <p><?php echo htmlspecialchars($school_address); ?></p>
        <h2>Loan Payment History Report</h2>
        <p>Generated on: <?php echo $report_date; ?></p>
    </div>

    <div class="filters">
        <strong>Filters Applied:</strong><br>
        <?php echo htmlspecialchars($filters_text); ?>
    </div>

    <div class="summary">
        <div class="summary-item">
            <h3><?php echo $total_payments; ?></h3>
            <p>Total Payments</p>
        </div>
        <div class="summary-item">
            <h3>GHC <?php echo number_format($total_amount, 2); ?></h3>
            <p>Total Amount Paid</p>
        </div>
    </div>

    <table>
        <thead>
            <tr>
                <th>Loan Name</th>
                <th>Payment Date</th>
                <th>Amount (GHC)</th>
                <th>Payment Method</th>
                <th>Remaining Balance (GHC)</th>
                <th>Recorded By</th>
                <th>Recorded At</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($payments)): ?>
                <tr>
                    <td colspan="7" class="no-payments">
                        No payments found matching the selected criteria.
                    </td>
                </tr>
            <?php else: ?>
                <?php foreach ($payments as $payment): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($payment['loan_name']); ?></td>
                        <td><?php echo htmlspecialchars($payment['payment_date']); ?></td>
                        <td class="amount"><?php echo number_format($payment['payment_amount'], 2); ?></td>
                        <td>
                            <span class="payment-method <?php echo str_replace('_', '', $payment['payment_method']); ?>">
                                <?php echo ucfirst(str_replace('_', ' ', $payment['payment_method'])); ?>
                            </span>
                        </td>
                        <td class="amount"><?php echo number_format($payment['remaining_balance'], 2); ?></td>
                        <td><?php echo htmlspecialchars($payment['recorded_by_name']); ?></td>
                        <td><?php echo htmlspecialchars($payment['created_at']); ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>

    <div class="footer">
        <p>This is a computer-generated report. For official purposes, please contact the school administration.</p>
        <p>Report generated by School Management System</p>
    </div>

    <script>
        // Auto-print when page loads
        window.onload = function() {
            window.print();
        };
    </script>
</body>
</html>
