<?php
require_once '../config.php';

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['add_loan'])) {
        $name = trim($_POST['name']);
        $liability_type = $_POST['liability_type'];
        $original_amount = floatval($_POST['original_amount']);
        $current_balance = floatval($_POST['current_balance']);
        $interest_rate = floatval($_POST['interest_rate']);
        $start_date = $_POST['start_date'];
        $end_date = $_POST['end_date'];
        $monthly_payment = floatval($_POST['monthly_payment']);
        $description = trim($_POST['description']);

        $stmt = $pdo->prepare("
            INSERT INTO liabilities (name, liability_type, original_amount, current_balance, interest_rate, start_date, end_date, monthly_payment, description)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");

        if ($stmt->execute([$name, $liability_type, $original_amount, $current_balance, $interest_rate, $start_date, $end_date, $monthly_payment, $description])) {
            $loanId = $pdo->lastInsertId();
            $message = 'Loan/liability added successfully.';

            // Create proper accounting journal entry
            require_once 'loan_accounting_functions.php';
            try {
                createLoanEntry($pdo, $loanId, $current_balance, $start_date, $_SESSION['user_id']);
            } catch (Exception $e) {
                $message .= ' Warning: Accounting entry could not be created: ' . $e->getMessage();
            }
        } else {
            $error = 'Failed to add loan/liability.';
        }
    } elseif (isset($_POST['edit_loan'])) {
        $loan_id = $_POST['loan_id'];
        $name = trim($_POST['name']);
        $liability_type = $_POST['liability_type'];
        $original_amount = floatval($_POST['original_amount']);
        $current_balance = floatval($_POST['current_balance']);
        $interest_rate = floatval($_POST['interest_rate']);
        $start_date = $_POST['start_date'];
        $end_date = $_POST['end_date'];
        $monthly_payment = floatval($_POST['monthly_payment']);
        $description = trim($_POST['description']);

        $stmt = $pdo->prepare("
            UPDATE liabilities
            SET name = ?, liability_type = ?, original_amount = ?, current_balance = ?, interest_rate = ?, start_date = ?, end_date = ?, monthly_payment = ?, description = ?
            WHERE id = ?
        ");

        if ($stmt->execute([$name, $liability_type, $original_amount, $current_balance, $interest_rate, $start_date, $end_date, $monthly_payment, $description, $loan_id])) {
            $message = 'Loan/liability updated successfully.';
        } else {
            $error = 'Failed to update loan/liability.';
        }
    } elseif (isset($_POST['record_payment'])) {
        $loan_id = $_POST['loan_id'];
        $payment_amount = floatval($_POST['payment_amount']);
        $payment_date = $_POST['payment_date'];
        $payment_method = $_POST['payment_method'] ?? 'bank_transfer'; // Default to bank_transfer if not specified

        // Get current balance
        $stmt = $pdo->prepare("SELECT current_balance FROM liabilities WHERE id = ?");
        $stmt->execute([$loan_id]);
        $current_balance = floatval($stmt->fetch()['current_balance']);

        $new_balance = max(0, $current_balance - $payment_amount);

        // Update balance
        $stmt = $pdo->prepare("UPDATE liabilities SET current_balance = ? WHERE id = ?");
        $stmt->execute([$new_balance, $loan_id]);

        // Insert payment record into loan_payments table
        $stmt = $pdo->prepare("
            INSERT INTO loan_payments (loan_id, payment_amount, payment_date, payment_method, recorded_by)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$loan_id, $payment_amount, $payment_date, $payment_method, $_SESSION['user_id']]);

        // Create proper accounting journal entry for loan payment
        require_once 'loan_accounting_functions.php';
        try {
            createLoanPaymentEntry($pdo, $loan_id, $payment_amount, $payment_method, $payment_date, $_SESSION['user_id']);
            $message = 'Loan payment recorded successfully with journal entry. New balance: GHC ' . number_format($new_balance, 2);
        } catch (Exception $e) {
            $message = 'Loan payment recorded but accounting entry failed: ' . $e->getMessage() . '. New balance: GHC ' . number_format($new_balance, 2);
        }
    }
}

// Get filters
$filter_type = $_GET['type'] ?? '';
$filter_status = $_GET['status'] ?? '';

// Build query
$query = "SELECT * FROM liabilities WHERE 1=1";
$params = [];

if ($filter_type) {
    $query .= " AND liability_type = ?";
    $params[] = $filter_type;
}

if ($filter_status === 'active') {
    $query .= " AND current_balance > 0";
} elseif ($filter_status === 'paid') {
    $query .= " AND current_balance = 0";
}

$query .= " ORDER BY start_date DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$loans = $stmt->fetchAll();

// Get loan for edit
$edit_loan = null;
if (isset($_GET['edit'])) {
    $loan_id = $_GET['edit'];
    $stmt = $pdo->prepare("SELECT * FROM liabilities WHERE id = ?");
    $stmt->execute([$loan_id]);
    $edit_loan = $stmt->fetch();
}

include 'includes/accounting_header.php';
?>

<style>
.loan-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.loan-header {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.loan-header h1 {
    color: #2c3e50;
    margin: 0 0 10px 0;
}

.loan-header p {
    color: #7f8c8d;
    margin: 0;
}

.filters-section {
    background: white;
    padding: 20px;
    border-radius: 10px;
    margin-bottom: 20px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.filters-form {
    display: flex;
    gap: 20px;
    align-items: end;
    flex-wrap: wrap;
}

.filter-group {
    display: flex;
    flex-direction: column;
    min-width: 150px;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.filter-group select {
    padding: 8px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.form-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 20px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.form-section h2 {
    color: #2c3e50;
    margin-bottom: 20px;
    border-bottom: 2px solid #e9ecef;
    padding-bottom: 10px;
}

.form-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 20px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group.full-width {
    grid-column: 1 / -1;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.form-group input,
.form-group select,
.form-group textarea {
    padding: 10px 12px;
    border: 1px solid #ced4da;
    border-radius: 5px;
    font-size: 14px;
    transition: border-color 0.2s;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #007bff;
    box-shadow: 0 0 0 2px rgba(0,123,255,0.25);
}

.form-group textarea {
    resize: vertical;
    min-height: 80px;
}

.form-actions {
    display: flex;
    gap: 15px;
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid #e9ecef;
}

.loans-table-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.table-container {
    overflow-x: auto;
}

.loans-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
}

.loans-table th,
.loans-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.loans-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
    position: sticky;
    top: 0;
}

.loans-table tr:hover {
    background: #f8f9fa;
}

.loan-type {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 0.8em;
    font-weight: bold;
    text-transform: uppercase;
}

.loan-type.loan {
    background: #dc3545;
    color: white;
}

.loan-type.mortgage {
    background: #fd7e14;
    color: white;
}

.loan-type.credit_card {
    background: #6f42c1;
    color: white;
}

.loan-type.accounts_payable {
    background: #20c997;
    color: white;
}

.amount {
    font-weight: 600;
    color: #2c3e50;
}

.balance-positive {
    color: #dc3545;
}

.balance-zero {
    color: #28a745;
}

.btn {
    padding: 8px 16px;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.2s;
}

.btn-primary {
    background: #007bff;
    color: white;
}

.btn-primary:hover {
    background: #0056b3;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #545b62;
}

.btn-success {
    background: #28a745;
    color: white;
}

.btn-success:hover {
    background: #1e7e34;
}

.btn-warning {
    background: #ffc107;
    color: #212529;
}

.btn-warning:hover {
    background: #e0a800;
}

.message {
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border-color: #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border-color: #f5c6cb;
}

.no-loans {
    text-align: center;
    color: #7f8c8d;
    font-style: italic;
    padding: 40px;
}

.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
    backdrop-filter: blur(5px);
}

.modal-content {
    background: white;
    margin: 10% auto;
    padding: 25px;
    border-radius: 10px;
    width: 90%;
    max-width: 500px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.3);
}

.modal-content h3 {
    color: #2c3e50;
    margin-bottom: 20px;
    text-align: center;
}

.close {
    position: absolute;
    right: 15px;
    top: 15px;
    color: #aaa;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
    transition: color 0.3s;
}

.close:hover {
    color: #2E7D32;
}

@media (max-width: 768px) {
    .filters-form {
        flex-direction: column;
        align-items: stretch;
    }

    .filter-group {
        min-width: auto;
    }

    .form-row {
        grid-template-columns: 1fr;
    }

    .form-actions {
        flex-direction: column;
    }
}
</style>

<div class="loan-container">
    <div class="loan-header">
        <h1>💰 Loan & Liability Management</h1>
        <p>Track loans, mortgages, and other financial obligations</p>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Filters -->
    <div class="filters-section">
        <form method="get" class="filters-form">
            <div class="filter-group">
                <label for="type">Type:</label>
                <select name="type" id="type">
                    <option value="">All Types</option>
                    <option value="loan" <?php echo $filter_type == 'loan' ? 'selected' : ''; ?>>Loan</option>
                    <option value="mortgage" <?php echo $filter_type == 'mortgage' ? 'selected' : ''; ?>>Mortgage</option>
                    <option value="credit_card" <?php echo $filter_type == 'credit_card' ? 'selected' : ''; ?>>Credit Card</option>
                    <option value="accounts_payable" <?php echo $filter_type == 'accounts_payable' ? 'selected' : ''; ?>>Accounts Payable</option>
                </select>
            </div>

            <div class="filter-group">
                <label for="status">Status:</label>
                <select name="status" id="status">
                    <option value="">All Status</option>
                    <option value="active" <?php echo $filter_status == 'active' ? 'selected' : ''; ?>>Active (Balance > 0)</option>
                    <option value="paid" <?php echo $filter_status == 'paid' ? 'selected' : ''; ?>>Paid Off</option>
                </select>
            </div>

            <div class="filter-group">
                <button type="submit" class="btn btn-primary">Filter</button>
                <a href="manage_loans.php" class="btn btn-secondary">Clear</a>
            </div>
        </form>
    </div>

    <!-- Add/Edit Loan Form -->
    <div class="form-section">
        <h2><?php echo $edit_loan ? 'Edit Loan/Liability' : 'Add New Loan/Liability'; ?></h2>

        <form method="post" action="manage_loans.php">
            <?php if ($edit_loan): ?>
                <input type="hidden" name="loan_id" value="<?php echo $edit_loan['id']; ?>">
            <?php endif; ?>

            <div class="form-row">
                <div class="form-group">
                    <label for="name">Name *</label>
                    <input type="text" name="name" id="name" required
                           value="<?php echo htmlspecialchars($edit_loan['name'] ?? ''); ?>"
                           placeholder="e.g., School Building Loan">
                </div>

                <div class="form-group">
                    <label for="liability_type">Type *</label>
                    <select name="liability_type" id="liability_type" required>
                        <option value="loan" <?php echo ($edit_loan['liability_type'] ?? '') == 'loan' ? 'selected' : ''; ?>>Loan</option>
                        <option value="mortgage" <?php echo ($edit_loan['liability_type'] ?? '') == 'mortgage' ? 'selected' : ''; ?>>Mortgage</option>
                        <option value="credit_card" <?php echo ($edit_loan['liability_type'] ?? '') == 'credit_card' ? 'selected' : ''; ?>>Credit Card</option>
                        <option value="accounts_payable" <?php echo ($edit_loan['liability_type'] ?? '') == 'accounts_payable' ? 'selected' : ''; ?>>Accounts Payable</option>
                    </select>
                </div>
            </div>

            <div class="form-row">
                <div class="form-group">
                    <label for="original_amount">Original Amount (GHC) *</label>
                    <input type="number" name="original_amount" id="original_amount" step="0.01" min="0" required
                           value="<?php echo htmlspecialchars($edit_loan['original_amount'] ?? ''); ?>"
                           placeholder="1000000.00">
                </div>

                <div class="form-group">
                    <label for="current_balance">Current Balance (GHC) *</label>
                    <input type="number" name="current_balance" id="current_balance" step="0.01" min="0" required
                           value="<?php echo htmlspecialchars($edit_loan['current_balance'] ?? ''); ?>"
                           placeholder="750000.00">
                </div>

                <div class="form-group">
                    <label for="interest_rate">Interest Rate (%) *</label>
                    <input type="number" name="interest_rate" id="interest_rate" step="0.01" min="0" required
                           value="<?php echo htmlspecialchars($edit_loan['interest_rate'] ?? ''); ?>"
                           placeholder="8.5">
                </div>
            </div>

            <div class="form-row">
                <div class="form-group">
                    <label for="start_date">Start Date *</label>
                    <input type="date" name="start_date" id="start_date" required
                           value="<?php echo htmlspecialchars($edit_loan['start_date'] ?? ''); ?>">
                </div>

                <div class="form-group">
                    <label for="end_date">End Date *</label>
                    <input type="date" name="end_date" id="end_date" required
                           value="<?php echo htmlspecialchars($edit_loan['end_date'] ?? ''); ?>">
                </div>

                <div class="form-group">
                    <label for="monthly_payment">Monthly Payment (GHC) *</label>
                    <input type="number" name="monthly_payment" id="monthly_payment" step="0.01" min="0" required
                           value="<?php echo htmlspecialchars($edit_loan['monthly_payment'] ?? ''); ?>"
                           placeholder="15000.00">
                </div>
            </div>

            <div class="form-group full-width">
                <label for="description">Description</label>
                <textarea name="description" id="description" rows="3"
                          placeholder="Additional details about this loan/liability..."><?php echo htmlspecialchars($edit_loan['description'] ?? ''); ?></textarea>
            </div>

            <div class="form-actions">
                <button type="submit" name="<?php echo $edit_loan ? 'edit_loan' : 'add_loan'; ?>" class="btn btn-success">
                    <?php echo $edit_loan ? 'Update Loan/Liability' : 'Add Loan/Liability'; ?>
                </button>
                <?php if ($edit_loan): ?>
                    <a href="manage_loans.php" class="btn btn-secondary">Cancel Edit</a>
                <?php endif; ?>
                <a href="balance_sheet_report.php" class="btn btn-primary">View Balance Sheet</a>
            </div>
        </form>
    </div>

    <!-- Loans Table -->
    <div class="loans-table-section">
        <h2>Existing Loans & Liabilities</h2>

        <div class="table-container">
            <table class="loans-table">
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Type</th>
                        <th>Original Amount</th>
                        <th>Current Balance</th>
                        <th>Interest Rate</th>
                        <th>Monthly Payment</th>
                        <th>End Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($loans)): ?>
                        <tr>
                            <td colspan="8" class="no-loans">
                                No loans or liabilities found matching the selected criteria.
                                <?php if (!$filter_type && !$filter_status): ?>
                                    <br><br>Use the form above to add your first loan or liability.
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($loans as $loan): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($loan['name']); ?></td>
                                <td>
                                    <span class="loan-type <?php echo $loan['liability_type']; ?>">
                                        <?php echo ucfirst(str_replace('_', ' ', $loan['liability_type'])); ?>
                                    </span>
                                </td>
                                <td class="amount">GHC <?php echo number_format($loan['original_amount'], 2); ?></td>
                                <td class="amount <?php echo $loan['current_balance'] > 0 ? 'balance-positive' : 'balance-zero'; ?>">
                                    GHC <?php echo number_format($loan['current_balance'], 2); ?>
                                </td>
                                <td><?php echo htmlspecialchars($loan['interest_rate']); ?>%</td>
                                <td class="amount">GHC <?php echo number_format($loan['monthly_payment'], 2); ?></td>
                                <td><?php echo htmlspecialchars($loan['end_date']); ?></td>
                                <td>
                                    <a href="?edit=<?php echo $loan['id']; ?>" class="btn btn-warning">Edit</a>
                                    <button class="btn btn-success" onclick="openPaymentModal(<?php echo $loan['id']; ?>, '<?php echo addslashes($loan['name']); ?>')">Record Payment</button>
                                    <a href="loan_payment_history.php?loan_id=<?php echo $loan['id']; ?>" class="btn btn-primary">View History</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Payment Modal -->
    <div id="paymentModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closePaymentModal()">&times;</span>
            <h3>Record Loan Payment</h3>
            <form method="post" action="manage_loans.php">
                <input type="hidden" name="loan_id" id="modal_loan_id">
                <input type="hidden" name="loan_name" id="modal_loan_name">

                <div class="form-group">
                    <label for="payment_amount">Payment Amount (GHC) *</label>
                    <input type="number" name="payment_amount" id="payment_amount" step="0.01" min="0.01" required placeholder="5000.00">
                </div>

                <div class="form-group">
                    <label for="payment_date">Payment Date *</label>
                    <input type="date" name="payment_date" id="payment_date" required value="<?php echo date('Y-m-d'); ?>">
                </div>

                <div class="form-group">
                    <label for="payment_method">Payment Method *</label>
                    <select name="payment_method" id="payment_method" required>
                        <option value="cash">Cash</option>
                        <option value="bank_transfer" selected>Bank Transfer</option>
                        <option value="cheque">Cheque</option>
                        <option value="mobile_money">Mobile Money</option>
                    </select>
                </div>

                <div class="form-actions">
                    <button type="submit" name="record_payment" class="btn btn-success">Record Payment</button>
                    <button type="button" class="btn btn-secondary" onclick="closePaymentModal()">Cancel</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function openPaymentModal(loanId, loanName) {
    document.getElementById('modal_loan_id').value = loanId;
    document.getElementById('modal_loan_name').value = loanName;
    document.getElementById('paymentModal').style.display = 'block';
}

function closePaymentModal() {
    document.getElementById('paymentModal').style.display = 'none';
    document.getElementById('payment_amount').value = '';
    document.getElementById('payment_date').value = '<?php echo date('Y-m-d'); ?>';
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('paymentModal');
    if (event.target == modal) {
        closePaymentModal();
    }
}
</script>

<?php
include '../includes/admin_footer.php';
?>
