<?php
require_once '../config.php';

$message = isset($_SESSION['budget_message']) ? $_SESSION['budget_message'] : '';
$error = isset($_SESSION['budget_error']) ? $_SESSION['budget_error'] : '';

// Clear session messages after displaying
unset($_SESSION['budget_message']);
unset($_SESSION['budget_error']);

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['add_budget'])) {
        $name = trim($_POST['name']);
        $academic_year_id = $_POST['academic_year_id'];
        $term_id = isset($_POST['term_id']) ? $_POST['term_id'] : null;
        $category_id = $_POST['category_id'];
        $budget_type = $_POST['budget_type'];
        $planned_amount = floatval($_POST['planned_amount']);
        $description = trim($_POST['description']);
        $expense_type = isset($_POST['expense_type']) ? $_POST['expense_type'] : null;
        $frequency = isset($_POST['frequency']) ? $_POST['frequency'] : null;

        $stmt = $pdo->prepare("
            INSERT INTO budgets (name, academic_year_id, term_id, category_id, budget_type, planned_amount, description, expense_type, frequency, created_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");

        if ($stmt->execute([$name, $academic_year_id, $term_id, $category_id, $budget_type, $planned_amount, $description, $expense_type, $frequency, $_SESSION['user_id']])) {
            $_SESSION['budget_message'] = 'Budget added successfully.';
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        } else {
            $error = 'Failed to add budget.';
        }
    } elseif (isset($_POST['edit_budget'])) {
        $budget_id = $_POST['budget_id'];
        $name = trim($_POST['name']);
        $academic_year_id = $_POST['academic_year_id'];
        $term_id = isset($_POST['term_id']) ? $_POST['term_id'] : null;
        $category_id = $_POST['category_id'];
        $budget_type = $_POST['budget_type'];
        $planned_amount = floatval($_POST['planned_amount']);
        $description = trim($_POST['description']);
        $expense_type = isset($_POST['expense_type']) ? $_POST['expense_type'] : null;
        $frequency = isset($_POST['frequency']) ? $_POST['frequency'] : null;

        $stmt = $pdo->prepare("
            UPDATE budgets
            SET name = ?, academic_year_id = ?, term_id = ?, category_id = ?, budget_type = ?, planned_amount = ?, description = ?, expense_type = ?, frequency = ?
            WHERE id = ?
        ");

        if ($stmt->execute([$name, $academic_year_id, $term_id, $category_id, $budget_type, $planned_amount, $description, $expense_type, $frequency, $budget_id])) {
            $_SESSION['budget_message'] = 'Budget updated successfully.';
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        } else {
            $error = 'Failed to update budget.';
        }
    }
}

// Get filters
$filter_academic_year = $_GET['academic_year'] ?? '';
$filter_category = $_GET['category'] ?? '';
$filter_type = $_GET['type'] ?? '';

// Build query
$query = "
    SELECT b.*, ay.year as academic_year, bc.name as category_name, t.name as term_name
    FROM budgets b
    LEFT JOIN academic_years ay ON b.academic_year_id = ay.id
    LEFT JOIN budget_categories bc ON b.category_id = bc.id
    LEFT JOIN terms t ON b.term_id = t.id
    WHERE 1=1
";
$params = [];

if ($filter_academic_year) {
    $query .= " AND b.academic_year_id = ?";
    $params[] = $filter_academic_year;
}
if ($filter_category) {
    $query .= " AND b.category_id = ?";
    $params[] = $filter_category;
}
if ($filter_type) {
    $query .= " AND b.budget_type = ?";
    $params[] = $filter_type;
}

$query .= " ORDER BY ay.year DESC, bc.name ASC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$budgets = $stmt->fetchAll();

// Categorize budgets
$income_budgets = [];
$recurring_budgets = [];
$non_recurring_budgets = [];

foreach ($budgets as $budget) {
    if ($budget['budget_type'] === 'income') {
        $income_budgets[] = $budget;
    } elseif ($budget['budget_type'] === 'expense') {
        if (in_array($budget['expense_type'], ['fixed', 'recurring'])) {
            $recurring_budgets[] = $budget;
        } elseif (in_array($budget['expense_type'], ['variable', 'one-time'])) {
            $non_recurring_budgets[] = $budget;
        }
    }
}

// Calculate budget totals
$total_budgeted_income = array_sum(array_column($income_budgets, 'planned_amount'));
$total_budgeted_expenses = array_sum(array_column($recurring_budgets, 'planned_amount')) + array_sum(array_column($non_recurring_budgets, 'planned_amount'));
$budgeted_surplus_deficit = $total_budgeted_income - $total_budgeted_expenses;

// Get data for dropdowns
$academic_years = $pdo->query("SELECT id, year FROM academic_years ORDER BY year DESC")->fetchAll();
$budget_categories = $pdo->query("SELECT id, name FROM budget_categories ORDER BY name")->fetchAll();
$terms = $pdo->query("SELECT id, name FROM terms WHERE is_active = 1 ORDER BY start_date")->fetchAll();

// Get current revenue by type (from journal entries - double-entry system)
$stmt = $pdo->prepare("
    SELECT
        coa.account_code,
        COALESCE(SUM(jel.credit_amount), 0) as revenue_amount
    FROM journal_entries je
    JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
    JOIN chart_of_accounts coa ON jel.account_id = coa.id
    WHERE YEAR(je.transaction_date) = YEAR(CURRENT_DATE)
    AND coa.account_type = 'revenue'
    AND jel.credit_amount > 0
    AND je.status = 'posted'
    AND coa.account_code IN ('4110', '4001')
    GROUP BY coa.account_code
");
$stmt->execute();
$revenue_results = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Initialize revenue amounts
$tuition_revenue = 0;
$registration_revenue = 0;

foreach ($revenue_results as $result) {
    if ($result['account_code'] === '4110') {
        $tuition_revenue = $result['revenue_amount'];
    } elseif ($result['account_code'] === '4001') {
        $registration_revenue = $result['revenue_amount'];
    }
}

// Calculate total revenue
$total_revenue = $tuition_revenue + $registration_revenue;

// Get budget for edit
$edit_budget = null;
if (isset($_GET['edit'])) {
    $budget_id = $_GET['edit'];
    $stmt = $pdo->prepare("
        SELECT b.*, ay.year as academic_year, bc.name as category_name, t.name as term_name
        FROM budgets b
        LEFT JOIN academic_years ay ON b.academic_year_id = ay.id
        LEFT JOIN budget_categories bc ON b.category_id = bc.id
        LEFT JOIN terms t ON b.term_id = t.id
        WHERE b.id = ?
    ");
    $stmt->execute([$budget_id]);
    $edit_budget = $stmt->fetch();
}

include 'includes/accounting_header.php';
?>

<style>
.budget-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.budget-header {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.budget-header h1 {
    color: #2c3e50;
    margin: 0 0 10px 0;
}

.budget-header p {
    color: #7f8c8d;
    margin: 0;
}

.filters-section {
    background: white;
    padding: 20px;
    border-radius: 10px;
    margin-bottom: 20px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.filters-form {
    display: flex;
    gap: 20px;
    align-items: end;
    flex-wrap: wrap;
}

.filter-group {
    display: flex;
    flex-direction: column;
    min-width: 150px;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.filter-group select {
    padding: 8px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.form-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 20px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.form-section h2 {
    color: #2c3e50;
    margin-bottom: 20px;
    border-bottom: 2px solid #e9ecef;
    padding-bottom: 10px;
}

.form-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 20px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group.full-width {
    grid-column: 1 / -1;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.form-group input,
.form-group select,
.form-group textarea {
    padding: 10px 12px;
    border: 1px solid #ced4da;
    border-radius: 5px;
    font-size: 14px;
    transition: border-color 0.2s;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #007bff;
    box-shadow: 0 0 0 2px rgba(0,123,255,0.25);
}

.form-group textarea {
    resize: vertical;
    min-height: 80px;
}

/* Conditional Expense Fields */
.expense-fields {
    display: none;
}

.form-actions {
    display: flex;
    gap: 15px;
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid #e9ecef;
}

.budgets-table-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.table-container {
    overflow-x: auto;
}

.budgets-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
}

.budgets-table th,
.budgets-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.budgets-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
    position: sticky;
    top: 0;
}

.budgets-table tr:hover {
    background: #f8f9fa;
}

.budget-type {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 0.8em;
    font-weight: bold;
    text-transform: uppercase;
}

.budget-type.income {
    background: #d4edda;
    color: #155724;
}

.budget-type.expense {
    background: #f8d7da;
    color: #721c24;
}

.amount {
    font-weight: 600;
    color: #2c3e50;
}

.btn {
    padding: 8px 16px;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.2s;
}

.btn-primary {
    background: #007bff;
    color: white;
}

.btn-primary:hover {
    background: #0056b3;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #545b62;
}

.btn-success {
    background: #28a745;
    color: white;
}

.btn-success:hover {
    background: #1e7e34;
}

.message {
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border-color: #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border-color: #f5c6cb;
}

.no-budgets {
    text-align: center;
    color: #7f8c8d;
    font-style: italic;
    padding: 40px;
}

/* Revenue Display Styles */
.revenue-display-section {
    margin-bottom: 20px;
}

.revenue-cards {
    display: flex;
    gap: 20px;
    flex-wrap: wrap;
}

.revenue-card {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    border-radius: 15px;
    padding: 25px;
    display: flex;
    align-items: center;
    box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
    color: white;
    flex: 1;
    min-width: 300px;
}

.revenue-card.tuition {
    background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
    box-shadow: 0 4px 15px rgba(0, 123, 255, 0.3);
}

.revenue-card.registration {
    background: linear-gradient(135deg, #fd7e14 0%, #e8680f 100%);
    box-shadow: 0 4px 15px rgba(253, 126, 20, 0.3);
}

.revenue-icon {
    font-size: 3rem;
    margin-right: 20px;
    opacity: 0.9;
}

.revenue-content h3 {
    margin: 0 0 10px 0;
    font-size: 1.2rem;
    font-weight: 600;
    opacity: 0.9;
}

.revenue-amount {
    font-size: 2.5rem;
    font-weight: bold;
    margin: 0 0 5px 0;
    text-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.revenue-note {
    margin: 0;
    font-size: 0.9rem;
    opacity: 0.8;
}

/* Budget Summary Styles */
.summary-cards {
    display: flex;
    gap: 20px;
    flex-wrap: wrap;
    margin-top: 20px;
}

.summary-card {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    border-radius: 15px;
    padding: 25px;
    display: flex;
    align-items: center;
    box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
    color: white;
    flex: 1;
    min-width: 250px;
    transition: transform 0.2s;
}

.summary-card:hover {
    transform: translateY(-2px);
}

.summary-card.surplus {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
}

.summary-card.deficit {
    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
    box-shadow: 0 4px 15px rgba(220, 53, 69, 0.3);
}

.summary-icon {
    font-size: 3rem;
    margin-right: 20px;
    opacity: 0.9;
}

.summary-content h3 {
    margin: 0 0 10px 0;
    font-size: 1.2rem;
    font-weight: 600;
    opacity: 0.9;
}

.summary-amount {
    font-size: 2.5rem;
    font-weight: bold;
    margin: 0 0 5px 0;
    text-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.summary-note {
    margin: 0;
    font-size: 0.9rem;
    opacity: 0.8;
}

@media (max-width: 768px) {
    .filters-form {
        flex-direction: column;
        align-items: stretch;
    }

    .filter-group {
        min-width: auto;
    }

    .form-row {
        grid-template-columns: 1fr;
    }

    .form-actions {
        flex-direction: column;
    }

    .summary-cards {
        flex-direction: column;
    }

    .summary-card {
        min-width: auto;
    }
}

/* Print Styles for Professional Budget Report */
@media print {
    /* Hide non-essential elements */
    .budget-header,
    .filters-section,
    .form-section,
    .budgets-table-section:not(.budgets-table-section:last-of-type),
    .btn,
    .message,
    script {
        display: none !important;
    }

    /* Show only revenue display and budget summary */
    .revenue-display-section,
    .budgets-table-section:last-of-type {
        display: block !important;
        page-break-inside: avoid;
    }

    /* Adjust layout for print */
    body {
        font-size: 12pt;
        line-height: 1.4;
        color: #000;
        background: white !important;
    }

    .budget-container {
        max-width: none;
        margin: 0;
        padding: 20px;
        background: white !important;
    }

    .revenue-display-section {
        margin-bottom: 30px;
    }

    .revenue-cards {
        display: flex;
        justify-content: space-between;
        gap: 20px;
    }

    .revenue-card {
        flex: 1;
        border: 1px solid #ddd;
        border-radius: 8px;
        padding: 15px;
        background: white !important;
        box-shadow: none !important;
        color: #000 !important;
    }

    .revenue-icon {
        font-size: 2rem;
        margin-right: 15px;
    }

    .revenue-content h3 {
        font-size: 1.1rem;
        margin: 0 0 8px 0;
    }

    .revenue-amount {
        font-size: 1.8rem;
        font-weight: bold;
        margin: 0 0 5px 0;
    }

    .budgets-table-section h2 {
        font-size: 1.5rem;
        margin-bottom: 20px;
        text-align: center;
        border-bottom: 2px solid #000;
        padding-bottom: 10px;
    }

    .summary-cards {
        display: flex;
        justify-content: space-between;
        gap: 20px;
    }

    .summary-card {
        flex: 1;
        border: 1px solid #ddd;
        border-radius: 8px;
        padding: 15px;
        background: white !important;
        box-shadow: none !important;
        color: #000 !important;
        text-align: center;
    }

    .summary-icon {
        font-size: 2rem;
        margin-bottom: 10px;
    }

    .summary-content h3 {
        font-size: 1.1rem;
        margin: 0 0 8px 0;
    }

    .summary-amount {
        font-size: 1.8rem;
        font-weight: bold;
        margin: 0 0 5px 0;
    }

    .summary-note {
        font-size: 0.9rem;
    }

    /* Page break before summary if needed */
    .budgets-table-section:last-of-type {
        page-break-before: always;
    }
}
</style>

<div class="budget-container">
    <div class="budget-header">
        <h1>💰 Budget Management</h1>
        <p>Create and manage budgets for different categories and academic years</p>
    </div>

    <!-- Current Revenue Display -->
    <div class="revenue-display-section">
        <div class="revenue-cards">
            <div class="revenue-card tuition">
                <div class="revenue-icon">🎓</div>
                <div class="revenue-content">
                    <h3>Tuition Revenue</h3>
                    <div class="revenue-amount">GHC <?php echo number_format($tuition_revenue, 2); ?></div>
                    <p class="revenue-note">Current year tuition fees collected</p>
                </div>
            </div>
            <div class="revenue-card registration">
                <div class="revenue-icon">📝</div>
                <div class="revenue-content">
                    <h3>Registration Revenue</h3>
                    <div class="revenue-amount">GHC <?php echo number_format($registration_revenue, 2); ?></div>
                    <p class="revenue-note">Current year registration fees collected</p>
                </div>
            </div>
            <div class="revenue-card total">
                <div class="revenue-icon">💰</div>
                <div class="revenue-content">
                    <h3>Total Revenue</h3>
                    <div class="revenue-amount">GHC <?php echo number_format($total_revenue, 2); ?></div>
                    <p class="revenue-note">Combined tuition and registration revenue</p>
                </div>
            </div>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Filters -->
    <div class="filters-section">
        <form method="get" class="filters-form">
            <div class="filter-group">
                <label for="academic_year">Academic Year:</label>
                <select name="academic_year" id="academic_year">
                    <option value="">All Years</option>
                    <?php foreach ($academic_years as $year): ?>
                        <option value="<?php echo $year['id']; ?>" <?php echo $filter_academic_year == $year['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($year['year']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <label for="category">Category:</label>
                <select name="category" id="category">
                    <option value="">All Categories</option>
                    <?php foreach ($budget_categories as $category): ?>
                        <option value="<?php echo $category['id']; ?>" <?php echo $filter_category == $category['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($category['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <label for="type">Type:</label>
                <select name="type" id="type">
                    <option value="">All Types</option>
                    <option value="income" <?php echo $filter_type == 'income' ? 'selected' : ''; ?>>Income</option>
                    <option value="expense" <?php echo $filter_type == 'expense' ? 'selected' : ''; ?>>Expense</option>
                </select>
            </div>

            <div class="filter-group">
                <button type="submit" class="btn btn-primary">Filter</button>
                <a href="manage_budgets.php" class="btn btn-secondary">Clear</a>
            </div>
        </form>
    </div>

    <!-- Add/Edit Budget Form -->
    <div class="form-section">
        <h2><?php echo $edit_budget ? 'Edit Budget' : 'Add New Budget'; ?></h2>

        <form method="post" action="manage_budgets.php">
            <?php if ($edit_budget): ?>
                <input type="hidden" name="budget_id" value="<?php echo $edit_budget['id']; ?>">
            <?php endif; ?>

            <div class="form-row">
                <div class="form-group">
                    <label for="name">Budget Name *</label>
                    <input type="text" name="name" id="name" required
                           value="<?php echo htmlspecialchars($edit_budget['name'] ?? ''); ?>"
                           placeholder="e.g., Annual Tuition Revenue">
                </div>

                <div class="form-group">
                    <label for="academic_year_id">Academic Year *</label>
                    <select name="academic_year_id" id="academic_year_id" required>
                        <option value="">Select Academic Year</option>
                        <?php foreach ($academic_years as $year): ?>
                            <option value="<?php echo $year['id']; ?>"
                                    <?php echo ($edit_budget['academic_year_id'] ?? '') == $year['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($year['year']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="term_id">Term</label>
                    <select name="term_id" id="term_id">
                        <option value="">Select Term (Optional)</option>
                        <?php foreach ($terms as $term): ?>
                            <option value="<?php echo $term['id']; ?>"
                                    <?php echo ($edit_budget['term_id'] ?? '') == $term['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($term['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>

            <div class="form-row">
                <div class="form-group">
                    <label for="category_id">Category *</label>
                    <select name="category_id" id="category_id" required>
                        <option value="">Select Category</option>
                        <?php foreach ($budget_categories as $category): ?>
                            <option value="<?php echo $category['id']; ?>"
                                    <?php echo ($edit_budget['category_id'] ?? '') == $category['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($category['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="budget_type">Budget Type *</label>
                    <select name="budget_type" id="budget_type" required>
                        <option value="income" <?php echo ($edit_budget['budget_type'] ?? '') == 'income' ? 'selected' : ''; ?>>Income</option>
                        <option value="expense" <?php echo ($edit_budget['budget_type'] ?? '') == 'expense' ? 'selected' : ''; ?>>Expense</option>
                    </select>
                </div>

                <div class="form-group">
                    <label for="planned_amount">Planned Amount (GHC) *</label>
                    <input type="number" name="planned_amount" id="planned_amount" step="0.01" min="0" required
                           value="<?php echo htmlspecialchars($edit_budget['planned_amount'] ?? ''); ?>"
                           placeholder="0.00">
                </div>
            </div>

            <!-- Conditional Expense Fields -->
            <div class="form-row expense-fields">
                <div class="form-group">
                    <label for="expense_type">Expense Type *</label>
                    <select name="expense_type" id="expense_type">
                        <option value="fixed" <?php echo ($edit_budget['expense_type'] ?? '') == 'fixed' ? 'selected' : ''; ?>>Fixed</option>
                        <option value="variable" <?php echo ($edit_budget['expense_type'] ?? '') == 'variable' ? 'selected' : ''; ?>>Variable</option>
                        <option value="recurring" <?php echo ($edit_budget['expense_type'] ?? '') == 'recurring' ? 'selected' : ''; ?>>Recurring</option>
                        <option value="one-time" <?php echo ($edit_budget['expense_type'] ?? '') == 'one-time' ? 'selected' : ''; ?>>One-time</option>
                    </select>
                </div>

                <div class="form-group">
                    <label for="frequency">Frequency *</label>
                    <select name="frequency" id="frequency">
                        <option value="monthly" <?php echo ($edit_budget['frequency'] ?? '') == 'monthly' ? 'selected' : ''; ?>>Monthly</option>
                        <option value="quarterly" <?php echo ($edit_budget['frequency'] ?? '') == 'quarterly' ? 'selected' : ''; ?>>Quarterly</option>
                        <option value="annually" <?php echo ($edit_budget['frequency'] ?? '') == 'annually' ? 'selected' : ''; ?>>Annually</option>
                        <option value="one-time" <?php echo ($edit_budget['frequency'] ?? '') == 'one-time' ? 'selected' : ''; ?>>One-time</option>
                    </select>
                </div>
            </div>

            <div class="form-group full-width">
                <label for="description">Description</label>
                <textarea name="description" id="description" rows="3"
                          placeholder="Additional details about this budget..."><?php echo htmlspecialchars($edit_budget['description'] ?? ''); ?></textarea>
            </div>

            <div class="form-actions">
                <button type="submit" name="<?php echo $edit_budget ? 'edit_budget' : 'add_budget'; ?>" class="btn btn-success">
                    <?php echo $edit_budget ? 'Update Budget' : 'Add Budget'; ?>
                </button>
                <?php if ($edit_budget): ?>
                    <a href="manage_budgets.php" class="btn btn-secondary">Cancel Edit</a>
                <?php endif; ?>
                <a href="budget_vs_actual_report.php" class="btn btn-primary">View Budget Reports</a>
            </div>
        </form>
    </div>

    <!-- Income Budgets -->
    <div class="budgets-table-section">
        <h2>💰 Income Budgets</h2>

        <div class="table-container">
            <table class="budgets-table">
                <thead>
                    <tr>
                        <th>Budget Name</th>
                        <th>Academic Year</th>
                        <th>Term</th>
                        <th>Category</th>
                        <th>Planned Amount</th>
                        <th>Description</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($income_budgets)): ?>
                        <tr>
                            <td colspan="7" class="no-budgets">
                                No income budgets found.
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($income_budgets as $budget): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($budget['name']); ?></td>
                                <td><?php echo htmlspecialchars($budget['academic_year']); ?></td>
                                <td><?php echo htmlspecialchars($budget['term_name'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($budget['category_name']); ?></td>
                                <td class="amount">GHC <?php echo number_format($budget['planned_amount'], 2); ?></td>
                                <td><?php echo htmlspecialchars($budget['description'] ?? ''); ?></td>
                                <td>
                                    <a href="?edit=<?php echo $budget['id']; ?>" class="btn btn-primary">Edit</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Recurring Expense Budgets -->
    <div class="budgets-table-section">
        <h2>🔄 Recurring Expense Budgets</h2>

        <div class="table-container">
            <table class="budgets-table">
                <thead>
                    <tr>
                        <th>Budget Name</th>
                        <th>Academic Year</th>
                        <th>Term</th>
                        <th>Category</th>
                        <th>Frequency</th>
                        <th>Planned Amount</th>
                        <th>Description</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($recurring_budgets)): ?>
                        <tr>
                            <td colspan="8" class="no-budgets">
                                No recurring expense budgets found.
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($recurring_budgets as $budget): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($budget['name']); ?></td>
                                <td><?php echo htmlspecialchars($budget['academic_year']); ?></td>
                                <td><?php echo htmlspecialchars($budget['term_name'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($budget['category_name']); ?></td>
                                <td><?php echo htmlspecialchars(ucfirst($budget['frequency'])); ?></td>
                                <td class="amount">GHC <?php echo number_format($budget['planned_amount'], 2); ?></td>
                                <td><?php echo htmlspecialchars($budget['description'] ?? ''); ?></td>
                                <td>
                                    <a href="?edit=<?php echo $budget['id']; ?>" class="btn btn-primary">Edit</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Non-Recurring Expense Budgets -->
    <div class="budgets-table-section">
        <h2>📋 Non-Recurring Expense Budgets</h2>

        <div class="table-container">
            <table class="budgets-table">
                <thead>
                    <tr>
                        <th>Budget Name</th>
                        <th>Academic Year</th>
                        <th>Term</th>
                        <th>Category</th>
                        <th>Frequency</th>
                        <th>Planned Amount</th>
                        <th>Description</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($non_recurring_budgets)): ?>
                        <tr>
                            <td colspan="8" class="no-budgets">
                                No non-recurring expense budgets found.
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($non_recurring_budgets as $budget): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($budget['name']); ?></td>
                                <td><?php echo htmlspecialchars($budget['academic_year']); ?></td>
                                <td><?php echo htmlspecialchars($budget['term_name'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($budget['category_name']); ?></td>
                                <td><?php echo htmlspecialchars(ucfirst($budget['frequency'])); ?></td>
                                <td class="amount">GHC <?php echo number_format($budget['planned_amount'], 2); ?></td>
                                <td><?php echo htmlspecialchars($budget['description'] ?? ''); ?></td>
                                <td>
                                    <a href="?edit=<?php echo $budget['id']; ?>" class="btn btn-primary">Edit</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Budget Summary and Surplus/Deficit Calculations -->
    <div class="budgets-table-section">
        <h2>📊 Budget Summary & Surplus/Deficit Analysis
            <button onclick="window.print()" class="btn btn-primary" style="float:right; margin-left:10px;">🖨️ Print Detailed Budget</button>
        </h2>

        <div class="summary-cards">
            <div class="summary-card">
                <div class="summary-icon">💰</div>
                <div class="summary-content">
                    <h3>Total Budgeted Income</h3>
                    <div class="summary-amount">GHC <?php echo number_format($total_budgeted_income, 2); ?></div>
                    <p class="summary-note">Sum of all income budget amounts</p>
                </div>
            </div>

            <div class="summary-card">
                <div class="summary-icon">💸</div>
                <div class="summary-content">
                    <h3>Total Budgeted Expenses</h3>
                    <div class="summary-amount">GHC <?php echo number_format($total_budgeted_expenses, 2); ?></div>
                    <p class="summary-note">Sum of all expense budget amounts</p>
                </div>
            </div>

            <div class="summary-card <?php echo $budgeted_surplus_deficit >= 0 ? 'surplus' : 'deficit'; ?>">
                <div class="summary-icon"><?php echo $budgeted_surplus_deficit >= 0 ? '📈' : '📉'; ?></div>
                <div class="summary-content">
                    <h3>Budgeted <?php echo $budgeted_surplus_deficit >= 0 ? 'Surplus' : 'Deficit'; ?></h3>
                    <div class="summary-amount">GHC <?php echo number_format(abs($budgeted_surplus_deficit), 2); ?></div>
                    <p class="summary-note">Income minus budgeted expenses</p>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const budgetTypeSelect = document.getElementById('budget_type');
    const expenseFields = document.querySelector('.expense-fields');

    function toggleExpenseFields() {
        if (budgetTypeSelect.value === 'expense') {
            expenseFields.style.display = 'grid';
        } else {
            expenseFields.style.display = 'none';
        }
    }

    // Initial check
    toggleExpenseFields();

    // Listen for changes
    budgetTypeSelect.addEventListener('change', toggleExpenseFields);
});
</script>

<?php
include '../includes/admin_footer.php';
?>
