<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';

// Check permissions - only admins and accounts users can manage asset categories
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accountant'])) {
    header('Location: ../login.php');
    exit;
}

$message = '';
$error = '';

// Handle success messages from URL parameters
if (isset($_GET['success'])) {
    if ($_GET['success'] === 'added' && isset($_GET['name'])) {
        $message = "Asset category '" . htmlspecialchars($_GET['name']) . "' added successfully!";
    } elseif ($_GET['success'] === 'updated' && isset($_GET['name'])) {
        $message = "Asset category '" . htmlspecialchars($_GET['name']) . "' updated successfully!";
    } elseif ($_GET['success'] === 'deleted' && isset($_GET['name'])) {
        $message = "Asset category '" . htmlspecialchars($_GET['name']) . "' deleted successfully!";
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_category'])) {
        // Add new category
        $name = trim($_POST['name']);
        $description = trim($_POST['description']);
        $is_depreciable = isset($_POST['is_depreciable']) ? 1 : 0;
        $is_appreciable = isset($_POST['is_appreciable']) ? 1 : 0;
        $default_depreciation_method = $_POST['default_depreciation_method'] ?: null;
        $default_depreciation_rate = floatval($_POST['default_depreciation_rate']);
        $default_depreciation_frequency = $_POST['default_depreciation_frequency'] ?: 'monthly';
        $default_appreciation_method = $_POST['default_appreciation_method'] ?: 'none';
        $default_appreciation_rate = floatval($_POST['default_appreciation_rate']);
        $revaluation_allowed = isset($_POST['revaluation_allowed']) ? 1 : 0;

        if (empty($name)) {
            $error = "Category name is required.";
        } elseif ($is_depreciable && empty($default_depreciation_method)) {
            $error = "Depreciation method is required for depreciable assets.";
        } elseif ($is_depreciable && $default_depreciation_rate <= 0) {
            $error = "Depreciation rate must be greater than 0 for depreciable assets.";
        } else {
            try {
                // Check if category name already exists
                $stmt = $pdo->prepare("SELECT id FROM asset_categories WHERE name = ?");
                $stmt->execute([$name]);
                if ($stmt->fetch()) {
                    $error = "A category with this name already exists.";
                } else {
                    $stmt = $pdo->prepare("
                        INSERT INTO asset_categories (
                            name, description, is_depreciable, is_appreciable,
                            default_depreciation_method, default_depreciation_rate,
                            default_depreciation_frequency, default_appreciation_method, default_appreciation_rate,
                            revaluation_allowed, is_system
                        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, FALSE)
                    ");
                    $stmt->execute([
                        $name, $description, $is_depreciable, $is_appreciable,
                        $default_depreciation_method, $default_depreciation_rate,
                        $default_depreciation_frequency, $default_appreciation_method, $default_appreciation_rate,
                        $revaluation_allowed
                    ]);

                    // Log activity
                    AccountantActivityLogger::logActivity(
                        $_SESSION['user_id'],
                        'asset_category_created',
                        "Created new asset category: $name"
                    );

                    // Redirect to prevent form resubmission on refresh
                    header('Location: manage_asset_categories.php?success=added&name=' . urlencode($name));
                    exit;
                }
            } catch (PDOException $e) {
                $error = "Error adding category: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['edit_category'])) {
        // Edit existing category
        $id = $_POST['category_id'];
        $name = trim($_POST['name']);
        $description = trim($_POST['description']);
        $is_depreciable = isset($_POST['is_depreciable']) ? 1 : 0;
        $is_appreciable = isset($_POST['is_appreciable']) ? 1 : 0;
        $default_depreciation_method = $_POST['default_depreciation_method'] ?: null;
        $default_depreciation_rate = floatval($_POST['default_depreciation_rate']);
        $default_depreciation_frequency = $_POST['default_depreciation_frequency'] ?: 'monthly';
        $default_appreciation_method = $_POST['default_appreciation_method'] ?: 'none';
        $default_appreciation_rate = floatval($_POST['default_appreciation_rate']);
        $revaluation_allowed = isset($_POST['revaluation_allowed']) ? 1 : 0;

        if (empty($name)) {
            $error = "Category name is required.";
        } elseif ($is_depreciable && empty($default_depreciation_method)) {
            $error = "Depreciation method is required for depreciable assets.";
        } elseif ($is_depreciable && $default_depreciation_rate <= 0) {
            $error = "Depreciation rate must be greater than 0 for depreciable assets.";
        } else {
            try {
                // Check if another category with this name exists
                $stmt = $pdo->prepare("SELECT id FROM asset_categories WHERE name = ? AND id != ?");
                $stmt->execute([$name, $id]);
                if ($stmt->fetch()) {
                    $error = "Another category with this name already exists.";
                } else {
                    $stmt = $pdo->prepare("
                        UPDATE asset_categories
                        SET name = ?, description = ?, is_depreciable = ?, is_appreciable = ?,
                            default_depreciation_method = ?, default_depreciation_rate = ?,
                            default_depreciation_frequency = ?, default_appreciation_method = ?, default_appreciation_rate = ?,
                            revaluation_allowed = ?
                        WHERE id = ? AND is_system = FALSE
                    ");
                    $stmt->execute([
                        $name, $description, $is_depreciable, $is_appreciable,
                        $default_depreciation_method, $default_depreciation_rate,
                        $default_depreciation_frequency, $default_appreciation_method, $default_appreciation_rate,
                        $revaluation_allowed, $id
                    ]);

                    if ($stmt->rowCount() > 0) {
                        // Log activity
                        AccountantActivityLogger::logActivity(
                            $_SESSION['user_id'],
                            'asset_category_updated',
                            "Updated asset category: $name"
                        );

                        // Redirect to prevent form resubmission on refresh
                        header('Location: manage_asset_categories.php?success=updated&name=' . urlencode($name));
                        exit;
                    } else {
                        $error = "Category not found or cannot be edited.";
                    }
                }
            } catch (PDOException $e) {
                $error = "Error updating category: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['delete_category'])) {
        // Delete category (only non-system categories)
        $id = $_POST['category_id'];

        try {
            // Check if category is used in assets
            $stmt = $pdo->prepare("SELECT COUNT(*) as usage_count FROM assets WHERE category_id = ?");
            $stmt->execute([$id]);
            $usage = $stmt->fetch();

            if ($usage['usage_count'] > 0) {
                $error = "Cannot delete category that is being used by assets.";
            } else {
                $stmt = $pdo->prepare("DELETE FROM asset_categories WHERE id = ? AND is_system = FALSE");
                $stmt->execute([$id]);

                    if ($stmt->rowCount() > 0) {
                        AccountantActivityLogger::logActivity(
                            $_SESSION['user_id'],
                            'asset_category_deleted',
                            "Deleted asset category ID: $id"
                        );

                        // Get category name for redirect
                        $stmt = $pdo->prepare("SELECT name FROM asset_categories WHERE id = ?");
                        $stmt->execute([$id]);
                        $deletedCategory = $stmt->fetch();
                        $name = $deletedCategory ? $deletedCategory['name'] : 'Unknown';

                        // Redirect to prevent form resubmission on refresh
                        header('Location: manage_asset_categories.php?success=deleted&name=' . urlencode($name));
                        exit;
                    } else {
                        $error = "Category not found or cannot be deleted.";
                    }
            }
        } catch (PDOException $e) {
            $error = "Error deleting category: " . $e->getMessage();
        }
    }
}

// Get all categories for display
try {
    $stmt = $pdo->query("
        SELECT ac.*,
               (SELECT COUNT(*) FROM assets a WHERE a.category_id = ac.id) as usage_count
        FROM asset_categories ac
        ORDER BY ac.is_system DESC, ac.name ASC
    ");
    $categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading categories: " . $e->getMessage();
    $categories = [];
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
.asset-categories-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.asset-categories-header {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.asset-categories-content {
    display: grid;
    grid-template-columns: 1fr 2fr;
    gap: 30px;
}

.add-category-section,
.categories-list-section {
    background: white;
    padding: 30px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.section-title {
    color: #2c3e50;
    margin: 0 0 20px 0;
    padding-bottom: 10px;
    border-bottom: 2px solid #e9ecef;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.form-group input,
.form-group select,
.form-group textarea {
    width: 100%;
    padding: 10px 12px;
    border: 1px solid #ced4da;
    border-radius: 5px;
    font-size: 14px;
}

.form-group textarea {
    resize: vertical;
    min-height: 80px;
}

.checkbox-group {
    display: flex;
    align-items: center;
    gap: 10px;
    margin-bottom: 15px;
}

.checkbox-group input[type="checkbox"] {
    width: auto;
    margin: 0;
}

.btn {
    padding: 10px 20px;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    margin-right: 10px;
}

.btn-primary {
    background: #007bff;
    color: white;
}

.btn-success {
    background: #28a745;
    color: white;
}

.btn-danger {
    background: #dc3545;
    color: white;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.message {
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
}

.success-message {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.error-message {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.categories-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
}

.categories-table th,
.categories-table td {
    padding: 12px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.categories-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.category-name {
    font-weight: bold;
    color: #007bff;
}

.config-summary {
    font-size: 0.9em;
    color: #6c757d;
}

.status-badge {
    display: inline-block;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 0.8em;
    font-weight: 600;
}

.system-badge {
    background: #e9ecef;
    color: #495057;
}

.custom-badge {
    background: #d1ecf1;
    color: #0c5460;
}

.dep-yes { color: #28a745; }
.dep-no { color: #dc3545; }
.app-yes { color: #28a745; }
.app-no { color: #dc3545; }

.modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    z-index: 1000;
}

.modal-content {
    background: white;
    margin: 5% auto;
    padding: 30px;
    border-radius: 10px;
    width: 90%;
    max-width: 700px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.3);
    max-height: 90vh;
    overflow-y: auto;
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    padding-bottom: 10px;
    border-bottom: 1px solid #e9ecef;
}

.modal-title {
    margin: 0;
    color: #2c3e50;
}

.close {
    font-size: 24px;
    cursor: pointer;
    color: #6c757d;
}

@media (max-width: 768px) {
    .asset-categories-content {
        grid-template-columns: 1fr;
    }
}
</style>

<div class="asset-categories-container">
    <div class="asset-categories-header">
        <h1>🏷️ Asset Categories Management</h1>
        <p>Configure depreciation and appreciation rules for asset categories</p>
    </div>

    <?php if ($message): ?>
        <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="asset-categories-content">
        <div class="add-category-section">
            <h2 class="section-title">➕ Add New Asset Category</h2>
            <button type="button" class="btn btn-primary" onclick="openAddModal()">Add New Category</button>
        </div>

        <div class="categories-list-section">
            <h2 class="section-title">📋 All Asset Categories</h2>
            <table class="categories-table">
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Depreciable</th>
                        <th>Appreciable</th>
                        <th>Method & Rate</th>
                        <th>Revaluation</th>
                        <th>Status</th>
                        <th>Usage</th>
                        <th class="actions-column">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($categories as $category): ?>
                        <tr>
                            <td>
                                <span class="category-name"><?php echo htmlspecialchars($category['name']); ?></span>
                                <?php if ($category['description']): ?>
                                    <br><small style="color: #6c757d;"><?php echo htmlspecialchars($category['description']); ?></small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="dep-<?php echo $category['is_depreciable'] ? 'yes' : 'no'; ?>">
                                    <?php echo $category['is_depreciable'] ? '✓ Yes' : '✗ No'; ?>
                                </span>
                            </td>
                            <td>
                                <span class="app-<?php echo $category['is_appreciable'] ? 'yes' : 'no'; ?>">
                                    <?php echo $category['is_appreciable'] ? '✓ Yes' : '✗ No'; ?>
                                </span>
                            </td>
                            <td>
                                <?php if ($category['is_depreciable'] && $category['default_depreciation_method']): ?>
                                    <?php
                                    $method_display = str_replace('_', ' ', ucfirst($category['default_depreciation_method']));
                                    echo htmlspecialchars($method_display) . '<br>';
                                    echo '<small>' . number_format($category['default_depreciation_rate'], 2) . '%</small>';
                                    ?>
                                <?php else: ?>
                                    <em>N/A</em>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php echo $category['revaluation_allowed'] ? '✓ Allowed' : '✗ Not Allowed'; ?>
                            </td>
                            <td>
                                <?php if ($category['is_system']): ?>
                                    <span class="status-badge system-badge">System</span>
                                <?php else: ?>
                                    <span class="status-badge custom-badge">Custom</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo $category['usage_count']; ?> assets</td>
                            <td class="actions-column">
                                <?php if (!$category['is_system']): ?>
                                    <button class="btn btn-secondary btn-sm" onclick="editCategory(<?php echo $category['id']; ?>, '<?php echo addslashes($category['name']); ?>', '<?php echo addslashes($category['description'] ?? ''); ?>', <?php echo $category['is_depreciable']; ?>, <?php echo $category['is_appreciable']; ?>, '<?php echo $category['default_depreciation_method'] ?? ''; ?>', <?php echo $category['default_depreciation_rate']; ?>, '<?php echo $category['default_depreciation_frequency'] ?? 'monthly'; ?>', '<?php echo $category['default_appreciation_method'] ?? 'none'; ?>', <?php echo $category['default_appreciation_rate']; ?>, <?php echo $category['revaluation_allowed']; ?>)">Edit</button>
                                    <button class="btn btn-danger btn-sm" onclick="deleteCategory(<?php echo $category['id']; ?>, '<?php echo addslashes($category['name']); ?>')">Delete</button>
                                <?php else: ?>
                                    <em style="color: #6c757d;">System Category</em>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>



<!-- Add Category Modal -->
<div id="addModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Add New Asset Category</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <form method="post">
            <div class="form-group">
                <label for="name">Category Name *</label>
                <input type="text" name="name" id="name" required placeholder="e.g., Office Equipment">
            </div>

            <div class="form-group">
                <label for="description">Description</label>
                <textarea name="description" id="description" placeholder="Optional description..."></textarea>
            </div>

            <div class="checkbox-group">
                <input type="checkbox" name="is_depreciable" id="is_depreciable" checked onchange="toggleDepreciationFields()">
                <label for="is_depreciable">Assets in this category can be depreciated</label>
            </div>

            <div class="checkbox-group">
                <input type="checkbox" name="is_appreciable" id="is_appreciable" onchange="toggleAppreciationFields()">
                <label for="is_appreciable">Assets in this category can appreciate/revalue</label>
            </div>

            <div id="depreciation-fields" class="form-group">
                <label for="default_depreciation_method">Default Depreciation Method</label>
                <select name="default_depreciation_method" id="default_depreciation_method">
                    <option value="">Select Method</option>
                    <option value="straight_line">Straight Line</option>
                    <option value="declining_balance">Declining Balance</option>
                    <option value="units_of_production">Units of Production</option>
                </select>
            </div>

            <div id="depreciation-rate-fields" class="form-group">
                <label for="default_depreciation_rate">Default Depreciation Rate (%)</label>
                <input type="number" name="default_depreciation_rate" id="default_depreciation_rate" step="0.01" min="0" max="100" placeholder="10.00">
            </div>

            <div class="form-group">
                <label for="default_depreciation_frequency">Default Depreciation Frequency</label>
                <select name="default_depreciation_frequency" id="default_depreciation_frequency">
                    <option value="monthly">Monthly</option>
                    <option value="quarterly">Quarterly</option>
                    <option value="annually">Annually</option>
                </select>
            </div>

            <div class="form-group">
                <label for="default_appreciation_method">Default Appreciation Method</label>
                <select name="default_appreciation_method" id="default_appreciation_method">
                    <option value="none">None</option>
                    <option value="fixed_percentage">Fixed Percentage</option>
                    <option value="compound">Compound</option>
                    <option value="market_based">Market Based</option>
                </select>
            </div>

            <div class="form-group">
                <label for="default_appreciation_rate">Default Appreciation Rate (% per year)</label>
                <input type="number" name="default_appreciation_rate" id="default_appreciation_rate" step="0.01" min="0" max="100" placeholder="5.00">
            </div>

            <div class="checkbox-group">
                <input type="checkbox" name="revaluation_allowed" id="revaluation_allowed">
                <label for="revaluation_allowed">Revaluation allowed for this category</label>
            </div>

            <button type="submit" name="add_category" class="btn btn-primary">Add Category</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<!-- Edit Category Modal -->
<div id="editModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Edit Asset Category</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <form method="post">
            <input type="hidden" name="category_id" id="edit_category_id">
            <div class="form-group">
                <label for="edit_name">Category Name *</label>
                <input type="text" name="name" id="edit_name" required>
            </div>
            <div class="form-group">
                <label for="edit_description">Description</label>
                <textarea name="description" id="edit_description"></textarea>
            </div>

            <div class="checkbox-group">
                <input type="checkbox" name="is_depreciable" id="edit_is_depreciable" onchange="toggleEditDepreciationFields()">
                <label for="edit_is_depreciable">Assets in this category can be depreciated</label>
            </div>

            <div class="checkbox-group">
                <input type="checkbox" name="is_appreciable" id="edit_is_appreciable">
                <label for="edit_is_appreciable">Assets in this category can appreciate/revalue</label>
            </div>

            <div id="edit_depreciation_fields" class="form-group">
                <label for="edit_default_depreciation_method">Default Depreciation Method</label>
                <select name="default_depreciation_method" id="edit_default_depreciation_method">
                    <option value="">Select Method</option>
                    <option value="straight_line">Straight Line</option>
                    <option value="declining_balance">Declining Balance</option>
                    <option value="units_of_production">Units of Production</option>
                </select>
            </div>

            <div id="edit_depreciation_rate_fields" class="form-group">
                <label for="edit_default_depreciation_rate">Default Depreciation Rate (%)</label>
                <input type="number" name="default_depreciation_rate" id="edit_default_depreciation_rate" step="0.01" min="0" max="100">
            </div>

            <div class="form-group">
                <label for="edit_default_depreciation_frequency">Default Depreciation Frequency</label>
                <select name="default_depreciation_frequency" id="edit_default_depreciation_frequency">
                    <option value="monthly">Monthly</option>
                    <option value="quarterly">Quarterly</option>
                    <option value="annually">Annually</option>
                </select>
            </div>

            <div class="form-group">
                <label for="edit_default_appreciation_method">Default Appreciation Method</label>
                <select name="default_appreciation_method" id="edit_default_appreciation_method">
                    <option value="none">None</option>
                    <option value="fixed_percentage">Fixed Percentage</option>
                    <option value="compound">Compound</option>
                    <option value="market_based">Market Based</option>
                </select>
            </div>

            <div class="form-group">
                <label for="edit_default_appreciation_rate">Default Appreciation Rate (% per year)</label>
                <input type="number" name="default_appreciation_rate" id="edit_default_appreciation_rate" step="0.01" min="0" max="100">
            </div>

            <div class="checkbox-group">
                <input type="checkbox" name="revaluation_allowed" id="edit_revaluation_allowed">
                <label for="edit_revaluation_allowed">Revaluation allowed for this category</label>
            </div>

            <button type="submit" name="edit_category" class="btn btn-success">Update Category</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<!-- Delete Category Modal -->
<div id="deleteModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Delete Asset Category</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <p>Are you sure you want to delete the asset category "<span id="delete_category_name"></span>"?</p>
        <p style="color: #dc3545; font-weight: 600;">This action cannot be undone and will permanently remove the category.</p>
        <form method="post">
            <input type="hidden" name="category_id" id="delete_category_id">
            <button type="submit" name="delete_category" class="btn btn-danger">Yes, Delete Category</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<script>
function openAddModal() {
    document.getElementById('addModal').style.display = 'block';
}

function editCategory(id, name, description, isDepreciable, isAppreciable, depMethod, depRate, depFreq, appMethod, appRate, revalAllowed) {
    document.getElementById('edit_category_id').value = id;
    document.getElementById('edit_name').value = name;
    document.getElementById('edit_description').value = description;
    document.getElementById('edit_is_depreciable').checked = isDepreciable == 1;
    document.getElementById('edit_is_appreciable').checked = isAppreciable == 1;
    document.getElementById('edit_default_depreciation_method').value = depMethod || '';
    document.getElementById('edit_default_depreciation_rate').value = depRate;
    document.getElementById('edit_default_depreciation_frequency').value = depFreq || 'monthly';
    document.getElementById('edit_default_appreciation_method').value = appMethod || 'none';
    document.getElementById('edit_default_appreciation_rate').value = appRate;
    document.getElementById('edit_revaluation_allowed').checked = revalAllowed == 1;

    toggleEditDepreciationFields();
    document.getElementById('editModal').style.display = 'block';
}

function deleteCategory(id, name) {
    document.getElementById('delete_category_id').value = id;
    document.getElementById('delete_category_name').textContent = name;
    document.getElementById('deleteModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('addModal').style.display = 'none';
    document.getElementById('editModal').style.display = 'none';
    document.getElementById('deleteModal').style.display = 'none';
}

function toggleDepreciationFields() {
    const isDepreciable = document.getElementById('is_depreciable').checked;
    document.getElementById('depreciation-fields').style.display = isDepreciable ? 'block' : 'none';
    document.getElementById('depreciation-rate-fields').style.display = isDepreciable ? 'block' : 'none';
}

function toggleEditDepreciationFields() {
    const isDepreciable = document.getElementById('edit_is_depreciable').checked;
    document.getElementById('edit_depreciation_fields').style.display = isDepreciable ? 'block' : 'none';
    document.getElementById('edit_depreciation_rate_fields').style.display = isDepreciable ? 'block' : 'none';
}

function toggleAppreciationFields() {
    // Could add logic here if appreciation fields need to be toggled
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    toggleDepreciationFields();
});

// Close modal when clicking outside
window.onclick = function(event) {
    if (event.target.className === 'modal') {
        event.target.style.display = 'none';
    }
}
</script>

<?php include '../includes/admin_footer.php'; ?>
