<?php
require_once '../config.php';

$message = '';
$message_type = '';

// Handle messages from redirects
if (isset($_GET['message']) && isset($_GET['type'])) {
    $message = $_GET['message'];
    $message_type = $_GET['type'];
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['add_allowance'])) {
        $name = trim($_POST['name']);
        $description = trim($_POST['description']);
        $is_taxable = isset($_POST['is_taxable']) ? 1 : 0;

        $stmt = $pdo->prepare("
            INSERT INTO allowance_types (name, description, is_taxable, is_active)
            VALUES (?, ?, ?, TRUE)
        ");

        if ($stmt->execute([$name, $description, $is_taxable])) {
            // Log the allowance type creation activity
            UserActivityLogger::logActivity($_SESSION['user_id'], 'create_allowance_type', "Created allowance type: {$name}");

            header("Location: manage_allowances.php?message=" . urlencode('Allowance type added successfully.') . "&type=success");
            exit();
        } else {
            $message = 'Failed to add allowance type.';
            $message_type = 'error';
        }
    } elseif (isset($_POST['edit_allowance'])) {
        $allowance_id = $_POST['allowance_id'];
        $name = trim($_POST['name']);
        $description = trim($_POST['description']);
        $is_taxable = isset($_POST['is_taxable']) ? 1 : 0;

        $stmt = $pdo->prepare("
            UPDATE allowance_types
            SET name = ?, description = ?, is_taxable = ?
            WHERE id = ?
        ");

        if ($stmt->execute([$name, $description, $is_taxable, $allowance_id])) {
            // Log the allowance type update activity
            UserActivityLogger::logActivity($_SESSION['user_id'], 'update_allowance_type', "Updated allowance type: {$name}");

            header("Location: manage_allowances.php?message=" . urlencode('Allowance type updated successfully.') . "&type=success");
            exit();
        } else {
            $message = 'Failed to update allowance type.';
            $message_type = 'error';
        }
    } elseif (isset($_POST['delete_allowance'])) {
        $allowance_id = $_POST['allowance_id'];

        // Check if allowance is used in any salary structures
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM staff_allowances WHERE allowance_type_id = ?");
        $stmt->execute([$allowance_id]);
        $usage_count = $stmt->fetchColumn();

        if ($usage_count > 0) {
            $message = 'Cannot delete allowance type as it is currently used in salary structures.';
            $message_type = 'error';
        } else {
            // Get the allowance name for logging
            $name_stmt = $pdo->prepare("SELECT name FROM allowance_types WHERE id = ?");
            $name_stmt->execute([$allowance_id]);
            $name = $name_stmt->fetchColumn();

            $stmt = $pdo->prepare("UPDATE allowance_types SET is_active = FALSE WHERE id = ?");
            if ($stmt->execute([$allowance_id])) {
                // Log the allowance type deletion activity
                UserActivityLogger::logActivity($_SESSION['user_id'], 'delete_allowance_type', "Deleted allowance type: {$name}");

                header("Location: manage_allowances.php?message=" . urlencode('Allowance type deleted successfully.') . "&type=success");
                exit();
            } else {
                $message = 'Failed to delete allowance type.';
                $message_type = 'error';
            }
        }
    }
}

// Fetch active allowance types
$stmt = $pdo->query("SELECT * FROM allowance_types WHERE is_active = TRUE ORDER BY name");
$allowances = $stmt->fetchAll();

include 'includes/accounting_header.php';
?>

<style>
.allowance-table {
    background: white;
    border-radius: 8px;
    overflow: hidden;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    margin-bottom: 20px;
}

.allowance-table table {
    width: 100%;
    border-collapse: collapse;
}

.allowance-table th,
.allowance-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.allowance-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.allowance-table tr:hover {
    background: #f8f9fa;
}

.form-row {
    display: flex;
    gap: 20px;
    margin-bottom: 15px;
}

.form-row .form-group {
    flex: 1;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #333;
}

.form-group input,
.form-group select,
.form-group textarea {
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
}

.taxable-badge {
    background: #28a745;
    color: white;
    padding: 2px 6px;
    border-radius: 3px;
    font-size: 0.8em;
    font-weight: bold;
}

.no-allowances {
    text-align: center;
    padding: 40px;
    color: #7f8c8d;
    background: #f8f9fa;
    border-radius: 8px;
    margin: 20px 0;
}

/* Button styling to match accounting sidebar theme */
.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    border-color: #2E7D32;
    color: white;
    padding: 10px 20px;
    border-radius: 4px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #4CAF50 0%, #66BB6A 100%);
    border-color: #4CAF50;
    color: white;
    transform: translateY(-1px);
    box-shadow: 0 2px 8px rgba(46, 125, 50, 0.3);
}

.btn {
    display: inline-block;
    padding: 8px 16px;
    border-radius: 4px;
    text-decoration: none;
    font-weight: 500;
    border: 1px solid transparent;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-small {
    padding: 6px 12px;
    font-size: 0.9em;
}

.btn:hover {
    transform: translateY(-1px);
}

.btn-danger {
    background-color: #dc3545;
    color: white;
    border-color: #dc3545;
}

.btn-danger:hover {
    background-color: #c82333;
    border-color: #bd2130;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
}

.modal-content {
    background-color: white;
    margin: 10% auto;
    padding: 20px;
    border-radius: 8px;
    width: 90%;
    max-width: 500px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.2);
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}

.modal-header h3 {
    margin: 0;
    color: #333;
}

.close {
    color: #aaa;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
}

.close:hover {
    color: #000;
}

.modal-body {
    margin-bottom: 20px;
}

.modal-footer {
    text-align: right;
}
</style>

<h2>💰 Manage Allowance Types</h2>

<?php if ($message): ?>
    <div class="message <?php echo $message_type; ?>"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>

<!-- Add Button -->
<button type="button" class="btn btn-primary" onclick="openAddModal()" style="margin-bottom: 20px;">Add New Allowance Type</button>

<!-- Active Allowance Types -->
<h3>Active Allowance Types</h3>

<?php if (empty($allowances)): ?>
    <div class="no-allowances">
        <h4>No Active Allowance Types</h4>
        <p>No allowance types have been set up yet. Use the form above to add the first one.</p>
    </div>
<?php else: ?>
    <table class="allowance-table">
        <thead>
            <tr>
                <th>Name</th>
                <th>Description</th>
                <th>Taxable</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($allowances as $allowance): ?>
                <tr>
                    <td><?php echo htmlspecialchars($allowance['name']); ?></td>
                    <td><?php echo htmlspecialchars($allowance['description']); ?></td>
                    <td><?php echo $allowance['is_taxable'] ? '<span class="taxable-badge">Yes</span>' : 'No'; ?></td>
                    <td>
                        <button type="button" class="btn btn-primary btn-small" onclick="openEditModal(<?php echo $allowance['id']; ?>, '<?php echo addslashes($allowance['name']); ?>', '<?php echo addslashes($allowance['description']); ?>', <?php echo $allowance['is_taxable']; ?>)">Edit</button>
                        <button type="button" class="btn btn-danger btn-small" onclick="openDeleteModal(<?php echo $allowance['id']; ?>, '<?php echo addslashes($allowance['name']); ?>')">Delete</button>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
<?php endif; ?>

<!-- Add Allowance Modal -->
<div id="addModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Add New Allowance Type</h3>
            <span class="close" onclick="closeModal('addModal')">&times;</span>
        </div>
        <div class="modal-body">
            <form method="post" action="manage_allowances.php" id="addForm">
                <div class="form-row">
                    <div class="form-group">
                        <label for="add_modal_name">Allowance Name:</label>
                        <input type="text" id="add_modal_name" name="name" required>
                    </div>
                    <div class="form-group">
                        <label for="add_modal_is_taxable">Taxable Allowance:</label>
                        <input type="checkbox" id="add_modal_is_taxable" name="is_taxable" value="1" checked>
                        <small class="form-text text-muted">Check if this allowance is subject to income tax</small>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group" style="flex: 2;">
                        <label for="add_modal_description">Description:</label>
                        <textarea id="add_modal_description" name="description" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="submit" name="add_allowance" class="btn btn-primary">Add Allowance Type</button>
                    <button type="button" class="btn btn-secondary" onclick="closeModal('addModal')">Cancel</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Allowance Modal -->
<div id="editModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Edit Allowance Type</h3>
            <span class="close" onclick="closeModal('editModal')">&times;</span>
        </div>
        <div class="modal-body">
            <form method="post" action="manage_allowances.php" id="editForm">
                <input type="hidden" name="allowance_id" id="edit_allowance_id">
                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_modal_name">Allowance Name:</label>
                        <input type="text" id="edit_modal_name" name="name" required>
                    </div>
                    <div class="form-group">
                        <label for="edit_modal_is_taxable">Taxable Allowance:</label>
                        <input type="checkbox" id="edit_modal_is_taxable" name="is_taxable" value="1">
                        <small class="form-text text-muted">Check if this allowance is subject to income tax</small>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group" style="flex: 2;">
                        <label for="edit_modal_description">Description:</label>
                        <textarea id="edit_modal_description" name="description" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="submit" name="edit_allowance" class="btn btn-primary">Update Allowance Type</button>
                    <button type="button" class="btn btn-secondary" onclick="closeModal('editModal')">Cancel</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Allowance Modal -->
<div id="deleteModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Delete Allowance Type</h3>
            <span class="close" onclick="closeModal('deleteModal')">&times;</span>
        </div>
        <div class="modal-body">
            <p>Are you sure you want to delete the allowance type "<span id="delete_allowance_name"></span>"?</p>
            <p class="text-danger">This action cannot be undone. The allowance type will be marked as inactive.</p>
            <form method="post" action="manage_allowances.php" id="deleteForm">
                <input type="hidden" name="allowance_id" id="delete_allowance_id">
                <div class="modal-footer">
                    <button type="submit" name="delete_allowance" class="btn btn-danger">Delete Allowance Type</button>
                    <button type="button" class="btn btn-secondary" onclick="closeModal('deleteModal')">Cancel</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function openAddModal() {
    // Clear form fields
    document.getElementById('add_modal_name').value = '';
    document.getElementById('add_modal_description').value = '';
    document.getElementById('add_modal_is_taxable').checked = true;
    document.getElementById('addModal').style.display = 'block';
}

function openEditModal(id, name, description, isTaxable) {
    document.getElementById('edit_allowance_id').value = id;
    document.getElementById('edit_modal_name').value = name;
    document.getElementById('edit_modal_description').value = description;
    document.getElementById('edit_modal_is_taxable').checked = isTaxable == 1;
    document.getElementById('editModal').style.display = 'block';
}

function openDeleteModal(id, name) {
    document.getElementById('delete_allowance_id').value = id;
    document.getElementById('delete_allowance_name').textContent = name;
    document.getElementById('deleteModal').style.display = 'block';
}

function closeModal(modalId) {
    document.getElementById(modalId).style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    if (event.target.className === 'modal') {
        event.target.style.display = 'none';
    }
}
</script>

<?php
include '../includes/admin_footer.php';
?>
