<?php
require_once 'email_notification.php';
require_once 'sms_notification.php';

class NotificationManager {
    private $pdo;
    private $email_service;
    private $sms_service;

    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->email_service = new EmailNotification();
        $this->sms_service = new SMSNotification();
    }

    /**
     * Send notification for loan addition
     */
    public function notifyLoanAdded($loan_data) {
        $settings = $this->getNotificationSettings('loan_added');

        if (!$settings['enabled']) {
            return;
        }

        $subject = "New Loan Added: {$loan_data['name']}";
        $message = $this->buildLoanMessage('added', $loan_data);

        $this->sendNotifications($settings, $subject, $message);
    }

    /**
     * Send notification for payment recorded
     */
    public function notifyPaymentRecorded($loan_data, $payment_data) {
        $settings = $this->getNotificationSettings('payment_recorded');

        if (!$settings['enabled']) {
            return;
        }

        $subject = "Loan Payment Recorded: {$loan_data['name']}";
        $message = $this->buildPaymentMessage($loan_data, $payment_data);

        $this->sendNotifications($settings, $subject, $message);
    }

    /**
     * Send due date reminder
     */
    public function sendDueDateReminder($loan_data, $days_until_due) {
        $settings = $this->getNotificationSettings('due_date_reminder');

        if (!$settings['enabled']) {
            return;
        }

        $subject = "Loan Due Date Reminder: {$loan_data['name']}";
        $message = $this->buildReminderMessage($loan_data, $days_until_due);

        $this->sendNotifications($settings, $subject, $message);
    }

    /**
     * Send notification for fee payment due
     */
    public function notifyFeeDue($student_data, $fee_data) {
        $settings = $this->getNotificationSettings('fee_due');

        if (!$settings['enabled']) {
            return;
        }

        $subject = "Fee Payment Due: {$student_data['first_name']} {$student_data['last_name']}";
        $message = $this->buildFeeDueMessage($student_data, $fee_data);

        $this->sendToParent($student_data, $settings, $subject, $message);
    }

    /**
     * Send notification for overdue fee
     */
    public function notifyOverdueFee($student_data, $fee_data, $days_overdue) {
        $settings = $this->getNotificationSettings('overdue_fee');

        if (!$settings['enabled']) {
            return;
        }

        $subject = "Overdue Fee Alert: {$student_data['first_name']} {$student_data['last_name']}";
        $message = $this->buildOverdueFeeMessage($student_data, $fee_data, $days_overdue);

        $this->sendToParent($student_data, $settings, $subject, $message);
    }

    /**
     * Send general announcement
     */
    public function sendAnnouncement($announcement_data, $recipients = []) {
        $settings = $this->getNotificationSettings('announcement');

        if (!$settings['enabled']) {
            return;
        }

        $subject = $announcement_data['title'];
        $message = $this->buildAnnouncementMessage($announcement_data);

        if (empty($recipients)) {
            // Send to all parents/students based on target audience
            $this->sendToAllParents($settings, $subject, $message, $announcement_data['target_audience']);
        } else {
            // Send to specific recipients
            foreach ($recipients as $recipient) {
                $this->sendToParent($recipient, $settings, $subject, $message);
            }
        }
    }

    /**
     * Get notification settings for a specific event
     */
    private function getNotificationSettings($event_type) {
        $stmt = $this->pdo->prepare("
            SELECT * FROM notification_settings
            WHERE event_type = ? AND is_active = 1
            LIMIT 1
        ");
        $stmt->execute([$event_type]);
        $settings = $stmt->fetch();

        if (!$settings) {
            // Return default settings
            return [
                'enabled' => false,
                'email_enabled' => false,
                'sms_enabled' => false,
                'email_recipients' => '',
                'sms_recipients' => ''
            ];
        }

        return $settings;
    }

    /**
     * Send notifications via enabled channels
     */
    private function sendNotifications($settings, $subject, $message) {
        // Send email notifications
        if ($settings['email_enabled'] && !empty($settings['email_recipients'])) {
            $emails = explode(',', $settings['email_recipients']);
            foreach ($emails as $email) {
                $email = trim($email);
                if (filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    $this->email_service->send($email, $subject, $message, true);
                }
            }
        }

        // Send SMS notifications
        if ($settings['sms_enabled'] && !empty($settings['sms_recipients'])) {
            $phones = explode(',', $settings['sms_recipients']);
            foreach ($phones as $phone) {
                $phone = trim($phone);
                if (!empty($phone)) {
                    $this->sms_service->send($phone, $message);
                }
            }
        }
    }

    /**
     * Send notification to a specific parent
     */
    private function sendToParent($student_data, $settings, $subject, $message) {
        // Send email if enabled and email available
        if ($settings['email_enabled'] && !empty($student_data['parent_email'])) {
            $this->email_service->send($student_data['parent_email'], $subject, $message, true);
        }

        // Send SMS if enabled and phone available
        if ($settings['sms_enabled'] && !empty($student_data['parent_phone'])) {
            $this->sms_service->send($student_data['parent_phone'], strip_tags($message));
        }
    }

    /**
     * Send to all parents based on target audience
     */
    private function sendToAllParents($settings, $subject, $message, $target_audience = 'all') {
        $query = "SELECT id, first_name, last_name, parent_email, parent_phone FROM students WHERE active = 1";

        if ($target_audience !== 'all') {
            // Add filtering based on target audience (e.g., class, grade level)
            // For now, assume 'all' sends to everyone
        }

        $stmt = $this->pdo->prepare($query);
        $stmt->execute();
        $students = $stmt->fetchAll();

        foreach ($students as $student) {
            $this->sendToParent($student, $settings, $subject, $message);
        }
    }

    /**
     * Build loan addition message
     */
    private function buildLoanMessage($action, $loan_data) {
        $html = "
        <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;'>
            <h2 style='color: #2c3e50;'>Loan {$action}: {$loan_data['name']}</h2>
            <div style='background: #f8f9fa; padding: 20px; border-radius: 5px; margin: 20px 0;'>
                <p><strong>Name:</strong> {$loan_data['name']}</p>
                <p><strong>Type:</strong> " . ucfirst(str_replace('_', ' ', $loan_data['liability_type'])) . "</p>
                <p><strong>Original Amount:</strong> GHC " . number_format($loan_data['original_amount'], 2) . "</p>
                <p><strong>Current Balance:</strong> GHC " . number_format($loan_data['current_balance'], 2) . "</p>
                <p><strong>Interest Rate:</strong> {$loan_data['interest_rate']}%</p>
                <p><strong>Monthly Payment:</strong> GHC " . number_format($loan_data['monthly_payment'], 2) . "</p>
                <p><strong>Start Date:</strong> {$loan_data['start_date']}</p>
                <p><strong>End Date:</strong> {$loan_data['end_date']}</p>
                <p><strong>Description:</strong> {$loan_data['description']}</p>
            </div>
            <p style='color: #7f8c8d; font-size: 12px;'>This is an automated notification from the School Management System.</p>
        </div>
        ";

        return $html;
    }

    /**
     * Build payment recorded message
     */
    private function buildPaymentMessage($loan_data, $payment_data) {
        $html = "
        <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;'>
            <h2 style='color: #28a745;'>Payment Recorded: {$loan_data['name']}</h2>
            <div style='background: #f8f9fa; padding: 20px; border-radius: 5px; margin: 20px 0;'>
                <p><strong>Loan:</strong> {$loan_data['name']}</p>
                <p><strong>Payment Amount:</strong> GHC " . number_format($payment_data['amount'], 2) . "</p>
                <p><strong>Payment Date:</strong> {$payment_data['date']}</p>
                <p><strong>New Balance:</strong> GHC " . number_format($loan_data['current_balance'], 2) . "</p>
            </div>
            <p style='color: #7f8c8d; font-size: 12px;'>This is an automated notification from the School Management System.</p>
        </div>
        ";

        return $html;
    }

    /**
     * Build due date reminder message
     */
    private function buildReminderMessage($loan_data, $days_until_due) {
        $urgency_color = $days_until_due <= 7 ? '#dc3545' : '#ffc107';

        $html = "
        <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;'>
            <h2 style='color: {$urgency_color};'>Loan Due Date Reminder</h2>
            <div style='background: #f8f9fa; padding: 20px; border-radius: 5px; margin: 20px 0;'>
                <p><strong>Loan:</strong> {$loan_data['name']}</p>
                <p><strong>Current Balance:</strong> GHC " . number_format($loan_data['current_balance'], 2) . "</p>
                <p><strong>Monthly Payment:</strong> GHC " . number_format($loan_data['monthly_payment'], 2) . "</p>
                <p><strong>Due Date:</strong> {$loan_data['end_date']}</p>
                <p><strong>Days Until Due:</strong> {$days_until_due} days</p>
            </div>
            <p style='color: #7f8c8d; font-size: 12px;'>This is an automated reminder from the School Management System.</p>
        </div>
        ";

        return $html;
    }

    /**
     * Build fee due message
     */
    private function buildFeeDueMessage($student_data, $fee_data) {
        $html = "
        <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;'>
            <h2 style='color: #ffc107;'>Fee Payment Due</h2>
            <div style='background: #f8f9fa; padding: 20px; border-radius: 5px; margin: 20px 0;'>
                <p><strong>Student:</strong> {$student_data['first_name']} {$student_data['last_name']}</p>
                <p><strong>Amount Due:</strong> GHC " . number_format($fee_data['amount'], 2) . "</p>
                <p><strong>Due Date:</strong> {$fee_data['due_date']}</p>
                <p><strong>Description:</strong> {$fee_data['description']}</p>
            </div>
            <p>Please ensure payment is made before the due date to avoid late fees.</p>
            <p style='color: #7f8c8d; font-size: 12px;'>This is an automated notification from the School Management System.</p>
        </div>
        ";

        return $html;
    }

    /**
     * Build overdue fee message
     */
    private function buildOverdueFeeMessage($student_data, $fee_data, $days_overdue) {
        $html = "
        <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;'>
            <h2 style='color: #dc3545;'>Overdue Fee Alert</h2>
            <div style='background: #f8f9fa; padding: 20px; border-radius: 5px; margin: 20px 0;'>
                <p><strong>Student:</strong> {$student_data['first_name']} {$student_data['last_name']}</p>
                <p><strong>Overdue Amount:</strong> GHC " . number_format($fee_data['amount'], 2) . "</p>
                <p><strong>Days Overdue:</strong> {$days_overdue} days</p>
                <p><strong>Original Due Date:</strong> {$fee_data['due_date']}</p>
            </div>
            <p style='color: #dc3545; font-weight: bold;'>Please make payment immediately to avoid further penalties.</p>
            <p style='color: #7f8c8d; font-size: 12px;'>This is an automated notification from the School Management System.</p>
        </div>
        ";

        return $html;
    }

    /**
     * Build announcement message
     */
    private function buildAnnouncementMessage($announcement_data) {
        $html = "
        <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;'>
            <h2 style='color: #007bff;'>School Announcement</h2>
            <div style='background: #f8f9fa; padding: 20px; border-radius: 5px; margin: 20px 0;'>
                <h3>{$announcement_data['title']}</h3>
                <p>{$announcement_data['message']}</p>
                <p><strong>Date:</strong> " . date('Y-m-d H:i:s') . "</p>
            </div>
            <p style='color: #7f8c8d; font-size: 12px;'>This is an official announcement from the School Management System.</p>
        </div>
        ";

        return $html;
    }
}
?>
