<?php
require_once '../config.php';
include 'includes/accounting_header.php';

$classes = $pdo->query("SELECT DISTINCT class FROM students ORDER BY class ASC")->fetchAll(PDO::FETCH_COLUMN);
$fees = $pdo->query("SELECT id, name FROM fees ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
$years = $pdo->query("SELECT id, year FROM academic_years ORDER BY year DESC")->fetchAll(PDO::FETCH_ASSOC);

// Get filter values from GET parameters
$class_filter = $_GET['class_filter'] ?? '';
$fee_filter = $_GET['fee_filter'] ?? '';
$year_filter = $_GET['year_filter'] ?? '';
$search_filter = $_GET['search'] ?? '';

// Build query with filters
$query = "
SELECT
    s.id,
    CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as name,
    s.class AS class_name,
    COALESCE((SELECT SUM(sf2.amount) FROM student_fees sf2 WHERE sf2.student_id = s.id), 0) AS total_fees,
    COALESCE((SELECT SUM(sf2.paid_amount) FROM student_fees sf2 WHERE sf2.student_id = s.id), 0) AS total_paid,
    GREATEST(0, COALESCE((SELECT SUM(sf2.amount) - SUM(COALESCE(sf2.paid_amount, 0)) FROM student_fees sf2 WHERE sf2.student_id = s.id), 0)) AS total_outstanding,
    GREATEST(0, COALESCE((SELECT SUM(sf2.amount) - SUM(COALESCE(sf2.paid_amount, 0)) FROM student_fees sf2 WHERE sf2.student_id = s.id), 0)) AS total_outstanding_all
FROM students s
";

$where = [];
$params = [];

if (!empty($class_filter)) {
    $where[] = "s.class = ?";
    $params[] = $class_filter;
}

if (!empty($fee_filter)) {
    $where[] = "EXISTS (SELECT 1 FROM student_fees sf3 WHERE sf3.student_id = s.id AND sf3.fee_id = ?)";
    $params[] = $fee_filter;
}

if (!empty($year_filter)) {
    $where[] = "EXISTS (SELECT 1 FROM student_fees sf3 JOIN fees f3 ON sf3.fee_id = f3.id WHERE sf3.student_id = s.id AND f3.academic_year_id = ?)";
    $params[] = $year_filter;
}

if (!empty($search_filter)) {
    $where[] = "CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) LIKE ?";
    $params[] = '%' . $search_filter . '%';
}

if ($where) {
    $query .= " WHERE " . implode(' AND ', $where);
}

$query .= " ORDER BY name ASC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$students = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch school settings for print
$school_settings = $pdo->query("SELECT school_name, address, phone, email FROM school_settings LIMIT 1")->fetch(PDO::FETCH_ASSOC);
?>

<style>
/* Basic styling for table and modal */
.table-container {
    margin: 20px;
    overflow-x: auto;
}

table {
    width: 100%;
    border-collapse: collapse;
    margin-bottom: 20px;
}

th, td {
    padding: 12px 15px;
    border: 1px solid #ddd;
    text-align: left;
}

#students-table th {
    background: white !important;
    color: black !important;
}

.status-paid {
    color: green;
    font-weight: bold;
}

.status-outstanding {
    color: red;
    font-weight: bold;
}

button.view-btn {
    background-color: #4CAF50;
    color: white;
    border: none;
    padding: 8px 12px;
    cursor: pointer;
    border-radius: 4px;
}

button.view-btn:hover {
    background-color: #45a049;
}

/* Filter form styling */
.filter-form {
    background-color: #f9f9f9;
    padding: 20px;
    border-radius: 8px;
    border: 1px solid #ddd;
    margin-bottom: 20px;
    display: flex;
    gap: 15px;
    align-items: center;
    flex-wrap: wrap;
}

.filter-form label {
    font-weight: bold;
    color: #333;
    margin-right: 5px;
}

.filter-form select,
.filter-form input[type="text"] {
    padding: 8px 12px;
    border: 1px solid #ccc;
    border-radius: 4px;
    font-size: 14px;
    min-width: 120px;
}

.filter-form select:focus,
.filter-form input[type="text"]:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 5px rgba(76, 175, 80, 0.3);
}

.filter-form button {
    background-color: #4CAF50;
    color: white;
    border: none;
    padding: 10px 15px;
    cursor: pointer;
    border-radius: 4px;
    font-size: 14px;
    font-weight: bold;
}

.filter-form button:hover {
    background-color: #45a049;
}

.filter-form a {
    text-decoration: none;
    background-color: #f44336;
    color: white;
    padding: 10px 15px;
    border-radius: 4px;
    font-size: 14px;
    font-weight: bold;
}

.filter-form a:hover {
    background-color: #d32f2f;
}

/* Modal styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    padding-top: 60px;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    overflow: auto;
    background-color: rgba(0,0,0,0.4);
}

.modal-content {
    background-color: #fefefe;
    margin: auto;
    padding: 20px;
    border: 1px solid #888;
    width: 80%;
    max-width: 800px;
    border-radius: 8px;
}

.close {
    color: #aaa;
    float: right;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
}

.close:hover,
.close:focus {
    color: black;
    text-decoration: none;
}
</style>

<div class="table-container">
    <h2>Student Fees Summary</h2>

    <!-- Filter Form -->
    <form method="GET" class="filter-form">
        <label>Class:
            <select name="class_filter">
                <option value="">All Classes</option>
                <?php foreach ($classes as $class): ?>
                    <option value="<?php echo htmlspecialchars($class); ?>" <?php echo ($class_filter == $class) ? 'selected' : ''; ?>><?php echo htmlspecialchars($class); ?></option>
                <?php endforeach; ?>
            </select>
        </label>

<label>Fee:
    <select name="fee_filter">
        <option value="">All Fees</option>
        <?php foreach ($fees as $fee): ?>
            <option value="<?php echo htmlspecialchars($fee['id']); ?>" <?php echo ($fee_filter == $fee['id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars($fee['name']); ?></option>
        <?php endforeach; ?>
    </select>
</label>

<label>Year:
    <select name="year_filter">
        <option value="">All Years</option>
        <?php foreach ($years as $year): ?>
            <option value="<?php echo htmlspecialchars($year['id']); ?>" <?php echo ($year_filter == $year['id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars($year['year']); ?></option>
        <?php endforeach; ?>
    </select>
</label>

        <label>Search Student:
            <input type="text" name="search" value="<?php echo htmlspecialchars($search_filter); ?>" placeholder="Student name">
        </label>

        <button type="submit" style="background-color: #4CAF50; color: white; border: none; padding: 8px 12px; cursor: pointer; border-radius: 4px;">Filter</button>
        <a href="fees_summary.php" style="text-decoration: none; background-color: #f44336; color: white; padding: 8px 12px; border-radius: 4px;">Clear</a>
    </form>
    <table id="students-table">
        <thead>
            <tr>
                <th style="background: white; color: black;">Student Name</th>
                <th style="background: white; color: black;">Class</th>
                <th style="background: white; color: black;">Total Fees Amount (GHC)</th>
                <th style="background: white; color: black;">Paid (GHC)</th>
                <th style="background: white; color: black;">Outstanding (GHC)</th>
                <th style="background: white; color: black;">Status</th>
                <th style="background: white; color: black;">View</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($students as $student):
                $status = ($student['total_outstanding_all'] <= 0) ? 'Not Owing' : 'Owing';
                $status_class = ($status === 'Not Owing') ? 'status-paid' : 'status-outstanding';
            ?>
            <tr data-student-id="<?php echo htmlspecialchars($student['id']); ?>">
                <td><?php echo htmlspecialchars($student['name']); ?></td>
                <td><?php echo htmlspecialchars($student['class_name']); ?></td>
                <td><?php echo number_format($student['total_fees'], 2); ?></td>
                <td><?php echo number_format($student['total_paid'], 2); ?></td>
                <td><?php echo number_format($student['total_outstanding'], 2); ?></td>
                <td class="<?php echo $status_class; ?>"><?php echo $status; ?></td>
                <td><button class="view-btn" data-student-id="<?php echo htmlspecialchars($student['id']); ?>" data-student-name="<?php echo htmlspecialchars($student['name']); ?>">View</button></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>

<!-- Modal -->
<div id="feeDetailsModal" class="modal">
    <div class="modal-content">
        <span class="close" onclick="closeModal()">&times;</span>
        <h3 id="modalStudentName">Student Fees Details</h3>
        <table id="feeDetailsTable" style="width:100%; border-collapse: collapse;">
            <thead>
                <tr>
                    <th style="background: white; color: black;">Fee Name</th>
                    <th style="background: white; color: black;">Amount (GHC)</th>
                    <th style="background: white; color: black;">Paid Amount (GHC)</th>
                    <th style="background: white; color: black;">Outstanding (GHC)</th>
                    <th style="background: white; color: black;">Status</th>
                </tr>
            </thead>
            <tbody id="feeDetailsBody">
                <!-- Filled dynamically -->
            </tbody>
        </table>
        <div style="margin-top: 20px; padding: 10px; border-top: 1px solid #ddd;">
            <p><strong>Total Fees:</strong> GHC <span id="totalFees">0.00</span></p>
            <p><strong>Total Paid:</strong> GHC <span id="totalPaid">0.00</span></p>
            <p><strong>Outstanding:</strong> GHC <span id="totalOutstanding">0.00</span></p>
            <button onclick="printModal()" style="margin-top: 10px; background-color: #4CAF50; color: white; border: none; padding: 8px 12px; cursor: pointer; border-radius: 4px;">Print</button>
        </div>
    </div>
</div>

<script>
var schoolName = "<?php echo addslashes($school_settings['school_name'] ?? 'School Name'); ?>";
var schoolAddress = "<?php echo addslashes($school_settings['address'] ?? 'School Address'); ?>";
var schoolContact = "<?php echo addslashes(($school_settings['phone'] ?? '') . ' | ' . ($school_settings['email'] ?? '')); ?>";

function openModal(studentId, studentName) {
    document.getElementById('modalStudentName').textContent = 'Fees Details for ' + studentName;
    var modal = document.getElementById('feeDetailsModal');
    var tbody = document.getElementById('feeDetailsBody');
    tbody.innerHTML = '<tr><td colspan="5">Loading...</td></tr>';
    modal.style.display = 'block';

var feeFilter = document.querySelector('select[name="fee_filter"]').value;
var yearFilter = document.querySelector('select[name="year_filter"]').value;

// Build query string
let queryString = 'student_id=' + studentId;
if (feeFilter) queryString += '&fee=' + encodeURIComponent(feeFilter);
if (yearFilter) queryString += '&year=' + encodeURIComponent(yearFilter);

    // Fetch fee details via AJAX
    fetch('fetch_student_fees.php?' + queryString)
        .then(response => response.json())
        .then(data => {
            tbody.innerHTML = '';
            if (data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="5">No fee records found.</td></tr>';
                return;
            }
            let totalFees = 0;
            let totalPaid = 0;
            let totalOutstanding = 0;

            data.forEach(fee => {
                var statusClass = fee.outstanding <= 0 ? 'status-paid' : 'status-outstanding';
                var statusText = fee.outstanding <= 0 ? 'Not Owing' : 'Owing';
                var row = '<tr>' +
                    '<td>' + fee.fee_name + '</td>' +
                    '<td>' + parseFloat(fee.amount).toFixed(2) + '</td>' +
                    '<td>' + parseFloat(fee.paid_amount).toFixed(2) + '</td>' +
                    '<td>' + parseFloat(fee.outstanding).toFixed(2) + '</td>' +
                    '<td class="' + statusClass + '">' + statusText + '</td>' +
                    '</tr>';
                tbody.insertAdjacentHTML('beforeend', row);

                totalFees += parseFloat(fee.amount);
                totalPaid += parseFloat(fee.paid_amount);
                totalOutstanding += parseFloat(fee.outstanding);
            });

            document.getElementById('totalFees').textContent = totalFees.toFixed(2);
            document.getElementById('totalPaid').textContent = totalPaid.toFixed(2);
            document.getElementById('totalOutstanding').textContent = totalOutstanding.toFixed(2);
        })
        .catch(error => {
            tbody.innerHTML = '<tr><td colspan="5">Error loading fee details.</td></tr>';
            console.error('Error fetching fee details:', error);
        });
}

function closeModal() {
    document.getElementById('feeDetailsModal').style.display = 'none';
}

function printModal() {
    // Add school name and other important info to print preview
    var modalContent = document.querySelector('.modal-content');
    var printContent = modalContent.cloneNode(true);
    var printButton = printContent.querySelector('button[onclick="printModal()"]');
    if (printButton) printButton.remove();

    var printWindow = window.open('', '', 'height=600,width=800');
    printWindow.document.write('<html><head><title>Print Fees Details</title>');
    printWindow.document.write('<style>body{font-family: Arial, sans-serif; margin: 20px;} h1, h2, h3 {text-align: center;} table {width: 100%; border-collapse: collapse; margin-top: 20px;} th, td {border: 1px solid #ddd; padding: 8px; text-align: left;} th {background-color: #f2f2f2;}</style>');
    printWindow.document.write('</head><body>');
    printWindow.document.write('<h1>' + schoolName + '</h1>');
    printWindow.document.write('<p style="text-align:center;">' + schoolAddress + '<br>' + schoolContact + '</p>');
    printWindow.document.write('<h2>Student Fees Details</h2>');
    printWindow.document.write(printContent.innerHTML);
    printWindow.document.write('</body></html>');
    printWindow.document.close();
    printWindow.focus();
    printWindow.print();
    printWindow.close();
}

// Add event listener for view buttons
document.addEventListener('DOMContentLoaded', function() {
    document.querySelectorAll('.view-btn').forEach(function(button) {
        button.addEventListener('click', function() {
            var studentId = this.getAttribute('data-student-id');
            var studentName = this.getAttribute('data-student-name');
            openModal(studentId, studentName);
        });
    });
});

// Close modal when clicking outside content
window.onclick = function(event) {
    var modal = document.getElementById('feeDetailsModal');
    if (event.target === modal) {
        modal.style.display = 'none';
    }
};
</script>

<?php
include 'includes/accounting_footer.php';
?>
