<?php
require_once '../config.php';

// Get filter parameters
$account_id = $_GET['account_id'] ?? '';
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-t');
$transaction_type = $_GET['transaction_type'] ?? 'all';

// Build cashbook query
try {
    $where_conditions = [];
    $params = [];

    // Date range
    $where_conditions[] = "DATE(ct.created_at) BETWEEN ? AND ?";
    $params[] = $start_date;
    $params[] = $end_date;

    // Account filter
    if (!empty($account_id)) {
        $where_conditions[] = "ct.account_id = ?";
        $params[] = $account_id;
    }

    // Transaction type filter
    if ($transaction_type !== 'all') {
        $where_conditions[] = "ct.transaction_type = ?";
        $params[] = $transaction_type;
    }

    $where_clause = implode(' AND ', $where_conditions);

    // Get cashbook entries
    $query = "
        SELECT
            ct.id,
            ct.created_at as transaction_date,
            ca.account_name,
            ca.account_type,
            ct.transaction_type,
            ct.amount,
            ct.description,
            ct.created_by
        FROM cash_account_transactions ct
        JOIN cash_accounts ca ON ct.account_id = ca.id
        WHERE {$where_clause}
        ORDER BY ct.created_at ASC, ct.id ASC
    ";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $cashbook_entries = $stmt->fetchAll();

    // Calculate running balances
    $running_balance = 0;
    $processed_entries = [];

    foreach ($cashbook_entries as $entry) {
        if ($entry['transaction_type'] === 'increase') {
            $running_balance += $entry['amount'];
            $debit = $entry['amount'];
            $credit = 0;
        } else {
            $running_balance -= $entry['amount'];
            $debit = 0;
            $credit = $entry['amount'];
        }

        $processed_entries[] = [
            'date' => $entry['transaction_date'],
            'account' => $entry['account_name'],
            'description' => $entry['description'],
            'debit' => $debit,
            'credit' => $credit,
            'balance' => $running_balance,
            'type' => $entry['transaction_type']
        ];
    }

    // Get opening balance
    if (!empty($account_id)) {
        $balance_query = "
            SELECT COALESCE(SUM(CASE WHEN transaction_type = 'increase' THEN amount ELSE -amount END), 0) as opening_balance
            FROM cash_account_transactions
            WHERE account_id = ? AND DATE(created_at) < ?
        ";
        $stmt = $pdo->prepare($balance_query);
        $stmt->execute([$account_id, $start_date]);
        $opening_balance = $stmt->fetch()['opening_balance'];
    } else {
        $balance_query = "
            SELECT COALESCE(SUM(CASE WHEN transaction_type = 'increase' THEN amount ELSE -amount END), 0) as opening_balance
            FROM cash_account_transactions
            WHERE DATE(created_at) < ?
        ";
        $stmt = $pdo->prepare($balance_query);
        $stmt->execute([$start_date]);
        $opening_balance = $stmt->fetch()['opening_balance'];
    }

} catch (PDOException $e) {
    die("Error loading cashbook data: " . $e->getMessage());
}

// Set headers for CSV download
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="cashbook_' . date('Y-m-d') . '.csv"');

// Create output stream
$output = fopen('php://output', 'w');

// Write CSV headers
fputcsv($output, ['Date', 'Account', 'Description', 'Cash In (GHC)', 'Cash Out (GHC)', 'Balance (GHC)']);

// Write opening balance
fputcsv($output, [
    date('M d, Y', strtotime($start_date)),
    '',
    'Opening Balance',
    '',
    '',
    number_format($opening_balance, 2)
]);

// Write transactions
foreach ($processed_entries as $entry) {
    fputcsv($output, [
        date('M d, Y H:i', strtotime($entry['date'])),
        $entry['account'],
        $entry['description'],
        $entry['debit'] > 0 ? number_format($entry['debit'], 2) : '',
        $entry['credit'] > 0 ? number_format($entry['credit'], 2) : '',
        number_format($entry['balance'], 2)
    ]);
}

// Write closing balance
$closing_balance = $opening_balance + array_sum(array_column($processed_entries, 'debit')) - array_sum(array_column($processed_entries, 'credit'));
fputcsv($output, [
    date('M d, Y', strtotime($end_date)),
    '',
    'Closing Balance',
    '',
    '',
    number_format($closing_balance, 2)
]);

fclose($output);
exit();
?>
