<?php
require_once '../config.php';
require_once '../includes/access_control.php';

// Get parameters from URL
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-t');
$report_type = $_GET['report_type'] ?? 'monthly';

// Calculate date range based on report type
if ($report_type === 'yearly') {
    $start_date = date('Y-01-01');
    $end_date = date('Y-12-31');
} elseif ($report_type === 'quarterly') {
    $current_quarter = ceil(date('n') / 3);
    $start_date = date('Y-m-01', mktime(0, 0, 0, ($current_quarter - 1) * 3 + 1, 1));
    $end_date = date('Y-m-t', mktime(0, 0, 0, $current_quarter * 3, 1));
}

// Set headers for file download
$filename = "cash_flow_statement_" . date('Y-m-d', strtotime($start_date)) . "_to_" . date('Y-m-d', strtotime($end_date)) . ".csv";
header('Content-Type: text/csv');
header('Content-Disposition: attachment; filename="' . $filename . '"');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Open output stream
$output = fopen('php://output', 'w');

    // Get cash flow data using actual cash transactions
    try {
        // Operating Activities - Cash received from customers and cash paid for expenses
        // Cash received from customers (payments on receivables)
        $stmt = $pdo->prepare("
            SELECT
                'Cash received from customers' as description,
                COALESCE(SUM(jel.debit_amount), 0) as amount
            FROM journal_entries je
            JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
            JOIN chart_of_accounts coa ON jel.account_id = coa.id
            WHERE je.transaction_date BETWEEN ? AND ?
            AND coa.account_code LIKE '111%'  -- Cash accounts
            AND jel.debit_amount > 0
            AND je.description LIKE '%payment received%'
        ");
        $stmt->execute([$start_date, $end_date]);
        $cash_from_customers = $stmt->fetch();

        // Cash paid for expenses
        $stmt = $pdo->prepare("
            SELECT
                'Cash paid for expenses' as description,
                COALESCE(SUM(jel.credit_amount), 0) as amount
            FROM journal_entries je
            JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
            JOIN chart_of_accounts coa ON jel.account_id = coa.id
            WHERE je.transaction_date BETWEEN ? AND ?
            AND coa.account_code LIKE '111%'  -- Cash accounts
            AND jel.credit_amount > 0
            AND je.description LIKE '%expense%'
        ");
        $stmt->execute([$start_date, $end_date]);
        $cash_for_expenses = $stmt->fetch();

        $operating_activities = [];
        if ($cash_from_customers['amount'] > 0) {
            $operating_activities[] = [
                'section' => 'operating',
                'description' => $cash_from_customers['description'],
                'category_type' => 'operating',
                'is_income' => true,
                'amount' => $cash_from_customers['amount']
            ];
        }
        if ($cash_for_expenses['amount'] > 0) {
            $operating_activities[] = [
                'section' => 'operating',
                'description' => $cash_for_expenses['description'],
                'category_type' => 'operating',
                'is_income' => false,
                'amount' => -$cash_for_expenses['amount']
            ];
        }

        // Investing Activities - Cash outflows for long-term asset purchases only
        // Exclude receivables and cash accounts - only include actual investing activities
        $stmt = $pdo->prepare("
            SELECT
                'investing' as section,
                coa.account_name as description,
                'investing' as category_type,
                FALSE as is_income,
                -COALESCE(SUM(jel.debit_amount), 0) as amount
            FROM journal_entries je
            JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
            JOIN chart_of_accounts coa ON jel.account_id = coa.id
            WHERE je.transaction_date BETWEEN ? AND ?
            AND coa.account_type = 'asset'
            AND coa.account_code NOT LIKE '111%'  -- Exclude cash accounts
            AND coa.account_code NOT LIKE '112%'  -- Exclude receivables
            AND jel.debit_amount > 0
            GROUP BY coa.id, coa.account_name
            HAVING amount != 0
            ORDER BY ABS(amount) DESC
        ");
        $stmt->execute([$start_date, $end_date]);
        $investing_activities = $stmt->fetchAll();

        // Financing Activities - Cash flows from loans and equity
        $stmt = $pdo->prepare("
            SELECT
                'financing' as section,
                coa.account_name as description,
                'financing' as category_type,
                CASE WHEN jel.credit_amount > 0 THEN TRUE ELSE FALSE END as is_income,
                CASE
                    WHEN jel.credit_amount > 0 THEN COALESCE(SUM(jel.credit_amount), 0)
                    ELSE -COALESCE(SUM(jel.debit_amount), 0)
                END as amount
            FROM journal_entries je
            JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
            JOIN chart_of_accounts coa ON jel.account_id = coa.id
            WHERE je.transaction_date BETWEEN ? AND ?
            AND coa.account_type IN ('liability', 'equity')
            AND (jel.debit_amount > 0 OR jel.credit_amount > 0)
            GROUP BY coa.id, coa.account_name
            HAVING amount != 0
            ORDER BY is_income DESC, ABS(amount) DESC
        ");
        $stmt->execute([$start_date, $end_date]);
        $financing_activities = $stmt->fetchAll();

        // Calculate totals
        $operating_total = array_sum(array_column($operating_activities, 'amount'));
        $investing_total = array_sum(array_column($investing_activities, 'amount'));
        $financing_total = array_sum(array_column($financing_activities, 'amount'));
        $net_cash_flow = $operating_total + $investing_total + $financing_total;

        // Get beginning cash balance from cash_accounts table (opening balance represents balance at start of period)
        $stmt = $pdo->prepare("
            SELECT COALESCE(SUM(opening_balance), 0) as beginning_cash
            FROM cash_accounts
        ");
        $stmt->execute();
        $beginning_cash_result = $stmt->fetch();
        $beginning_cash = $beginning_cash_result['beginning_cash'];

        $ending_cash = $beginning_cash + $net_cash_flow;

    // Write CSV headers
    fputcsv($output, ['Cash Flow Statement']);
    fputcsv($output, ['Period:', date('M d, Y', strtotime($start_date)) . ' to ' . date('M d, Y', strtotime($end_date))]);
    fputcsv($output, ['Report Type:', ucfirst($report_type)]);
    fputcsv($output, []);
    fputcsv($output, ['']);

    // Summary
    fputcsv($output, ['CASH FLOW SUMMARY']);
    fputcsv($output, ['Description', 'Amount (GHC)']);

    fputcsv($output, [
        'Beginning Cash Balance',
        number_format($beginning_cash, 2)
    ]);

    fputcsv($output, [
        'Net Operating Cash Flow',
        number_format($operating_total, 2)
    ]);

    fputcsv($output, [
        'Net Investing Cash Flow',
        number_format($investing_total, 2)
    ]);

    fputcsv($output, [
        'Net Financing Cash Flow',
        number_format($financing_total, 2)
    ]);

    fputcsv($output, [
        'Net Cash Flow',
        number_format($net_cash_flow, 2)
    ]);

    fputcsv($output, [
        'Ending Cash Balance',
        number_format($ending_cash, 2)
    ]);

    fputcsv($output, ['']);

    // Operating Activities
    fputcsv($output, ['OPERATING ACTIVITIES']);
    fputcsv($output, ['Description', 'Amount (GHC)']);

    foreach ($operating_activities as $activity) {
        fputcsv($output, [
            $activity['description'],
            number_format($activity['amount'], 2)
        ]);
    }

    fputcsv($output, [
        'NET OPERATING CASH FLOW',
        number_format($operating_total, 2)
    ]);

    fputcsv($output, ['']);

    // Investing Activities
    fputcsv($output, ['INVESTING ACTIVITIES']);
    fputcsv($output, ['Description', 'Amount (GHC)']);

    foreach ($investing_activities as $activity) {
        fputcsv($output, [
            $activity['description'],
            number_format($activity['amount'], 2)
        ]);
    }

    fputcsv($output, [
        'NET INVESTING CASH FLOW',
        number_format($investing_total, 2)
    ]);

    fputcsv($output, ['']);

    // Financing Activities
    fputcsv($output, ['FINANCING ACTIVITIES']);
    fputcsv($output, ['Description', 'Amount (GHC)']);

    foreach ($financing_activities as $activity) {
        fputcsv($output, [
            $activity['description'],
            number_format($activity['amount'], 2)
        ]);
    }

    fputcsv($output, [
        'NET FINANCING CASH FLOW',
        number_format($financing_total, 2)
    ]);

    fputcsv($output, ['']);

    // Net Cash Flow Summary
    fputcsv($output, ['NET CASH FLOW SUMMARY']);
    fputcsv($output, ['Description', 'Amount (GHC)']);

    fputcsv($output, [
        'Net Increase/(Decrease) in Cash',
        number_format($net_cash_flow, 2)
    ]);

    fputcsv($output, [
        'Cash at Beginning of Period',
        number_format($beginning_cash, 2)
    ]);

    fputcsv($output, [
        'Cash at End of Period',
        number_format($ending_cash, 2)
    ]);

    fputcsv($output, ['']);

    // Analysis
    fputcsv($output, ['CASH FLOW ANALYSIS']);
    fputcsv($output, ['Metric', 'Value', 'Analysis']);

    $operating_health = $operating_total >= 0 ? 'Positive' : 'Negative';
    $operating_analysis = $operating_total >= 0
        ? 'Healthy core operations generating cash'
        : 'Core operations consuming cash - requires attention';

    fputcsv($output, [
        'Operating Cash Flow',
        'GHC ' . number_format($operating_total, 2),
        $operating_analysis
    ]);

    $investing_analysis = $investing_total < 0
        ? 'Investing in assets for growth'
        : 'Selling assets or reducing investments';

    fputcsv($output, [
        'Investing Cash Flow',
        'GHC ' . number_format($investing_total, 2),
        $investing_analysis
    ]);

    $financing_analysis = $financing_total < 0
        ? 'Paying down debt or returning capital'
        : 'Raising additional financing';

    fputcsv($output, [
        'Financing Cash Flow',
        'GHC ' . number_format($financing_total, 2),
        $financing_analysis
    ]);

    $overall_health = $net_cash_flow >= 0 ? 'Improving' : 'Declining';
    $overall_analysis = $net_cash_flow >= 0
        ? 'Cash position is improving'
        : 'Cash position is declining - monitor closely';

    fputcsv($output, [
        'Overall Cash Position',
        $overall_health,
        $overall_analysis
    ]);

    fclose($output);

} catch (PDOException $e) {
    // Handle error
    fputcsv($output, ['Error: ' . $e->getMessage()]);
    fclose($output);
    exit();
}
?>
