<?php
require_once '../config.php';

$message = '';
$error = '';

// Get filter parameters
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-t');
$report_type = $_GET['report_type'] ?? 'monthly'; // monthly, quarterly, yearly

// Calculate date range based on report type
if ($report_type === 'yearly') {
    $start_date = date('Y-01-01');
    $end_date = date('Y-12-31');
} elseif ($report_type === 'quarterly') {
    $current_quarter = ceil(date('n') / 3);
    $start_date = date('Y-m-01', mktime(0, 0, 0, ($current_quarter - 1) * 3 + 1, 1));
    $end_date = date('Y-m-t', mktime(0, 0, 0, $current_quarter * 3, 1));
}

// Get cash flow data using double-entry system
try {
    // Get all cash transactions (debits and credits to cash accounts)
    $stmt = $pdo->prepare("
        SELECT
            je.id,
            je.transaction_date,
            je.description,
            je.source_type,
            jel.account_id,
            jel.debit_amount,
            jel.credit_amount,
            coa.account_name,
            coa.account_code,
            coa.account_type
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE je.transaction_date BETWEEN ? AND ?
        AND (coa.account_code LIKE '10%' OR (coa.account_code LIKE '11%' AND coa.account_code < '1121') OR coa.account_code = '221')
        AND (jel.debit_amount > 0 OR jel.credit_amount > 0)
        AND je.status = 'posted'
        ORDER BY je.transaction_date, je.id
    ");
    $stmt->execute([$start_date, $end_date]);
    $cash_transactions = $stmt->fetchAll();

    // Categorize transactions into operating, investing, and financing activities
    $operating_activities = [];
    $investing_activities = [];
    $financing_activities = [];

    foreach ($cash_transactions as $transaction) {
        $amount = $transaction['debit_amount'] > 0 ? $transaction['debit_amount'] : -$transaction['credit_amount'];
        $description = $transaction['description'];

        // Determine activity type based on source_type and account types involved
        $activity_type = 'operating'; // default

        if ($transaction['source_type'] === 'income') {
            $activity_type = 'operating';
        } elseif ($transaction['source_type'] === 'expense') {
            $activity_type = 'operating';
        } elseif ($transaction['source_type'] === 'asset') {
            $activity_type = 'investing';
        } elseif ($transaction['source_type'] === 'liability' || $transaction['source_type'] === 'equity') {
            $activity_type = 'financing';
        } elseif ($transaction['source_type'] === 'fee_payment' || $transaction['source_type'] === '') {
            $activity_type = 'operating';
        }

        // For investing activities, check if it's related to long-term assets
        if ($activity_type === 'operating' && strpos($description, 'asset') !== false) {
            $activity_type = 'investing';
        }

        // For financing activities, check for loan or equity related terms
        if ($activity_type === 'operating' && (strpos(strtolower($description), 'loan') !== false || strpos(strtolower($description), 'equity') !== false)) {
            $activity_type = 'financing';
        }

        $activity = [
            'section' => $activity_type,
            'description' => $description,
            'category_type' => $activity_type,
            'is_income' => $amount > 0,
            'amount' => $amount
        ];

        if ($activity_type === 'operating') {
            $operating_activities[] = $activity;
        } elseif ($activity_type === 'investing') {
            $investing_activities[] = $activity;
        } elseif ($activity_type === 'financing') {
            $financing_activities[] = $activity;
        }
    }

    // Calculate totals
    $operating_total = array_sum(array_column($operating_activities, 'amount'));
    $investing_total = array_sum(array_column($investing_activities, 'amount'));
    $financing_total = array_sum(array_column($financing_activities, 'amount'));
    $net_cash_flow = $operating_total + $investing_total + $financing_total;

    // Get beginning cash balance from actual cash account balances before the period
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(jel.debit_amount - jel.credit_amount), 0) as beginning_cash
        FROM journal_entry_lines jel
        JOIN journal_entries je ON jel.journal_entry_id = je.id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE je.transaction_date < ?
        AND (coa.account_code LIKE '10%' OR (coa.account_code LIKE '11%' AND coa.account_code < '1121') OR coa.account_code = '221')
        AND je.status = 'posted'
    ");
    $stmt->execute([$start_date]);
    $beginning_cash_result = $stmt->fetch();
    $beginning_cash = $beginning_cash_result['beginning_cash'];
    
    $ending_cash = $beginning_cash + $net_cash_flow;

    // Cash flow trends (last 6 months) using actual cash transactions
    $cash_flow_trends = [];
    for ($i = 5; $i >= 0; $i--) {
        $month_start = date('Y-m-01', strtotime("-{$i} months"));
        $month_end = date('Y-m-t', strtotime("-{$i} months"));
        $month_name = date('M Y', strtotime("-{$i} months"));

        // Get actual cash inflows (debits to cash accounts)
        $stmt = $pdo->prepare("
            SELECT COALESCE(SUM(jel.debit_amount), 0) as inflows
            FROM journal_entries je
            JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
            JOIN chart_of_accounts coa ON jel.account_id = coa.id
            WHERE je.transaction_date BETWEEN ? AND ?
            AND (coa.account_code LIKE '10%' OR (coa.account_code LIKE '11%' AND coa.account_code < '1121') OR coa.account_code = '221')
            AND jel.debit_amount > 0
            AND je.status = 'posted'
        ");
        $stmt->execute([$month_start, $month_end]);
        $inflows = $stmt->fetch()['inflows'];

        // Get actual cash outflows (credits to cash accounts)
        $stmt = $pdo->prepare("
            SELECT COALESCE(SUM(jel.credit_amount), 0) as outflows
            FROM journal_entries je
            JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
            JOIN chart_of_accounts coa ON jel.account_id = coa.id
            WHERE je.transaction_date BETWEEN ? AND ?
            AND (coa.account_code LIKE '10%' OR (coa.account_code LIKE '11%' AND coa.account_code < '1121') OR coa.account_code = '221')
            AND jel.credit_amount > 0
            AND je.status = 'posted'
        ");
        $stmt->execute([$month_start, $month_end]);
        $outflows = $stmt->fetch()['outflows'];

        $net_cash_flow = $inflows - $outflows;

        $cash_flow_trends[] = [
            'month' => $month_name,
            'inflows' => $inflows,
            'outflows' => $outflows,
            'net' => $net_cash_flow
        ];
    }

} catch (PDOException $e) {
    $error = "Error fetching cash flow data: " . $e->getMessage();
}

include 'includes/accounting_header.php';
?>

<div class="cash-flow-container">
    <div class="report-header">
        <h1>💰 Cash Flow Statement</h1>
        <div class="report-filters">
            <form method="get" class="filter-form">
                <div class="filter-group">
                    <label for="report_type">Report Type:</label>
                    <select name="report_type" id="report_type" onchange="this.form.submit()">
                        <option value="monthly" <?php echo $report_type === 'monthly' ? 'selected' : ''; ?>>Monthly</option>
                        <option value="quarterly" <?php echo $report_type === 'quarterly' ? 'selected' : ''; ?>>Quarterly</option>
                        <option value="yearly" <?php echo $report_type === 'yearly' ? 'selected' : ''; ?>>Yearly</option>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="start_date">Start Date:</label>
                    <input type="date" name="start_date" id="start_date" value="<?php echo $start_date; ?>" onchange="this.form.submit()">
                </div>

                <div class="filter-group">
                    <label for="end_date">End Date:</label>
                    <input type="date" name="end_date" id="end_date" value="<?php echo $end_date; ?>" onchange="this.form.submit()">
                </div>

                <a href="export_cash_flow.php?<?php echo http_build_query($_GET); ?>" class="btn btn-secondary">Export</a>
            </form>
        </div>
    </div>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Cash Flow Summary -->
    <div class="summary-cards">
        <div class="card beginning-cash-card">
            <div class="card-icon">💵</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($beginning_cash, 2); ?></h3>
                <p>Beginning Cash Balance</p>
                <small>As of <?php echo date('M d, Y', strtotime($start_date)); ?></small>
            </div>
        </div>

        <div class="card operating-cash-card">
            <div class="card-icon">🔄</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($operating_total, 2); ?></h3>
                <p>Net Operating Cash</p>
                <small>From operations</small>
            </div>
        </div>

        <div class="card investing-cash-card">
            <div class="card-icon">📈</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($investing_total, 2); ?></h3>
                <p>Net Investing Cash</p>
                <small>From investments</small>
            </div>
        </div>

        <div class="card financing-cash-card">
            <div class="card-icon">🏦</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($financing_total, 2); ?></h3>
                <p>Net Financing Cash</p>
                <small>From financing</small>
            </div>
        </div>

        <div class="card ending-cash-card">
            <div class="card-icon">💰</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($ending_cash, 2); ?></h3>
                <p>Ending Cash Balance</p>
                <small>As of <?php echo date('M d, Y', strtotime($end_date)); ?></small>
            </div>
        </div>

        <div class="card net-cash-card <?php echo $net_cash_flow >= 0 ? 'positive' : 'negative'; ?>">
            <div class="card-icon"><?php echo $net_cash_flow >= 0 ? '📈' : '📉'; ?></div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($net_cash_flow, 2); ?></h3>
                <p>Net Cash Flow</p>
                <small><?php echo $net_cash_flow >= 0 ? 'Increase' : 'Decrease'; ?></small>
            </div>
        </div>
    </div>

    <!-- Cash Flow Statement -->
    <div class="report-section">
        <h2>Statement of Cash Flows</h2>
        <p class="report-period">For the period: <?php echo date('M d, Y', strtotime($start_date)); ?> to <?php echo date('M d, Y', strtotime($end_date)); ?></p>

        <!-- Operating Activities -->
        <div class="cash-flow-section">
            <h3>Operating Activities</h3>
            <div class="table-container">
                <table class="cash-flow-table">
                    <thead>
                        <tr>
                            <th>Description</th>
                            <th>Amount (GHC)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($operating_activities as $activity): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($activity['description']); ?></td>
                                <td class="<?php echo $activity['amount'] >= 0 ? 'positive' : 'negative'; ?>">
                                    <?php echo number_format($activity['amount'], 2); ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        <tr class="total-row">
                            <td><strong>Net Operating Cash Flow</strong></td>
                            <td class="<?php echo $operating_total >= 0 ? 'positive' : 'negative'; ?>">
                                <strong><?php echo number_format($operating_total, 2); ?></strong>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Investing Activities -->
        <div class="cash-flow-section">
            <h3>Investing Activities</h3>
            <div class="table-container">
                <table class="cash-flow-table">
                    <thead>
                        <tr>
                            <th>Description</th>
                            <th>Amount (GHC)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($investing_activities as $activity): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($activity['description']); ?></td>
                                <td class="<?php echo $activity['amount'] >= 0 ? 'positive' : 'negative'; ?>">
                                    <?php echo number_format($activity['amount'], 2); ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        <tr class="total-row">
                            <td><strong>Net Investing Cash Flow</strong></td>
                            <td class="<?php echo $investing_total >= 0 ? 'positive' : 'negative'; ?>">
                                <strong><?php echo number_format($investing_total, 2); ?></strong>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Financing Activities -->
        <div class="cash-flow-section">
            <h3>Financing Activities</h3>
            <div class="table-container">
                <table class="cash-flow-table">
                    <thead>
                        <tr>
                            <th>Description</th>
                            <th>Amount (GHC)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($financing_activities as $activity): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($activity['description']); ?></td>
                                <td class="<?php echo $activity['amount'] >= 0 ? 'positive' : 'negative'; ?>">
                                    <?php echo number_format($activity['amount'], 2); ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        <tr class="total-row">
                            <td><strong>Net Financing Cash Flow</strong></td>
                            <td class="<?php echo $financing_total >= 0 ? 'positive' : 'negative'; ?>">
                                <strong><?php echo number_format($financing_total, 2); ?></strong>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Net Cash Flow -->
        <div class="net-cash-flow">
            <div class="table-container">
                <table class="cash-flow-table">
                    <tr class="total-row final-total">
                        <td><strong>Net Increase/(Decrease) in Cash</strong></td>
                        <td class="<?php echo $net_cash_flow >= 0 ? 'positive' : 'negative'; ?>">
                            <strong><?php echo number_format($net_cash_flow, 2); ?></strong>
                        </td>
                    </tr>
                    <tr class="total-row">
                        <td><strong>Cash at Beginning of Period</strong></td>
                        <td><strong><?php echo number_format($beginning_cash, 2); ?></strong></td>
                    </tr>
                    <tr class="total-row final-total">
                        <td><strong>Cash at End of Period</strong></td>
                        <td class="<?php echo $ending_cash >= 0 ? 'positive' : 'negative'; ?>">
                            <strong><?php echo number_format($ending_cash, 2); ?></strong>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
    </div>

    <!-- Cash Flow Trends Chart -->
    <div class="report-section">
        <h2>Cash Flow Trends (Last 6 Months)</h2>
        <div class="chart-container">
            <canvas id="cashFlowChart" width="400" height="200"></canvas>
        </div>
    </div>

    <!-- Cash Flow Analysis -->
    <div class="report-section">
        <h2>💡 Cash Flow Analysis</h2>
        <div class="analysis-grid">
            <div class="analysis-card">
                <h4>Operating Cash Flow</h4>
                <p class="<?php echo $operating_total >= 0 ? 'positive' : 'negative'; ?>">
                    GHC <?php echo number_format($operating_total, 2); ?>
                </p>
                <small><?php echo $operating_total >= 0 ? 'Positive operating cash flow indicates healthy core operations' : 'Negative operating cash flow requires attention'; ?></small>
            </div>

            <div class="analysis-card">
                <h4>Investing Activities</h4>
                <p class="negative">
                    GHC <?php echo number_format(abs($investing_total), 2); ?>
                </p>
                <small>Investments in assets are necessary for growth but reduce current cash</small>
            </div>

            <div class="analysis-card">
                <h4>Financing Activities</h4>
                <p class="<?php echo $financing_total >= 0 ? 'positive' : 'negative'; ?>">
                    GHC <?php echo number_format(abs($financing_total), 2); ?>
                </p>
                <small><?php echo $financing_total >= 0 ? 'Borrowings and other financing inflows' : 'Loan repayments and other financing costs'; ?></small>
            </div>

            <div class="analysis-card">
                <h4>Overall Health</h4>
                <p class="<?php echo $net_cash_flow >= 0 ? 'positive' : 'negative'; ?>">
                    <?php echo $net_cash_flow >= 0 ? 'Improving' : 'Declining'; ?>
                </p>
                <small><?php echo $net_cash_flow >= 0 ? 'Cash position is improving' : 'Cash position is declining - monitor closely'; ?></small>
            </div>
        </div>
    </div>
</div>

<style>
.cash-flow-container {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.report-header {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.report-header h1 {
    color: #2c3e50;
    margin: 0 0 20px 0;
}

.filter-form {
    display: flex;
    gap: 20px;
    align-items: end;
    flex-wrap: wrap;
}

.filter-group {
    display: flex;
    flex-direction: column;
    min-width: 150px;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.filter-group select,
.filter-group input {
    padding: 8px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.card {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
    transition: transform 0.2s;
}

.card:hover {
    transform: translateY(-5px);
}

.card-icon {
    font-size: 2.5em;
    margin-bottom: 15px;
}

.card h3 {
    color: #2c3e50;
    font-size: 1.8em;
    margin-bottom: 5px;
}

.card p {
    color: #7f8c8d;
    font-size: 1.1em;
    margin: 0;
}

.card small {
    color: #95a5a6;
    font-size: 0.9em;
}

.beginning-cash-card .card-icon,
.ending-cash-card .card-icon { color: #27ae60; }
.operating-cash-card .card-icon { color: #3498db; }
.investing-cash-card .card-icon { color: #e74c3c; }
.financing-cash-card .card-icon { color: #9b59b6; }
.net-cash-card.positive .card-icon { color: #27ae60; }
.net-cash-card.negative .card-icon { color: #e74c3c; }

.report-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.report-section h2 {
    color: #2c3e50;
    margin-bottom: 20px;
    border-bottom: 2px solid #e9ecef;
    padding-bottom: 10px;
}

.report-period {
    color: #7f8c8d;
    font-style: italic;
    margin-bottom: 30px;
}

.cash-flow-section {
    margin-bottom: 40px;
}

.cash-flow-section h3 {
    color: #34495e;
    margin-bottom: 20px;
    padding: 10px 15px;
    background: #f8f9fa;
    border-radius: 5px;
}

.table-container {
    overflow-x: auto;
}

.cash-flow-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
}

.cash-flow-table th,
.cash-flow-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.cash-flow-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.cash-flow-table tr:hover {
    background: #f8f9fa;
}

.positive {
    color: #27ae60;
    font-weight: 600;
}

.negative {
    color: #e74c3c;
    font-weight: 600;
}

.total-row {
    background: #f8f9fa;
    font-weight: 600;
}

.final-total {
    background: #2c3e50;
    color: white;
}

.net-cash-flow {
    margin-top: 40px;
    padding-top: 30px;
    border-top: 3px solid #2c3e50;
}

.chart-container {
    margin: 20px 0;
    text-align: center;
}

.analysis-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
}

.analysis-card {
    background: #f8f9fa;
    padding: 20px;
    border-radius: 8px;
    border-left: 4px solid #3498db;
}

.analysis-card h4 {
    margin: 0 0 10px 0;
    color: #2c3e50;
}

.analysis-card p {
    font-size: 1.5em;
    font-weight: 600;
    margin: 0 0 10px 0;
}

.analysis-card small {
    color: #7f8c8d;
    line-height: 1.4;
}

.error-message {
    background: #f8d7da;
    color: #721c24;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #f5c6cb;
}

@media (max-width: 768px) {
    .filter-form {
        flex-direction: column;
        align-items: stretch;
    }

    .filter-group {
        min-width: auto;
    }

    .summary-cards {
        grid-template-columns: 1fr;
    }

    .analysis-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Cash Flow Trends Chart
    const cashFlowCtx = document.getElementById('cashFlowChart').getContext('2d');
    const trendLabels = <?php echo json_encode(array_column($cash_flow_trends, 'month')); ?>;
    const inflowData = <?php echo json_encode(array_column($cash_flow_trends, 'inflows')); ?>;
    const outflowData = <?php echo json_encode(array_column($cash_flow_trends, 'outflows')); ?>;
    const netData = <?php echo json_encode(array_column($cash_flow_trends, 'net')); ?>;

    new Chart(cashFlowCtx, {
        type: 'bar',
        data: {
            labels: trendLabels,
            datasets: [
                {
                    label: 'Cash Inflows',
                    data: inflowData,
                    backgroundColor: 'rgba(46, 204, 113, 0.6)',
                    borderColor: 'rgba(46, 204, 113, 1)',
                    borderWidth: 1
                },
                {
                    label: 'Cash Outflows',
                    data: outflowData,
                    backgroundColor: 'rgba(231, 76, 60, 0.6)',
                    borderColor: 'rgba(231, 76, 60, 1)',
                    borderWidth: 1
                }
            ]
        },
        options: {
            responsive: true,
            scales: {
                x: {
                    stacked: false
                },
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return 'GHC ' + value.toLocaleString();
                        }
                    }
                }
            },
            plugins: {
                legend: {
                    position: 'top'
                }
            }
        }
    });
});
</script>

<?php
include '../includes/admin_footer.php';
?>
