<?php
/**
 * Asset Audit Logging Functions
 *
 * Functions for logging asset-related actions for audit purposes.
 */

/**
 * Log an asset action
 * @param PDO $pdo
 * @param int $asset_id
 * @param int $user_id
 * @param string $action
 * @param array $old_values (optional)
 * @param array $new_values (optional)
 * @param array $changes (optional)
 */
function logAssetAction($pdo, $asset_id, $user_id, $action, $old_values = null, $new_values = null, $changes = null) {
    try {
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
        $session_id = session_id();

        // Get asset name for logging
        $stmt = $pdo->prepare("SELECT name FROM assets WHERE id = ?");
        $stmt->execute([$asset_id]);
        $asset = $stmt->fetch(PDO::FETCH_ASSOC);
        $asset_name = $asset ? $asset['name'] : 'Unknown Asset';

        // Log to asset_audit_logs table
        $stmt = $pdo->prepare("
            INSERT INTO asset_audit_logs
            (asset_id, user_id, action, old_values, new_values, changes, ip_address, user_agent, session_id)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");

        $stmt->execute([
            $asset_id,
            $user_id,
            $action,
            $old_values ? json_encode($old_values) : null,
            $new_values ? json_encode($new_values) : null,
            $changes ? json_encode($changes) : null,
            $ip_address,
            $user_agent,
            $session_id
        ]);

        // Also log to accountant activity logs
        require_once '../includes/accountant_activity_logger.php';

        $description = "Asset {$action}: {$asset_name}";
        if ($changes && is_array($changes)) {
            $change_details = [];
            foreach ($changes as $field => $change) {
                if (isset($change['old']) && isset($change['new'])) {
                    $change_details[] = "{$field}: '{$change['old']}' → '{$change['new']}'";
                }
            }
            if (!empty($change_details)) {
                $description .= " | Changes: " . implode(', ', $change_details);
            }
        }

        AccountantActivityLogger::logActivity($user_id, "asset_{$action}", $description, 0.00, [
            'asset_id' => $asset_id,
            'asset_name' => $asset_name,
            'ip_address' => $ip_address
        ]);

    } catch (Exception $e) {
        // Log the error but don't fail the main operation
        error_log("Asset audit logging failed: " . $e->getMessage());
    }
}

/**
 * Get asset audit logs
 * @param PDO $pdo
 * @param int $asset_id (optional)
 * @param int $user_id (optional)
 * @param string $action (optional)
 * @param int $limit (optional)
 * @return array
 */
function getAssetAuditLogs($pdo, $asset_id = null, $user_id = null, $action = null, $limit = 100) {
    try {
        $where = [];
        $params = [];

        if ($asset_id) {
            $where[] = "asset_id = ?";
            $params[] = $asset_id;
        }

        if ($user_id) {
            $where[] = "user_id = ?";
            $params[] = $user_id;
        }

        if ($action) {
            $where[] = "action = ?";
            $params[] = $action;
        }

        $where_clause = $where ? "WHERE " . implode(" AND ", $where) : "";

        $stmt = $pdo->prepare("
            SELECT aal.*, u.username, ast.name as asset_name
            FROM asset_audit_logs aal
            LEFT JOIN users u ON aal.user_id = u.id
            LEFT JOIN assets ast ON aal.asset_id = ast.id
            {$where_clause}
            ORDER BY aal.created_at DESC
            LIMIT ?
        ");
        $params[] = $limit;
        $stmt->execute($params);

        return $stmt->fetchAll(PDO::FETCH_ASSOC);

    } catch (Exception $e) {
        error_log("Failed to get asset audit logs: " . $e->getMessage());
        return [];
    }
}

/**
 * Get asset audit statistics
 * @param PDO $pdo
 * @param int $days (optional, default 30)
 * @return array
 */
function getAssetAuditStats($pdo, $days = 30) {
    try {
        // Get summary statistics
        $stmt = $pdo->prepare("
            SELECT
                COUNT(*) as total_logs,
                COUNT(DISTINCT CASE WHEN DATE(created_at) = CURDATE() THEN user_id END) as today_logs,
                COUNT(DISTINCT user_id) as unique_users,
                COUNT(DISTINCT asset_id) as unique_assets
            FROM asset_audit_logs
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
        ");
        $stmt->execute([$days]);
        $summary = $stmt->fetch(PDO::FETCH_ASSOC);

        // Get action breakdown by date
        $stmt = $pdo->prepare("
            SELECT
                action,
                COUNT(*) as count,
                DATE(created_at) as date
            FROM asset_audit_logs
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
            GROUP BY action, DATE(created_at)
            ORDER BY date DESC, count DESC
        ");
        $stmt->execute([$days]);
        $action_breakdown = $stmt->fetchAll(PDO::FETCH_ASSOC);

        return array_merge($summary, ['action_breakdown' => $action_breakdown]);

    } catch (Exception $e) {
        error_log("Failed to get asset audit stats: " . $e->getMessage());
        return [
            'total_logs' => 0,
            'today_logs' => 0,
            'unique_users' => 0,
            'unique_assets' => 0,
            'action_breakdown' => []
        ];
    }
}

/**
 * Clean old audit logs
 * @param PDO $pdo
 * @param int $days_to_keep (optional, default 365)
 * @return int Number of deleted records
 */
function cleanAssetAuditLogs($pdo, $days_to_keep = 365) {
    try {
        $stmt = $pdo->prepare("
            DELETE FROM asset_audit_logs
            WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)
        ");
        $stmt->execute([$days_to_keep]);

        return $stmt->rowCount();

    } catch (Exception $e) {
        error_log("Failed to clean asset audit logs: " . $e->getMessage());
        return 0;
    }
}
?>
