<?php
require_once '../config.php';
require_once '../includes/access_control.php';

if (false) { // Temporarily disable feature access check to remove subscription upgrade message
    requireFeatureAccess('accounts_management');
} else {
    // Do nothing, allow access temporarily
}

include 'includes/accounting_header.php';

$message = '';
$error = '';

// Handle approval/rejection actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $expense_id = $_POST['expense_id'] ?? 0;
    $action = $_POST['action'] ?? '';
    $comments = $_POST['comments'] ?? '';

    if ($expense_id && in_array($action, ['approve', 'reject', 'pay'])) {
        try {
            $pdo->beginTransaction();

            if ($action === 'approve') {
                $stmt = $pdo->prepare("
                    UPDATE expenses
                    SET status = 'approved', approved_at = NOW(), approved_by = ?, approval_comments = ?
                    WHERE id = ? AND status = 'pending'
                ");
                $stmt->execute([$_SESSION['user_id'], $comments, $expense_id]);
            } elseif ($action === 'reject') {
                $stmt = $pdo->prepare("
                    UPDATE expenses
                    SET status = 'rejected', approved_at = NOW(), approved_by = ?, approval_comments = ?
                    WHERE id = ? AND status = 'pending'
                ");
                $stmt->execute([$_SESSION['user_id'], $comments, $expense_id]);
            } elseif ($action === 'pay') {
                $stmt = $pdo->prepare("
                    UPDATE expenses
                    SET status = 'paid', paid_at = NOW(), paid_by = ?
                    WHERE id = ? AND status = 'approved'
                ");
                $stmt->execute([$_SESSION['user_id'], $expense_id]);

                // Get expense amount for transaction
                $expense_stmt = $pdo->prepare("SELECT amount FROM expenses WHERE id = ?");
                $expense_stmt->execute([$expense_id]);
                $expense = $expense_stmt->fetch();

                if ($expense) {
                    // Get primary expense account for debiting payment
                    $account_stmt = $pdo->prepare("SELECT id FROM cash_accounts WHERE designation = 'primary_expense' LIMIT 1");
                    $account_stmt->execute();
                    $account = $account_stmt->fetch();

                    if ($account) {
                        // Record transaction debiting the expense account (cash outflow)
                        $transaction_stmt = $pdo->prepare("INSERT INTO cash_account_transactions (account_id, transaction_type, amount, description, created_by) VALUES (?, 'decrease', ?, ?, ?)");
                        $transaction_description = "Expense payment: ID {$expense_id}";
                        $transaction_stmt->execute([$account['id'], $expense['amount'], $transaction_description, $_SESSION['user_id'] ?? null]);

                        // Update account balance
                        $update_balance_stmt = $pdo->prepare("UPDATE cash_accounts SET current_balance = current_balance - ? WHERE id = ?");
                        $update_balance_stmt->execute([$expense['amount'], $account['id']]);
                    }
                }
            }

            $pdo->commit();
            $message = "Expense " . ($action === 'approve' ? 'approved' : ($action === 'pay' ? 'paid' : 'rejected')) . " successfully!";
        } catch (PDOException $e) {
            $pdo->rollBack();
            $error = "Error updating expense: " . $e->getMessage();
        }
    }
}

// Get filter parameters
$status_filter = $_GET['status'] ?? 'pending';
$search = $_GET['search'] ?? '';
$category_filter = $_GET['category'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$sort_by = $_GET['sort_by'] ?? 'expense_date';
$sort_order = $_GET['sort_order'] ?? 'DESC';

// Build query
$where_conditions = [];
$params = [];

if (!empty($status_filter)) {
    $where_conditions[] = "e.status = ?";
    $params[] = $status_filter;
}

if (!empty($search)) {
    $where_conditions[] = "(e.title LIKE ? OR e.vendor_name LIKE ? OR e.description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if (!empty($category_filter)) {
    $where_conditions[] = "e.category_id = ?";
    $params[] = $category_filter;
}

if (!empty($date_from)) {
    $where_conditions[] = "e.expense_date >= ?";
    $params[] = $date_from;
}

if (!empty($date_to)) {
    $where_conditions[] = "e.expense_date <= ?";
    $params[] = $date_to;
}

$where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get total count for pagination
try {
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total
        FROM expenses e
        LEFT JOIN budget_categories bc ON e.category_id = bc.id
        LEFT JOIN users u ON e.created_by = u.id
        LEFT JOIN users au ON e.approved_by = au.id
        $where_clause
    ");
    $stmt->execute($params);
    $total_expenses = $stmt->fetch()['total'];
    $total_pages = ceil($total_expenses / 20);
    $current_page = max(1, min($total_pages, $_GET['page'] ?? 1));
    $offset = ($current_page - 1) * 20;

    // Get expenses with pagination
    $stmt = $pdo->prepare("
        SELECT e.*, bc.name as category_name, u.username as created_by_name,
               s.last_name as created_by_last_name,
               au.username as approved_by_name, e.approved_at, e.approval_comments
        FROM expenses e
        LEFT JOIN budget_categories bc ON e.category_id = bc.id
        LEFT JOIN users u ON e.created_by = u.id
        LEFT JOIN users au ON e.approved_by = au.id
        LEFT JOIN (
            SELECT user_id, CONCAT(first_name, ' ', last_name) as last_name FROM teachers
            UNION ALL
            SELECT user_id, CONCAT(first_name, ' ', last_name) as last_name FROM librarians
            UNION ALL
            SELECT user_id, CONCAT(first_name, ' ', last_name) as last_name FROM non_teaching
            UNION ALL
            SELECT user_id, CONCAT(first_name, ' ', last_name) as last_name FROM directors
            UNION ALL
            SELECT user_id, CONCAT(first_name, ' ', last_name) as last_name FROM head_teachers
            UNION ALL
            SELECT user_id, CONCAT(first_name, ' ', last_name) as last_name FROM admins
        ) s ON e.created_by = s.user_id
        $where_clause
        ORDER BY e.$sort_by $sort_order
        LIMIT ? OFFSET ?
    ");
    $i = 1;
    foreach ($params as $param) {
        $stmt->bindValue($i++, $param);
    }
    $stmt->bindValue($i++, 20, PDO::PARAM_INT);
    $stmt->bindValue($i++, $offset, PDO::PARAM_INT);
    $stmt->execute();
    $expenses = $stmt->fetchAll();

    // Get status summary
    $stmt = $pdo->prepare("
        SELECT status, COUNT(*) as count, COALESCE(SUM(amount), 0) as total_amount
        FROM expenses
        GROUP BY status
    ");
    $stmt->execute();
    $status_summary_raw = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Convert to the expected format
    $status_summary = [];
    foreach ($status_summary_raw as $row) {
        $status_summary[$row['status']] = [
            'count' => $row['count'],
            'total_amount' => $row['total_amount']
        ];
    }

} catch (PDOException $e) {
    $error = "Error loading expenses: " . $e->getMessage();
    $expenses = [];
    $status_summary = [];
}

// Get categories for filter
try {
    $stmt = $pdo->query("SELECT id, name FROM budget_categories WHERE name != 'Revenue' ORDER BY name");
    $categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $categories = [];
}
?>

<style>
.approve-expenses-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 0;
    background: transparent;
    min-height: auto;
    box-shadow: none;
}

.expense-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 30px;
    border-radius: 12px;
    margin-bottom: 30px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.expense-header h1 {
    margin: 0;
    font-size: 2.2em;
    font-weight: 300;
}

.expense-header p {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.expense-content {
    display: grid;
    grid-template-columns: 1fr 320px;
    gap: 30px;
}

.expense-main {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    border: 1px solid #e1e5e9;
}

.filters-section {
    background: #f8f9fa;
    padding: 25px;
    border-bottom: 1px solid #dee2e6;
}

.filters-form {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 20px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 8px;
    color: #495057;
    font-size: 0.95em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.filter-group input,
.filter-group select {
    padding: 12px 16px;
    border: 2px solid #2E7D32;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.3s ease;
}

.filter-group input:focus,
.filter-group select:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.filter-actions {
    display: flex;
    gap: 12px;
}

.btn {
    padding: 12px 24px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(46, 125, 50, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(46, 125, 50, 0.4);
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
}

.btn-success {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
}

.btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(40, 167, 69, 0.4);
}

.btn-danger {
    background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(220, 53, 69, 0.3);
}

.btn-danger:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(220, 53, 69, 0.4);
}

.expenses-table {
    width: 100%;
    border-collapse: collapse;
}

.expenses-table th,
.expenses-table td {
    padding: 16px 20px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.expenses-table th {
    background: #fff;
    font-weight: 700;
    color: #000;
    position: sticky;
    top: 0;
    text-transform: uppercase;
    font-size: 0.85em;
    letter-spacing: 0.5px;
}

.expenses-table tr:hover {
    background: #f8f9fa;
}

.expense-title {
    font-weight: 600;
    color: #2c3e50;
    max-width: 200px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.expense-amount {
    font-weight: 700;
    color: #e74c3c;
    font-size: 1.1em;
}

.expense-category {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 0.8em;
    display: inline-block;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.expense-date {
    color: #6c757d;
    font-size: 0.9em;
    font-weight: 500;
}

.expense-vendor {
    color: #495057;
    font-size: 0.9em;
    max-width: 150px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
    font-weight: 500;
}

.status-badge {
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 0.8em;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    display: inline-block;
}

.status-pending {
    background: #fff3cd;
    color: #856404;
    border: 1px solid #ffeaa7;
}

.status-approved {
    background: #d1ecf1;
    color: #0c5460;
    border: 1px solid #bee5eb;
}

.status-rejected {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.status-paid {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.actions-cell {
    white-space: nowrap;
}

.action-btn {
    padding: 6px 12px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: 0.85em;
    margin-right: 8px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    transition: all 0.3s ease;
}

.approve-btn {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
}

.approve-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 10px rgba(40, 167, 69, 0.3);
}

.reject-btn {
    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
    color: white;
}

.reject-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 10px rgba(220, 53, 69, 0.3);
}

.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-bottom: 25px;
}

.summary-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    padding: 20px;
    border-radius: 12px;
    text-align: center;
    border: 1px solid #e1e5e9;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    transition: transform 0.3s ease;
}

.summary-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
}

.summary-card h3 {
    margin: 0 0 8px 0;
    font-size: 2em;
    color: #2c3e50;
    font-weight: 700;
}

.summary-card p {
    margin: 0;
    color: #6c757d;
    font-size: 0.95em;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.pagination {
    display: flex;
    justify-content: center;
    gap: 8px;
    margin-top: 25px;
    padding-top: 25px;
    border-top: 1px solid #dee2e6;
}

.pagination a,
.pagination span {
    padding: 10px 16px;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    text-decoration: none;
    color: #2E7D32;
    font-weight: 600;
    transition: all 0.3s ease;
}

.pagination .active {
    background: #2E7D32;
    color: white;
    border-color: #2E7D32;
}

.pagination .disabled {
    color: #6c757d;
    cursor: not-allowed;
}

.message {
    padding: 15px;
    margin-bottom: 20px;
    border-radius: 8px;
    font-weight: 600;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.no-expenses {
    text-align: center;
    padding: 50px 20px;
    color: #6c757d;
}

.no-expenses h3 {
    margin: 0 0 10px 0;
    color: #495057;
}

.expense-sidebar {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    padding: 25px;
    border: 1px solid #e1e5e9;
}

.expense-sidebar h3 {
    margin-top: 0;
    color: #2c3e50;
    margin-bottom: 20px;
}

.quick-actions {
    display: flex;
    flex-direction: column;
    gap: 10px;
}

.quick-actions .btn {
    text-align: center;
    padding: 12px;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
}

.modal-content {
    background-color: #fefefe;
    margin: 10% auto;
    padding: 0;
    border: 1px solid #888;
    width: 90%;
    max-width: 500px;
    border-radius: 12px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.15);
}

.modal-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 20px;
    border-radius: 12px 12px 0 0;
}

.modal-header h2 {
    margin: 0;
    font-size: 1.5em;
    font-weight: 300;
}

.modal-body {
    padding: 20px;
}

.modal-footer {
    padding: 20px;
    background-color: #f8f9fa;
    border-radius: 0 0 12px 12px;
    display: flex;
    justify-content: flex-end;
    gap: 10px;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #495057;
}

.form-group textarea {
    width: 100%;
    padding: 12px;
    border: 2px solid #dee2e6;
    border-radius: 8px;
    font-size: 14px;
    resize: vertical;
    min-height: 80px;
}

.form-group textarea:focus {
    outline: none;
    border-color: #2E7D32;
    box-shadow: 0 0 0 3px rgba(46, 125, 50, 0.1);
}

.expense-details {
    background: #f8f9fa;
    padding: 15px;
    border-radius: 8px;
    margin-bottom: 20px;
}

.expense-details h4 {
    margin: 0 0 10px 0;
    color: #2c3e50;
    font-size: 1.1em;
}

.expense-details p {
    margin: 5px 0;
    font-size: 0.9em;
    color: #495057;
}

.expense-details .amount {
    font-weight: 700;
    color: #e74c3c;
    font-size: 1.1em;
}
</style>

<div class="approve-expenses-container">
    <div class="expense-header">
        <div>
            <h1>✅ Expense Approval</h1>
            <p>Review and approve pending expenses</p>
        </div>
        <div>
            <a href="record_expense.php" class="btn btn-success">➕ Add New Expense</a>
            <a href="index.php" class="btn btn-secondary">🏠 Dashboard</a>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="expense-content">
        <div class="expense-main">
            <!-- Summary Cards -->
            <div class="summary-cards">
                <div class="summary-card">
                    <h3><?php echo $status_summary['pending']['count'] ?? 0; ?></h3>
                    <p>Pending Approval</p>
                </div>
                <div class="summary-card">
                    <h3><?php echo $status_summary['approved']['count'] ?? 0; ?></h3>
                    <p>Approved</p>
                </div>
                <div class="summary-card">
                    <h3><?php echo $status_summary['rejected']['count'] ?? 0; ?></h3>
                    <p>Rejected</p>
                </div>
                <div class="summary-card">
                    <h3><?php echo $status_summary['paid']['count'] ?? 0; ?></h3>
                    <p>Paid</p>
                </div>
            </div>

            <!-- Filters -->
            <div class="filters-section">
                <form method="get" class="filters-form">
                    <div class="filter-group">
                        <label for="status">Status</label>
                        <select name="status" id="status">
                            <option value="">All Statuses</option>
                            <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="approved" <?php echo $status_filter === 'approved' ? 'selected' : ''; ?>>Approved</option>
                            <option value="rejected" <?php echo $status_filter === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                            <option value="paid" <?php echo $status_filter === 'paid' ? 'selected' : ''; ?>>Paid</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="search">Search</label>
                        <input type="text" name="search" id="search" value="<?php echo htmlspecialchars($search); ?>"
                               placeholder="Title, vendor, or description...">
                    </div>

                    <div class="filter-group">
                        <label for="category">Category</label>
                        <select name="category" id="category">
                            <option value="">All Categories</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo $category['id']; ?>"
                                        <?php echo $category_filter == $category['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($category['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="date_from">From Date</label>
                        <input type="date" name="date_from" id="date_from" value="<?php echo htmlspecialchars($date_from); ?>">
                    </div>

                    <div class="filter-group">
                        <label for="date_to">To Date</label>
                        <input type="date" name="date_to" id="date_to" value="<?php echo htmlspecialchars($date_to); ?>">
                    </div>

                    <div class="filter-actions">
                        <button type="submit" class="btn btn-primary">🔍 Filter</button>
                        <a href="approve_expenses.php" class="btn btn-secondary">🔄 Reset</a>
                    </div>
                </form>
            </div>

            <!-- Expenses Table -->
            <?php if (empty($expenses)): ?>
                <div class="no-expenses">
                    <h3>No expenses found</h3>
                    <p>Try adjusting your filters or <a href="record_expense.php">record a new expense</a>.</p>
                </div>
            <?php else: ?>
                <div class="table-container">
                    <table class="expenses-table">
                        <thead>
                            <tr>
                                <th>Title</th>
                                <th>Amount</th>
                                <th>Category</th>
                                <th>Date</th>
                                <th>Vendor</th>
                                <th>Prepared By</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($expenses as $expense): ?>
                                <tr>
                                    <td>
                                        <div class="expense-title" title="<?php echo htmlspecialchars($expense['title']); ?>">
                                            <?php echo htmlspecialchars($expense['title']); ?>
                                        </div>
                                        <?php if (!empty($expense['description'])): ?>
                                            <div style="font-size: 0.8em; color: #6c757d; margin-top: 2px;">
                                                <?php echo htmlspecialchars(substr($expense['description'], 0, 50)) . (strlen($expense['description']) > 50 ? '...' : ''); ?>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td class="expense-amount">
                                        GHC <?php echo number_format($expense['amount'], 2); ?>
                                    </td>
                                    <td>
                                        <span class="expense-category">
                                            <?php echo htmlspecialchars($expense['category_name'] ?? 'Uncategorized'); ?>
                                        </span>
                                    </td>
                                    <td class="expense-date">
                                        <?php echo date('M d, Y', strtotime($expense['expense_date'])); ?>
                                    </td>
                                    <td>
                                        <div class="expense-vendor" title="<?php echo htmlspecialchars($expense['vendor_name'] ?? ''); ?>">
                                            <?php echo htmlspecialchars($expense['vendor_name'] ?? 'N/A'); ?>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="expense-vendor">
                                            <?php echo htmlspecialchars($expense['created_by_last_name'] ?? 'N/A'); ?>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="status-badge status-<?php echo $expense['status']; ?>">
                                            <?php echo ucfirst($expense['status']); ?>
                                        </span>
                                        <?php if ($expense['approved_by_name']): ?>
                                            <div style="font-size: 0.7em; color: #6c757d; margin-top: 2px;">
                                                by <?php echo htmlspecialchars($expense['approved_by_name']); ?>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td class="actions-cell">
                                        <?php if ($expense['status'] === 'pending'): ?>
                                            <button class="action-btn approve-btn" onclick="openApprovalModal(<?php echo $expense['id']; ?>, 'approve', '<?php echo htmlspecialchars($expense['title']); ?>', <?php echo $expense['amount']; ?>)">
                                                ✅ Approve
                                            </button>
                                            <button class="action-btn reject-btn" onclick="openApprovalModal(<?php echo $expense['id']; ?>, 'reject', '<?php echo htmlspecialchars($expense['title']); ?>', <?php echo $expense['amount']; ?>)">
                                                ❌ Reject
                                            </button>
                                        <?php elseif ($expense['status'] === 'approved'): ?>
                                            <button class="action-btn approve-btn" onclick="openApprovalModal(<?php echo $expense['id']; ?>, 'pay', '<?php echo htmlspecialchars($expense['title']); ?>', <?php echo $expense['amount']; ?>)">
                                                💰 Pay
                                            </button>
                                        <?php else: ?>
                                            <button class="action-btn" style="background: #6c757d; cursor: not-allowed;" disabled>
                                                <?php echo ucfirst($expense['status']); ?>
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <div class="pagination">
                        <?php if ($current_page > 1): ?>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $current_page - 1])); ?>">Previous</a>
                        <?php else: ?>
                            <span class="disabled">Previous</span>
                        <?php endif; ?>

                        <?php for ($i = max(1, $current_page - 2); $i <= min($total_pages, $current_page + 2); $i++): ?>
                            <?php if ($i == $current_page): ?>
                                <span class="active"><?php echo $i; ?></span>
                            <?php else: ?>
                                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                            <?php endif; ?>
                        <?php endfor; ?>

                        <?php if ($current_page < $total_pages): ?>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $current_page + 1])); ?>">Next</a>
                        <?php else: ?>
                            <span class="disabled">Next</span>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>

        <div class="expense-sidebar">
            <h3>📊 Status Summary</h3>
            <?php if (empty($status_summary)): ?>
                <p style="color: #7f8c8d; font-style: italic;">No expenses to summarize</p>
            <?php else: ?>
                <div style="max-height: 300px; overflow-y: auto;">
                    <?php
                    $status_labels = [
                        'pending' => 'Pending Approval',
                        'approved' => 'Approved',
                        'rejected' => 'Rejected',
                        'paid' => 'Paid'
                    ];
                    $status_colors = [
                        'pending' => '#ffc107',
                        'approved' => '#17a2b8',
                        'rejected' => '#dc3545',
                        'paid' => '#28a745'
                    ];
                    foreach ($status_summary as $status => $data) {
                        if ($data['count'] > 0) {
                            $color = $status_colors[$status] ?? '#6c757d';
                            echo "<div style='display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #f0f0f0;'>";
                            echo "<div>";
                            echo "<strong style='color: {$color}'>{$status_labels[$status]}</strong>";
                            echo "<div style='font-size: 0.8em; color: #7f8c8d;'>{$data['count']} expenses</div>";
                            echo "</div>";
                            echo "<div style='text-align: right;'>";
                            echo "<div style='font-weight: 600; color: #e74c3c;'>GHC " . number_format($data['total_amount'], 2) . "</div>";
                            echo "</div>";
                            echo "</div>";
                        }
                    }
                    ?>
                </div>
            <?php endif; ?>

            <h3 style="margin-top: 30px;">🔗 Quick Actions</h3>
            <div class="quick-actions">
                <a href="record_expense.php" class="btn btn-success">Add New Expense</a>
                <a href="expense_list.php" class="btn btn-primary">View All Expenses</a>
                <a href="balance_sheet_report.php" class="btn btn-primary">View Balance Sheet</a>
                <a href="profit_loss_report.php" class="btn btn-primary">View P&L Report</a>
            </div>
        </div>
    </div>
</div>

<!-- Approval Modal -->
<div id="approvalModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 id="modalTitle">Approve Expense</h2>
        </div>
        <div class="modal-body">
            <div class="expense-details">
                <h4 id="expenseTitle"></h4>
                <p><strong>Amount:</strong> <span id="expenseAmount" class="amount"></span></p>
                <p><strong>Action:</strong> <span id="actionType"></span></p>
            </div>

            <form id="approvalForm" method="post">
                <input type="hidden" name="expense_id" id="modalExpenseId">
                <input type="hidden" name="action" id="modalAction">

                <div class="form-group">
                    <label for="comments">Comments (Optional)</label>
                    <textarea name="comments" id="comments" placeholder="Add any comments or notes about this decision..."></textarea>
                </div>
            </form>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
            <button type="button" class="btn" id="confirmBtn" onclick="submitApproval()">Confirm</button>
        </div>
    </div>
</div>

<script>
function openApprovalModal(expenseId, action, title, amount) {
    document.getElementById('modalExpenseId').value = expenseId;
    document.getElementById('modalAction').value = action;
    document.getElementById('expenseTitle').textContent = title;
    document.getElementById('expenseAmount').textContent = 'GHC ' + amount.toLocaleString();
    document.getElementById('actionType').textContent = action.charAt(0).toUpperCase() + action.slice(1);

    const modal = document.getElementById('approvalModal');
    const modalTitle = document.getElementById('modalTitle');
    const confirmBtn = document.getElementById('confirmBtn');

    if (action === 'approve') {
        modalTitle.textContent = 'Approve Expense';
        confirmBtn.textContent = 'Approve';
        confirmBtn.className = 'btn btn-success';
    } else if (action === 'reject') {
        modalTitle.textContent = 'Reject Expense';
        confirmBtn.textContent = 'Reject';
        confirmBtn.className = 'btn btn-danger';
    } else if (action === 'pay') {
        modalTitle.textContent = 'Pay Expense';
        confirmBtn.textContent = 'Pay';
        confirmBtn.className = 'btn btn-success';
    }

    modal.style.display = 'block';
}

function closeModal() {
    document.getElementById('approvalModal').style.display = 'none';
    document.getElementById('comments').value = '';
}

function submitApproval() {
    document.getElementById('approvalForm').submit();
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('approvalModal');
    if (event.target == modal) {
        closeModal();
    }
}
</script>

<?php include 'includes/accounting_footer.php'; ?>
