<?php
require_once '../config.php';
require_once '../includes/access_control.php';
require_once '../includes/accountant_activity_logger.php';
require_once 'accounting_functions.php';
require_once 'payment_methods.php';

// Access control temporarily disabled for incomes management page
// requireFeatureAccess('accounts_management');

// Handle form submission before any output
$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $source = trim($_POST['source']);
        $amount = $_POST['amount'];
        $income_date = $_POST['income_date'];
        $category_id = $_POST['category_id'];
        $payment_method = $_POST['payment_method'];
        $credit_account = $_POST['credit_account'] ?? null;
        $description = trim($_POST['description']);
        $academic_year_id = $_POST['academic_year_id'];

        // Validate payment method
        if (empty($payment_method)) {
            throw new Exception('Payment method is required');
        }

        // Get cash account ID from payment method
        $cash_account_id = PaymentMethods::getAccountId($payment_method);
        if (!$cash_account_id) {
            throw new Exception('Invalid payment method selected');
        }

        // Validation
        if (empty($source)) {
            throw new Exception('Income source is required');
        }
        if (empty($amount) || !is_numeric($amount) || $amount <= 0) {
            throw new Exception('Valid amount is required');
        }
        if (empty($income_date)) {
            throw new Exception('Income date is required');
        }
        if (empty($category_id)) {
            throw new Exception('Income category is required');
        }
        if (empty($cash_account_id)) {
            throw new Exception('Cash account is required');
        }

        // Verify income category exists
        $stmt = $pdo->prepare("SELECT id, name FROM budget_categories WHERE id = ? AND type = 'income'");
        $stmt->execute([$category_id]);
        $income_category = $stmt->fetch();
        if (!$income_category) {
            throw new Exception('Invalid income category selected');
        }

        // Always require credit account
        if (empty($credit_account)) {
            throw new Exception('Account to credit is required');
        }
        $credit_account_id = $credit_account;

        // Verify credit account exists and is Revenue or Equity
        $stmt = $pdo->prepare("SELECT id, account_name, account_type FROM chart_of_accounts WHERE id = ? AND account_type IN ('Revenue', 'Equity') AND is_active = TRUE");
        $stmt->execute([$credit_account_id]);
        $credit_account_info = $stmt->fetch();
        if (!$credit_account_info) {
            throw new Exception('Invalid account to credit selected');
        }

        // Verify cash account exists and is an asset account
        $stmt = $pdo->prepare("SELECT id, account_name, account_type FROM chart_of_accounts WHERE id = ? AND account_type = 'Asset' AND is_active = TRUE");
        $stmt->execute([$cash_account_id]);
        $cash_account = $stmt->fetch();
        if (!$cash_account) {
            throw new Exception('Invalid cash account selected');
        }

        // Insert income record
        $stmt = $pdo->prepare("
            INSERT INTO incomes (source, amount, income_date, category_id, description, academic_year_id)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([$source, $amount, $income_date, $category_id, $description, $academic_year_id]);
        $income_id = $pdo->lastInsertId();

        // Generate entry number
        $entryNumber = generateEntryNumber($pdo, $income_date);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'income', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $income_date,
            $source . ($description ? ': ' . $description : ''),
            $income_id,
            $_SESSION['user_id'] ?? 1
        ]);
        $entryId = $pdo->lastInsertId();

        // Create debit line (Cash Account)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, debit_amount, line_number, description)
            VALUES (?, ?, ?, 1, ?)
        ");
        $stmt->execute([
            $entryId,
            $cash_account_id,
            $amount,
            'Cash receipt: ' . $source
        ]);

        // Create credit line
        $credit_description = 'Income received: ' . $source;

        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, 2, ?)
        ");
        $stmt->execute([
            $entryId,
            $credit_account_id,
            $amount,
            $credit_description
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$amount, $cash_account_id]); // Debit cash (increase asset)

        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$amount, $credit_account_id]); // Credit account (increase liability/equity/revenue)

        // Get credit account name for logging
        $stmt = $pdo->prepare("SELECT account_name FROM chart_of_accounts WHERE id = ?");
        $stmt->execute([$credit_account_id]);
        $credit_account_info = $stmt->fetch();
        $credit_account_name = $credit_account_info ? $credit_account_info['account_name'] : 'Unknown Account';

        // Log the income activity
        AccountantActivityLogger::logActivity(
            $_SESSION['user_id'],
            'income_recorded',
            "Recorded income of GHC $amount from $source (credited to {$credit_account_name})",
            $amount
        );

        // Redirect to avoid form resubmission on refresh
        header("Location: manage_incomes.php?success=1");
        exit();
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

include 'includes/accounting_header.php';

// Get income categories for dropdown
try {
    $stmt = $pdo->query("SELECT id, name FROM budget_categories WHERE type = 'income' ORDER BY name ASC");
    $income_categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $income_categories = [];
}

// Direct mapping from income type to category ID based on known categories
$income_category_map = [
    'school_fees' => 2, // Tuition Fees
    'pta' => 4, // Donations (PTA Donations)
    'donation' => 4, // Donations
    'other' => 5 // Other Income
];

// Get income types and their corresponding revenue accounts
$income_types = [
    'school_fees' => 'School Fees Revenue',
    'pta' => 'PTA Revenue',
    'donation' => 'Donations Revenue',
    'other' => 'Other Income'
];



// Get academic years for dropdown
try {
    $stmt = $pdo->query("SELECT id, year FROM academic_years ORDER BY year DESC");
    $academic_years = $stmt->fetchAll();
} catch (PDOException $e) {
    $academic_years = [];
}
?>

<style>
.add-income-container {
    max-width: 800px;
    margin: 0 auto;
    padding: 0;
    background: transparent;
    min-height: auto;
    box-shadow: none;
}

.income-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 30px;
    border-radius: 12px;
    margin-bottom: 30px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.income-header h1 {
    margin: 0;
    font-size: 2.2em;
    font-weight: 300;
}

.income-header p {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.form-container {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    padding: 30px;
    border: 1px solid #e1e5e9;
}

.form-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 25px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 8px;
    color: #495057;
    font-size: 0.95em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.form-group input,
.form-group select,
.form-group textarea {
    padding: 12px 16px;
    border: 2px solid #2E7D32;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.3s ease;
    font-family: inherit;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.form-group textarea {
    resize: vertical;
    min-height: 80px;
}

.form-actions {
    display: flex;
    gap: 15px;
    justify-content: center;
    margin-top: 30px;
    padding-top: 30px;
    border-top: 1px solid #dee2e6;
}

.btn {
    padding: 12px 30px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.btn-primary {
    background: linear-gradient(135deg, #FF9800 0%, #FF5722 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(255, 152, 0, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(255, 152, 0, 0.4);
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
}

.message {
    padding: 15px;
    margin-bottom: 20px;
    border-radius: 8px;
    font-weight: 600;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.required {
    color: #dc3545;
}
</style>

<div class="add-income-container">
    <div class="income-header">
        <div>
            <h1>💰 Record Income</h1>
            <p>Add a new income entry to the system</p>
        </div>
        <div>
            <a href="manage_incomes.php" class="btn btn-secondary">🏠 Back to Incomes</a>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="form-container">
        <form method="post">
            <div class="form-grid">

                <div class="form-group">
                    <label for="income_type">Income Type <span class="required">*</span></label>
                    <select id="income_type" name="income_type" required>
                        <option value="">Select Income Type</option>
                        <?php foreach ($income_types as $key => $label): ?>
                            <option value="<?php echo $key; ?>">
                                <?php echo htmlspecialchars($label); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label for="source">Income Source <span class="required">*</span></label>
                    <input type="text" id="source" name="source" required
                           placeholder="e.g., School Fees, PTA Donation">
                </div>

                <div class="form-group">
                    <label for="category_id">Income Category <span class="required">*</span></label>
                    <select id="category_id" name="category_id" required>
                        <option value="">Select Income Category</option>
                        <?php foreach ($income_categories as $category): ?>
                            <option value="<?php echo $category['id']; ?>">
                                <?php echo htmlspecialchars($category['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="amount">Amount (GHC) <span class="required">*</span></label>
                    <input type="number" id="amount" name="amount" step="0.01" min="0" required
                           placeholder="0.00">
                </div>

                <div class="form-group">
                    <label for="income_date">Income Date <span class="required">*</span></label>
                    <input type="date" id="income_date" name="income_date" required>
                </div>

                <div class="form-group">
                    <label for="payment_method">Received Via <span class="required">*</span></label>
                    <select id="payment_method" name="payment_method" required>
                        <option value="">Select Payment Method</option>
                        <?php
                        $payment_methods = PaymentMethods::getFormOptions();
                        foreach ($payment_methods as $option): ?>
                            <option value="<?php echo htmlspecialchars($option['value']); ?>">
                                <?php echo htmlspecialchars($option['label']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="credit_account">Account to Credit <span class="required">*</span></label>
                    <select id="credit_account" name="credit_account" required>
                        <option value="">Select Account to Credit</option>
                        <?php
                        // Get Equity and Revenue accounts for income crediting
                        try {
                            $stmt = $pdo->query("SELECT id, account_code, account_name, account_type FROM chart_of_accounts WHERE account_type IN ('Equity', 'Revenue') AND is_active = TRUE ORDER BY account_type, account_code");
                            $credit_accounts = $stmt->fetchAll();
                            foreach ($credit_accounts as $account): ?>
                                <option value="<?php echo $account['id']; ?>" data-type="<?php echo $account['account_type']; ?>">
                                    <?php echo htmlspecialchars($account['account_code'] . ' - ' . $account['account_name'] . ' (' . $account['account_type'] . ')'); ?>
                                </option>
                            <?php endforeach;
                        } catch (PDOException $e) {
                            // Handle error silently
                        }
                        ?>
                    </select>
                    <small style="color: #6c757d; font-size: 0.85em;">Select which Equity or Revenue account should be credited for this income</small>
                </div>

                <div class="form-group">
                    <label for="academic_year_id">Academic Year</label>
                    <select id="academic_year_id" name="academic_year_id">
                        <option value="">Select Academic Year</option>
                        <?php foreach ($academic_years as $year): ?>
                            <option value="<?php echo $year['id']; ?>">
                                <?php echo htmlspecialchars($year['year']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>

            <div class="form-group" style="margin-top: 25px;">
                <label for="description">Description</label>
                <textarea id="description" name="description" rows="4"
                          placeholder="Additional details about the income..."></textarea>
            </div>

            <div class="form-actions">
                <button type="submit" class="btn btn-primary">💾 Record Income</button>
                <a href="manage_incomes.php" class="btn btn-secondary">❌ Cancel</a>
            </div>
        </form>
    </div>
</div>

<script>
// Set default date to today
document.getElementById('income_date').valueAsDate = new Date();

// Auto-select current academic year if available
<?php if (!empty($academic_years)): ?>
document.getElementById('academic_year_id').value = '<?php echo $academic_years[0]['id']; ?>';
<?php endif; ?>

// Income type to category mapping
const incomeCategoryMap = <?php echo json_encode($income_category_map); ?>;

// Income type to default source mapping
const incomeTypeSources = {
    'school_fees': 'School Fees',
    'pta': 'PTA Contribution',
    'donation': 'Donation',
    'other': ''
};

// Donation category ID
const donationCategoryId = <?php echo $income_category_map['donation']; ?>;

// Function to update credit account field visibility
function updateCreditAccountField() {
    const categorySelect = document.getElementById('category_id');
    const creditAccountGroup = document.getElementById('credit_account_group');
    const creditAccountSelect = document.getElementById('credit_account');

    // Show/hide credit account field if donation category is selected
    if (categorySelect.value == donationCategoryId) {
        creditAccountGroup.style.display = 'block';
        // Make credit account required for donations
        creditAccountSelect.setAttribute('required', 'required');
    } else {
        creditAccountGroup.style.display = 'none';
        creditAccountSelect.removeAttribute('required');
        creditAccountSelect.value = '';
    }
}

// Auto-select income category and fill source when income type changes
document.getElementById('income_type').addEventListener('change', function() {
    const selectedType = this.value;
    const categorySelect = document.getElementById('category_id');
    const sourceInput = document.getElementById('source');

    if (selectedType && incomeCategoryMap[selectedType]) {
        categorySelect.value = incomeCategoryMap[selectedType];
    } else {
        categorySelect.value = '';
    }

    if (selectedType && incomeTypeSources[selectedType]) {
        sourceInput.value = incomeTypeSources[selectedType];
    } else {
        sourceInput.value = '';
    }

    // Update credit account field visibility
    updateCreditAccountField();
});

// Update credit account field when category changes directly
document.getElementById('category_id').addEventListener('change', function() {
    updateCreditAccountField();
});
</script>

<?php include 'includes/accounting_footer.php'; ?>
