<?php
/**
 * Accounting helper functions for proper accrual-based accounting
 */

require_once 'payment_methods.php';

// Include logger with fallback
$loggerPath = '../src/logger.php';
if (file_exists($loggerPath)) {
    require_once $loggerPath;
} else {
    // Fallback logger class if file doesn't exist
    class Logger {
        public function info($message) {}
        public function error($message) {}
        public function warning($message) {}
    }

    function setup_logger() {
        return new Logger();
    }
}

/**
 * Get chart of accounts ID by account name
 * @param PDO $pdo
 * @param string $accountName
 * @return int|null Account ID
 */
function getChartAccountId($pdo, $accountName) {
    $stmt = $pdo->prepare("SELECT id FROM chart_of_accounts WHERE account_name = ? AND is_active = TRUE");
    $stmt->execute([$accountName]);
    $account = $stmt->fetch();
    return $account ? $account['id'] : null;
}

/**
 * Generate journal entry number
 * @param PDO $pdo
 * @param string $date
 * @return string Entry number
 */
function generateEntryNumber($pdo, $date) {
    static $counters = [];

    $year = date('Y', strtotime($date));

    if (!isset($counters[$year])) {
        $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM journal_entries WHERE YEAR(transaction_date) = ?");
        $stmt->execute([$year]);
        $counters[$year] = $stmt->fetch()['count'] + 1;
    } else {
        $counters[$year]++;
    }

    return 'JE-' . $year . '-' . str_pad($counters[$year], 4, '0', STR_PAD_LEFT);
}

/**
 * Create journal entry for fee assignment (accrual basis)
 * Debit: Student Fees Receivable
 * Credit: Tuition Fees
 */
function createFeeAssignmentEntry($pdo, $studentFeeId, $amount, $date, $createdBy) {
    try {
        $pdo->beginTransaction();

        // Get Student Fees Receivable account (1121)
        $receivableAccountId = getChartAccountId($pdo, 'Student Fees Receivable');
        if (!$receivableAccountId) {
            throw new Exception('Student Fees Receivable account not found');
        }

        // Get Tuition Fees account (4000)
        $revenueAccountId = getChartAccountId($pdo, 'Tuition Fees');
        if (!$revenueAccountId) {
            throw new Exception('Tuition Fees account not found');
        }

        $entryNumber = generateEntryNumber($pdo, $date);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'fee_assignment', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $date,
            "Fee assigned to student (Accrual basis)",
            $studentFeeId,
            $createdBy
        ]);

        $logger = setup_logger();
        $logger->info("Fee assignment journal entry created: {$entryNumber} for student fee ID {$studentFeeId}");
        $entryId = $pdo->lastInsertId();

        // Debit: Student Fees Receivable (increases asset)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, debit_amount, line_number, description)
            VALUES (?, ?, ?, 1, ?)
        ");
        $stmt->execute([
            $entryId,
            $receivableAccountId,
            $amount,
            'Student fees receivable (fee assigned)'
        ]);

        // Credit: Tuition Fees (increases income)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, 2, ?)
        ");
        $stmt->execute([
            $entryId,
            $revenueAccountId,
            $amount,
            'Tuition fees earned'
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$amount, $receivableAccountId]); // Debit increases asset

        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$amount, $revenueAccountId]); // Credit increases income

        $pdo->commit();
        return $entryId;

    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

/**
 * Create journal entry for expense liability (accrual basis - when expense is recorded)
 * Debit: Expense Account, Credit: Accounts Payable
 */
function createExpenseLiabilityEntry($pdo, $expenseId, $amount, $date, $createdBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get expense details
        $stmt = $pdo->prepare("
            SELECT e.*, bc.name as category_name
            FROM expenses e
            LEFT JOIN budget_categories bc ON e.category_id = bc.id
            WHERE e.id = ?
        ");
        $stmt->execute([$expenseId]);
        $expense = $stmt->fetch();

        if (!$expense) {
            throw new Exception('Expense not found');
        }

        // Get expense account from budget category
        $expenseAccountId = getChartAccountId($pdo, $expense['category_name']);
        if (!$expenseAccountId) {
            // Map budget categories to chart of accounts
            $categoryMapping = [
                'Utilities' => 'Utilities',
                'Maintenance' => 'Supplies',
                'Teaching Materials' => 'Supplies',
                'Office Supplies' => 'Supplies',
                'Transportation' => 'Supplies',
                'Technology' => 'Supplies',
                'Salaries' => 'Salaries and Wages',
                'Other Expenses' => 'Supplies'
            ];

            $mappedAccount = $categoryMapping[$expense['category_name']] ?? 'Supplies';
            $expenseAccountId = getChartAccountId($pdo, $mappedAccount);
            if (!$expenseAccountId) {
                throw new Exception('Expense account not found for category: ' . $expense['category_name']);
            }
        }

        // Get Accounts Payable account
        $payableAccountId = getChartAccountId($pdo, 'Accounts Payable');
        if (!$payableAccountId) {
            throw new Exception('Accounts Payable account not found');
        }

        $entryNumber = generateEntryNumber($pdo, $date);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'expense', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $date,
            "Expense recorded: {$expense['title']} (Accrual basis - liability)",
            $expenseId,
            $createdBy
        ]);
        $entryId = $pdo->lastInsertId();

        // Debit: Expense Account (increases expense)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, debit_amount, line_number, description)
            VALUES (?, ?, ?, 1, ?)
        ");
        $stmt->execute([
            $entryId,
            $expenseAccountId,
            $amount,
            "Expense incurred: {$expense['title']} (accrual basis)"
        ]);

        // Credit: Accounts Payable (increases liability)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, 2, ?)
        ");
        $stmt->execute([
            $entryId,
            $payableAccountId,
            $amount,
            'Accounts payable (expense liability)'
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$amount, $expenseAccountId]); // Debit increases expense

        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$amount, $payableAccountId]); // Credit increases liability

        if ($manageTransaction) {
            $pdo->commit();
        }

        $logger = setup_logger();
        $logger->info("Expense liability journal entry created: {$entryNumber} for expense ID {$expenseId}");

        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}

/**
 * Create journal entry for expense approval (accrual basis)
 * Debit: Expense Account, Credit: Accounts Payable
 */
function createExpenseApprovalEntry($pdo, $expenseId, $amount, $date, $approvedBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get expense details
        $stmt = $pdo->prepare("
            SELECT e.*, bc.name as category_name
            FROM expenses e
            LEFT JOIN budget_categories bc ON e.category_id = bc.id
            WHERE e.id = ?
        ");
        $stmt->execute([$expenseId]);
        $expense = $stmt->fetch();

        if (!$expense) {
            throw new Exception('Expense not found');
        }

        // Get expense account from budget category
        $expenseAccountId = getChartAccountId($pdo, $expense['category_name']);
        if (!$expenseAccountId) {
            // Map budget categories to chart of accounts
            $categoryMapping = [
                'Utilities' => 'Utilities',
                'Maintenance' => 'Supplies',
                'Teaching Materials' => 'Supplies',
                'Office Supplies' => 'Supplies',
                'Transportation' => 'Supplies',
                'Technology' => 'Supplies',
                'Salaries' => 'Salaries and Wages',
                'Other Expenses' => 'Supplies'
            ];

            $mappedAccount = $categoryMapping[$expense['category_name']] ?? 'Supplies';
            $expenseAccountId = getChartAccountId($pdo, $mappedAccount);
            if (!$expenseAccountId) {
                throw new Exception('Expense account not found for category: ' . $expense['category_name']);
            }
        }

        // Get Accounts Payable account
        $payableAccountId = getChartAccountId($pdo, 'Accounts Payable');
        if (!$payableAccountId) {
            throw new Exception('Accounts Payable account not found');
        }

        $entryNumber = generateEntryNumber($pdo, $date);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'expense', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $date,
            "Expense approved: {$expense['title']} (Accrual basis)",
            $expenseId,
            $approvedBy
        ]);
        $entryId = $pdo->lastInsertId();

        // Debit: Expense Account (increases expense)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, debit_amount, line_number, description)
            VALUES (?, ?, ?, 1, ?)
        ");
        $stmt->execute([
            $entryId,
            $expenseAccountId,
            $amount,
            "Expense incurred: {$expense['title']} (accrual basis)"
        ]);

        // Credit: Accounts Payable (increases liability)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, 2, ?)
        ");
        $stmt->execute([
            $entryId,
            $payableAccountId,
            $amount,
            'Accounts payable (expense approved)'
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$amount, $expenseAccountId]); // Debit increases expense

        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$amount, $payableAccountId]); // Credit increases liability

        if ($manageTransaction) {
            $pdo->commit();
        }

        $logger = setup_logger();
        $logger->info("Expense approval journal entry created: {$entryNumber} for expense ID {$expenseId}");

        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}

/**
 * Create journal entry for expense payment (cash basis)
 * Debit: Expense Account + Debit: Accounts Payable
 * Credit: Cash Account (based on payment method)
 */
function createExpensePaymentEntry($pdo, $expenseId, $amount, $paymentMethod, $date, $paidBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get expense details
        $stmt = $pdo->prepare("
            SELECT e.*, bc.name as category_name
            FROM expenses e
            LEFT JOIN budget_categories bc ON e.category_id = bc.id
            WHERE e.id = ?
        ");
        $stmt->execute([$expenseId]);
        $expense = $stmt->fetch();

        if (!$expense) {
            throw new Exception('Expense not found');
        }

        // Get expense account from budget category
        $expenseAccountId = getChartAccountId($pdo, $expense['category_name']);
        if (!$expenseAccountId) {
            // Map budget categories to chart of accounts
            $categoryMapping = [
                'Utilities' => 'Utilities',
                'Maintenance' => 'Supplies',
                'Teaching Materials' => 'Supplies',
                'Office Supplies' => 'Supplies',
                'Transportation' => 'Supplies',
                'Technology' => 'Supplies',
                'Salaries' => 'Salaries and Wages',
                'Other Expenses' => 'Supplies'
            ];

            $mappedAccount = $categoryMapping[$expense['category_name']] ?? 'Supplies';
            $expenseAccountId = getChartAccountId($pdo, $mappedAccount);
            if (!$expenseAccountId) {
                throw new Exception('Expense account not found for category: ' . $expense['category_name']);
            }
        }

        // Get Accounts Payable account
        $payableAccountId = getChartAccountId($pdo, 'Accounts Payable');
        if (!$payableAccountId) {
            throw new Exception('Accounts Payable account not found');
        }

            // Get cash account based on payment method
        $cashAccountName = PaymentMethods::getCashAccountForPaymentMethod($paymentMethod);
        $cashAccountId = getChartAccountId($pdo, $cashAccountName);
        if (!$cashAccountId) {
            throw new Exception("Cash account '{$cashAccountName}' not found");
        }

        $entryNumber = generateEntryNumber($pdo, $date);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'payment', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $date,
            "Expense paid: {$expense['title']} via {$paymentMethod} (Cash basis)",
            $expenseId,
            $paidBy
        ]);
        $entryId = $pdo->lastInsertId();

        // Debit: Accounts Payable (decreases liability)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, debit_amount, line_number, description)
            VALUES (?, ?, ?, 1, ?)
        ");
        $stmt->execute([
            $entryId,
            $payableAccountId,
            $amount,
            'Accounts payable (expense paid)'
        ]);

        // Credit: Cash Account (decreases cash)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, 2, ?)
        ");
        $stmt->execute([
            $entryId,
            $cashAccountId,
            $amount,
            "Cash payment: {$expense['title']} ({$cashAccountName})"
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
        $stmt->execute([$amount, $payableAccountId]); // Debit decreases liability

        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
        $stmt->execute([$amount, $cashAccountId]); // Credit decreases asset

        if ($manageTransaction) {
            $pdo->commit();
        }
        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}

/**
 * Create journal entry for asset depreciation (accrual basis)
 * Debit: Depreciation Expense
 * Credit: Accumulated Depreciation
 */
function createDepreciationEntry($pdo, $assetId, $depreciationAmount, $transactionDate, $createdBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get asset details for description
        $stmt = $pdo->prepare("SELECT name FROM assets WHERE id = ?");
        $stmt->execute([$assetId]);
        $asset = $stmt->fetch();

        if (!$asset) {
            throw new Exception('Asset not found');
        }

        // Get Depreciation Expense account (5400)
        $depreciationExpenseId = getChartAccountId($pdo, 'Depreciation Expense');
        if (!$depreciationExpenseId) {
            throw new Exception('Depreciation Expense account not found');
        }

        // Get Accumulated Depreciation account (1220)
        $accumulatedDepreciationId = getChartAccountId($pdo, 'Accumulated Depreciation');
        if (!$accumulatedDepreciationId) {
            throw new Exception('Accumulated Depreciation account not found');
        }

        $entryNumber = generateEntryNumber($pdo, $transactionDate);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'depreciation', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $transactionDate,
            "Depreciation expense for asset: {$asset['name']}",
            $assetId,
            $createdBy
        ]);
        $entryId = $pdo->lastInsertId();

        // Debit: Depreciation Expense (increases expense)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, debit_amount, line_number, description)
            VALUES (?, ?, ?, 1, ?)
        ");
        $stmt->execute([
            $entryId,
            $depreciationExpenseId,
            $depreciationAmount,
            "Depreciation expense: {$asset['name']}"
        ]);

        // Credit: Accumulated Depreciation (increases contra-asset)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, 2, ?)
        ");
        $stmt->execute([
            $entryId,
            $accumulatedDepreciationId,
            $depreciationAmount,
            "Accumulated depreciation: {$asset['name']}"
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$depreciationAmount, $depreciationExpenseId]); // Debit increases expense

        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$depreciationAmount, $accumulatedDepreciationId]); // Credit increases contra-asset

        if ($manageTransaction) {
            $pdo->commit();
        }
        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}

/**
 * Create journal entry for payroll expenses (accrual basis)
 * Debit: Salaries and Wages Expense, Employer SSNIT Expense, PAYE Expense
 * Credit: SSNIT Payable, PAYE Payable, Net Salaries Payable
 */
function createPayrollExpenseEntry($pdo, $payrollRunId, $totalGrossSalary, $totalEmployerSSNIT, $totalTier2Pension, $totalPAYE, $totalNetSalary, $transactionDate, $createdBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get required accounts
        $salaryExpenseId = getChartAccountId($pdo, 'Salaries and Wages Expense');
        $employerSSNITExpenseId = getChartAccountId($pdo, 'Employer SSNIT Expense');
        $tier2PensionExpenseId = getChartAccountId($pdo, 'Tier 2 Pension Expense');
        $payeExpenseId = getChartAccountId($pdo, 'PAYE Expense');
        $ssnitPayableId = getChartAccountId($pdo, 'SSNIT Payable');
        $tier2PensionPayableId = getChartAccountId($pdo, 'Tier 2 Pension Payable');
        $payePayableId = getChartAccountId($pdo, 'PAYE Payable');
        $netSalariesPayableId = getChartAccountId($pdo, 'Net Salaries Payable');

        if (!$salaryExpenseId || !$employerSSNITExpenseId || !$tier2PensionExpenseId || !$payeExpenseId ||
            !$ssnitPayableId || !$tier2PensionPayableId || !$payePayableId || !$netSalariesPayableId) {
            throw new Exception('Required payroll accounts not found');
        }

        $entryNumber = generateEntryNumber($pdo, $transactionDate);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'payroll_expense', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $transactionDate,
            "Payroll expenses for payroll run ID: {$payrollRunId}",
            $payrollRunId,
            $createdBy
        ]);
        $entryId = $pdo->lastInsertId();

        $lineNumber = 1;

        // Debit: Salaries and Wages Expense
        if ($totalGrossSalary > 0) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $salaryExpenseId,
                $totalGrossSalary,
                $lineNumber++,
                'Salaries and wages expense'
            ]);
        }

        // Debit: Employer SSNIT Expense
        if ($totalEmployerSSNIT > 0) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $employerSSNITExpenseId,
                $totalEmployerSSNIT,
                $lineNumber++,
                'Employer SSNIT contribution expense'
            ]);
        }

        // Debit: PAYE Expense
        if ($totalPAYE > 0) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $payeExpenseId,
                $totalPAYE,
                $lineNumber++,
                'PAYE tax expense'
            ]);
        }

        // Debit: Tier 2 Pension Expense
        if ($totalTier2Pension > 0) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $tier2PensionExpenseId,
                $totalTier2Pension,
                $lineNumber++,
                'Tier 2 Pension contribution expense'
            ]);
        }

        // Credit: SSNIT Payable (employee and employer portions)
        // Note: totalEmployerSSNIT already includes both employee and employer portions
        $totalSSNITPayable = $totalEmployerSSNIT;
        if ($totalSSNITPayable > 0) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, credit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $ssnitPayableId,
                $totalSSNITPayable,
                $lineNumber++,
                'SSNIT payable (employee and employer contributions)'
            ]);
        }

        // Credit: PAYE Payable
        if ($totalPAYE > 0) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, credit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $payePayableId,
                $totalPAYE,
                $lineNumber++,
                'PAYE tax payable'
            ]);
        }

        // Credit: Tier 2 Pension Payable
        if ($totalTier2Pension > 0) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, credit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $tier2PensionPayableId,
                $totalTier2Pension,
                $lineNumber++,
                'Tier 2 Pension payable (employer contribution)'
            ]);
        }

        // Credit: Net Salaries Payable (using gross salary for balancing)
        if ($totalGrossSalary > 0) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, credit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $netSalariesPayableId,
                $totalGrossSalary,
                $lineNumber++,
                'Salaries payable to employees'
            ]);
        }

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        if ($totalGrossSalary > 0) {
            $stmt->execute([$totalGrossSalary, $salaryExpenseId]); // Debit increases expense
        }
        if ($totalEmployerSSNIT > 0) {
            $stmt->execute([$totalEmployerSSNIT, $employerSSNITExpenseId]); // Debit increases expense
        }
        if ($totalTier2Pension > 0) {
            $stmt->execute([$totalTier2Pension, $tier2PensionExpenseId]); // Debit increases expense
        }
        if ($totalPAYE > 0) {
            $stmt->execute([$totalPAYE, $payeExpenseId]); // Debit increases expense
        }

        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        if ($totalSSNITPayable > 0) {
            $stmt->execute([$totalSSNITPayable, $ssnitPayableId]); // Credit increases liability
        }
        if ($totalPAYE > 0) {
            $stmt->execute([$totalPAYE, $payePayableId]); // Credit increases liability
        }
        if ($totalGrossSalary > 0) {
            $stmt->execute([$totalGrossSalary, $netSalariesPayableId]); // Credit increases liability
        }

        if ($manageTransaction) {
            $pdo->commit();
        }

        $logger = setup_logger();
        $logger->info("Payroll expense journal entry created: {$entryNumber} for payroll run ID {$payrollRunId}");

        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}

/**
 * Create journal entry for payroll payment (cash basis)
 * Debit: SSNIT Payable, PAYE Payable, Net Salaries Payable
 * Credit: Cash/Bank Account
 */
function createPayrollPaymentEntry($pdo, $payrollRunId, $totalEmployeeSSNIT, $totalPAYE, $totalNetSalary, $paymentMethod, $transactionDate, $paidBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get required accounts
        $ssnitPayableId = getChartAccountId($pdo, 'SSNIT Payable');
        $payePayableId = getChartAccountId($pdo, 'PAYE Payable');
        $netSalariesPayableId = getChartAccountId($pdo, 'Net Salaries Payable');

        // Get cash account based on payment method
        $cashAccountName = PaymentMethods::getCashAccountForPaymentMethod($paymentMethod);
        $cashAccountId = getChartAccountId($pdo, $cashAccountName);

        if (!$ssnitPayableId || !$payePayableId || !$netSalariesPayableId || !$cashAccountId) {
            throw new Exception('Required payroll payment accounts not found');
        }

        $entryNumber = generateEntryNumber($pdo, $transactionDate);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'payroll_payment', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $transactionDate,
            "Payroll payments for payroll run ID: {$payrollRunId} via {$paymentMethod}",
            $payrollRunId,
            $paidBy
        ]);
        $entryId = $pdo->lastInsertId();

        $lineNumber = 1;

        // Debit: SSNIT Payable (decreases liability)
        if ($totalEmployeeSSNIT > 0) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $ssnitPayableId,
                $totalEmployeeSSNIT,
                $lineNumber++,
                'Employee SSNIT payment'
            ]);
        }

        // Debit: PAYE Payable (decreases liability)
        if ($totalPAYE > 0) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $payePayableId,
                $totalPAYE,
                $lineNumber++,
                'PAYE tax payment'
            ]);
        }

        // Debit: Net Salaries Payable (decreases liability)
        if ($totalNetSalary > 0) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $netSalariesPayableId,
                $totalNetSalary,
                $lineNumber++,
                'Net salaries payment to employees'
            ]);
        }

        // Credit: Cash/Bank Account (decreases asset)
        $totalPayment = $totalEmployeeSSNIT + $totalPAYE + $totalNetSalary;
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $entryId,
            $cashAccountId,
            $totalPayment,
            $lineNumber++,
            "Payroll payments via {$paymentMethod} ({$cashAccountName})"
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
        if ($totalEmployeeSSNIT > 0) {
            $stmt->execute([$totalEmployeeSSNIT, $ssnitPayableId]); // Debit decreases liability
        }
        if ($totalPAYE > 0) {
            $stmt->execute([$totalPAYE, $payePayableId]); // Debit decreases liability
        }
        if ($totalNetSalary > 0) {
            $stmt->execute([$totalNetSalary, $netSalariesPayableId]); // Debit decreases liability
        }

        $stmt->execute([$totalPayment, $cashAccountId]); // Credit decreases asset

        if ($manageTransaction) {
            $pdo->commit();
        }

        $logger = setup_logger();
        $logger->info("Payroll payment journal entry created: {$entryNumber} for payroll run ID {$payrollRunId}");

        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}

/**
 * Create journal entry for invoice approval (accrual basis)
 * Debit: Expense Accounts (based on PO items)
 * Credit: Accounts Payable
 */
function createInvoiceApprovalEntry($pdo, $invoiceId, $approvedBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get invoice details with PO and supplier info
        $stmt = $pdo->prepare("
            SELECT si.*, po.po_number, s.name AS supplier_name
            FROM supplier_invoices si
            JOIN purchase_orders po ON si.purchase_order_id = po.id
            JOIN suppliers s ON po.supplier_id = s.id
            WHERE si.id = ?
        ");
        $stmt->execute([$invoiceId]);
        $invoice = $stmt->fetch();

        if (!$invoice) {
            throw new Exception('Invoice not found');
        }

        // Get Accounts Payable account
        $payableAccountId = getChartAccountId($pdo, 'Accounts Payable');
        if (!$payableAccountId) {
            throw new Exception('Accounts Payable account not found');
        }

        // Get invoice items with PO item details
        $stmt = $pdo->prepare("
            SELECT sii.*, poi.description, poi.quantity as po_quantity, poi.unit_price as po_unit_price,
                   bc.name as budget_category_name
            FROM supplier_invoice_items sii
            JOIN purchase_order_items poi ON sii.purchase_order_item_id = poi.id
            LEFT JOIN budget_categories bc ON poi.category_id = bc.id
            WHERE sii.supplier_invoice_id = ?
        ");
        $stmt->execute([$invoiceId]);
        $invoiceItems = $stmt->fetchAll();

        if (empty($invoiceItems)) {
            throw new Exception('No invoice items found');
        }

        $entryNumber = generateEntryNumber($pdo, date('Y-m-d'));
        $totalAmount = $invoice['total_amount'];

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'invoice_approval', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            date('Y-m-d'),
            "Invoice approval: {$invoice['invoice_number']} from {$invoice['supplier_name']} (PO: {$invoice['po_number']})",
            $invoiceId,
            $approvedBy
        ]);
        $entryId = $pdo->lastInsertId();

        $lineNumber = 1;

        // Group items by expense account for consolidated entries
        $expenseEntries = [];

        foreach ($invoiceItems as $item) {
            // Determine expense account
            $expenseAccountName = 'Supplies'; // Default

            if ($item['budget_category_name']) {
                // Map budget categories to chart of accounts
                $categoryMapping = [
                    'Utilities' => 'Utilities',
                    'Maintenance' => 'Supplies',
                    'Teaching Materials' => 'Supplies',
                    'Office Supplies' => 'Supplies',
                    'Transportation' => 'Supplies',
                    'Technology' => 'Supplies',
                    'Salaries' => 'Salaries and Wages',
                    'Other Expenses' => 'Supplies'
                ];

                $expenseAccountName = $categoryMapping[$item['budget_category_name']] ?? 'Supplies';
            }

            $expenseAccountId = getChartAccountId($pdo, $expenseAccountName);
            if (!$expenseAccountId) {
                throw new Exception("Expense account '{$expenseAccountName}' not found");
            }

            if (!isset($expenseEntries[$expenseAccountId])) {
                $expenseEntries[$expenseAccountId] = [
                    'account_id' => $expenseAccountId,
                    'account_name' => $expenseAccountName,
                    'amount' => 0
                ];
            }

            $expenseEntries[$expenseAccountId]['amount'] += $item['total_price'];
        }

        // Create debit entries for each expense account
        foreach ($expenseEntries as $expenseEntry) {
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, line_number, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entryId,
                $expenseEntry['account_id'],
                $expenseEntry['amount'],
                $lineNumber++,
                "Invoice expense: {$expenseEntry['account_name']} ({$invoice['invoice_number']})"
            ]);
        }

        // Credit: Accounts Payable
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $entryId,
            $payableAccountId,
            $totalAmount,
            $lineNumber++,
            "Accounts payable: Invoice {$invoice['invoice_number']} from {$invoice['supplier_name']}"
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        foreach ($expenseEntries as $expenseEntry) {
            $stmt->execute([$expenseEntry['amount'], $expenseEntry['account_id']]); // Debit increases expense
        }

        $stmt->execute([$totalAmount, $payableAccountId]); // Credit increases liability

        if ($manageTransaction) {
            $pdo->commit();
        }

        $logger = setup_logger();
        $logger->info("Invoice approval journal entry created: {$entryNumber} for invoice ID {$invoiceId}");

        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}

/**
 * Create journal entry for statutory payments (SSNIT/PAYE to government)
 * Debit: SSNIT Payable, PAYE Payable
 * Credit: Cash/Bank Account
 */
function createStatutoryPaymentEntry($pdo, $paymentType, $amount, $paymentMethod, $transactionDate, $paidBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get required accounts
        if ($paymentType === 'ssnit') {
            $payableAccountId = getChartAccountId($pdo, 'SSNIT Payable');
            $description = 'SSNIT statutory payment';
        } elseif ($paymentType === 'paye') {
            $payableAccountId = getChartAccountId($pdo, 'PAYE Payable');
            $description = 'PAYE statutory payment';
        } elseif ($paymentType === 'tier2') {
            $payableAccountId = getChartAccountId($pdo, 'Tier 2 Pension Payable');
            $description = 'Tier 2 Pension statutory payment';
        } else {
            throw new Exception('Invalid payment type. Must be ssnit, paye, or tier2');
        }

        // Get cash account based on payment method
        $cashAccountName = PaymentMethods::getCashAccountForPaymentMethod($paymentMethod);
        $cashAccountId = getChartAccountId($pdo, $cashAccountName);

        if (!$payableAccountId || !$cashAccountId) {
            throw new Exception('Required statutory payment accounts not found');
        }

        $entryNumber = generateEntryNumber($pdo, $transactionDate);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'statutory_payment', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $transactionDate,
            "{$description} via {$paymentMethod}",
            null, // No specific source ID for statutory payments
            $paidBy
        ]);
        $entryId = $pdo->lastInsertId();

        // Debit: Payable Account (decreases liability)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, debit_amount, line_number, description)
            VALUES (?, ?, ?, 1, ?)
        ");
        $stmt->execute([
            $entryId,
            $payableAccountId,
            $amount,
            $description
        ]);

        // Credit: Cash/Bank Account (decreases asset)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, 2, ?)
        ");
        $stmt->execute([
            $entryId,
            $cashAccountId,
            $amount,
            "Statutory payment via {$paymentMethod} ({$cashAccountName})"
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
        $stmt->execute([$amount, $payableAccountId]); // Debit decreases liability

        $stmt->execute([$amount, $cashAccountId]); // Credit decreases asset

        if ($manageTransaction) {
            $pdo->commit();
        }

        $logger = setup_logger();
        $logger->info("Statutory payment journal entry created: {$entryNumber} for {$paymentType} payment");

        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}
?>
