<?php
require_once '../config.php';

if (!in_array($_SESSION['role'], ['admin', 'teacher'])) {
    header('Location: ../dashboard.php');
    exit;
}

$message = '';

$user_id = $_SESSION['user_id'];

// Get teacher id
$stmt = $pdo->prepare("SELECT id FROM staff WHERE user_id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher not found.");
}

$teacher_id = $teacher['id'];

// Get classes assigned to this teacher
$stmt = $pdo->prepare("SELECT c.id, c.name FROM classes c JOIN teacher_classes tc ON c.id = tc.class_id WHERE tc.teacher_id = ?");
$stmt->execute([$teacher_id]);
$classes = $stmt->fetchAll();

// Get filter values
$filter_class = $_GET['class'] ?? '';
$filter_academic_year = $_GET['academic_year'] ?? '';

// Get students for the selected class and academic year
$query = "SELECT s.*, u.username, p.full_name as parent_name, p.mobile_phone as parent_phone FROM students s JOIN student_users u ON s.user_id = u.id LEFT JOIN parents p ON s.id = p.student_id WHERE 1=1";
$params = [];

if ($filter_class) {
    $query .= " AND s.class = ?";
    $params[] = $filter_class;
}
if ($filter_academic_year) {
    $query .= " AND s.academic_year = ?";
    $params[] = $filter_academic_year;
}

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$students = $stmt->fetchAll();

$academic_years_stmt = $pdo->query("SELECT DISTINCT academic_year FROM students ORDER BY academic_year");
$academic_years = $academic_years_stmt->fetchAll(PDO::FETCH_COLUMN);

$role = $_SESSION['role'];

if ($role == 'admin') {
    include '../includes/admin_header.php';
} else {
    include '../includes/teacher_header.php';
}
?>

<style>
    .view-students-container {
        padding: 20px;
        background-color: #f8f9fa;
        min-height: 100vh;
    }

    .page-header {
        background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
        color: white;
        padding: 30px;
        border-radius: 15px;
        margin-bottom: 30px;
        text-align: center;
        box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    }

    .page-header h2 {
        margin: 0;
        font-size: 2.5em;
        font-weight: 700;
    }

    .filter-form {
        background: white;
        padding: 25px;
        border-radius: 15px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
        display: flex;
        flex-wrap: wrap;
        gap: 20px;
        align-items: center;
    }

    .filter-form label {
        font-weight: 600;
        color: #2c3e50;
        margin-right: 10px;
    }

    .filter-form select {
        padding: 10px 15px;
        border: 2px solid #ddd;
        border-radius: 8px;
        font-size: 1em;
        background-color: #fff;
        transition: border-color 0.3s ease;
    }

    .filter-form select:focus {
        outline: none;
        border-color: #27ae60;
    }

    .students-table-container {
        background: white;
        border-radius: 15px;
        padding: 25px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
        overflow-x: auto;
    }

    .students-table-container h3 {
        color: #2c3e50;
        margin-bottom: 20px;
        font-size: 1.8em;
    }

    table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    th, td {
        padding: 15px;
        text-align: left;
        border-bottom: 1px solid #ddd;
    }

    th {
        background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
        color: white;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    tr:nth-child(even) {
        background-color: #f8f9fa;
    }

    tr:hover {
        background-color: #e8f5e8;
    }

    .classes-list {
        background: white;
        border-radius: 15px;
        padding: 25px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
    }

    .classes-list h3 {
        color: #2c3e50;
        margin-bottom: 20px;
        font-size: 1.8em;
    }

    .classes-list ul {
        list-style: none;
        padding: 0;
    }

    .classes-list li {
        padding: 15px;
        border: 1px solid #ddd;
        border-radius: 8px;
        margin-bottom: 10px;
        display: flex;
        justify-content: space-between;
        align-items: center;
        background-color: #f8f9fa;
    }

    .classes-list a {
        background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
        color: white;
        padding: 8px 15px;
        text-decoration: none;
        border-radius: 5px;
        font-weight: 600;
        transition: all 0.3s ease;
    }

    .classes-list a:hover {
        background: linear-gradient(135deg, #2980b9 0%, #2471a3 100%);
        text-decoration: none;
        color: white;
    }

    .btn-print {
        background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
        color: white;
        padding: 10px 15px;
        text-decoration: none;
        border-radius: 5px;
        font-weight: 600;
        transition: all 0.3s ease;
        display: inline-flex;
        align-items: center;
        gap: 5px;
    }

    .btn-print:hover {
        background: linear-gradient(135deg, #34495e 0%, #2c3e50 100%);
        text-decoration: none;
        color: white;
    }

    @media (max-width: 768px) {
        .filter-form {
            flex-direction: column;
            align-items: stretch;
        }

        .filter-form label {
            margin-bottom: 5px;
        }

        table {
            font-size: 0.9em;
        }

        th, td {
            padding: 10px;
        }
    }
</style>

<div class="view-students-container">
    <div class="page-header">
        <h2>👨‍🎓 View Students</h2>
    </div>

    <form method="get" action="view_students.php" class="filter-form">
        <div>
            <label for="class">Select Class:</label>
            <select id="class" name="class" onchange="this.form.submit()">
                <option value="">-- Select Class --</option>
                <?php foreach ($classes as $class): ?>
                    <option value="<?php echo htmlspecialchars($class['name']); ?>" <?php if ($class['name'] == $filter_class) echo 'selected'; ?>>
                        <?php echo htmlspecialchars($class['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div>
            <label for="academic_year">Filter by Academic Year:</label>
            <select id="academic_year" name="academic_year" onchange="this.form.submit()">
                <option value="">All</option>
                <?php foreach ($academic_years as $year): ?>
                    <option value="<?php echo htmlspecialchars($year); ?>" <?php if ($year == $filter_academic_year) echo 'selected'; ?>>
                        <?php echo htmlspecialchars($year); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <noscript><button type="submit" class="btn btn-secondary">Filter</button></noscript>
    </form>

    <?php if ($filter_class): ?>
        <div class="students-table-container">
            <div class="table-header">
                <h3>Students in Class: <?php echo htmlspecialchars($filter_class); ?></h3>
                <a href="../admin/print_students.php?class=<?php echo urlencode($filter_class); ?>" class="btn-print" target="_blank">
                    <i class="fas fa-print"></i> Print Students
                </a>
            </div>
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Username</th>
                        <th>Name</th>
                        <th>Class</th>
                        <th>Academic Year</th>
                        <th>Phone</th>
                        <th>Parent Name</th>
                        <th>Parent Phone</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($students as $student): ?>
                        <tr>
                            <td><?php echo $student['id']; ?></td>
                            <td><?php echo htmlspecialchars($student['username']); ?></td>
                            <td><?php echo htmlspecialchars($student['first_name'] . ' ' . ($student['middle_name'] ? $student['middle_name'] . ' ' : '') . $student['last_name']); ?></td>
                            <td><?php echo htmlspecialchars($student['class']); ?></td>
                            <td><?php echo htmlspecialchars($student['academic_year']); ?></td>
                            <td><?php echo htmlspecialchars($student['phone']); ?></td>
                            <td><?php echo htmlspecialchars($student['parent_name'] ?? 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($student['parent_phone'] ?? 'N/A'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php else: ?>
        <div class="classes-list">
            <h3>Your Assigned Classes</h3>
            <ul>
                <?php foreach ($classes as $class): ?>
                    <li>
                        <span><?php echo htmlspecialchars($class['name']); ?></span>
                        <a href="view_students.php?class=<?php echo urlencode($class['name']); ?>">View Students</a>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>
</div>




<?php
if ($role == 'admin') {
    include '../includes/admin_footer.php';
} else {
    include '../includes/teacher_footer.php';
}
?>
