<?php
require_once '../config.php';

if (!in_array($_SESSION['role'], ['admin', 'teacher'])) {
    header('Location: ../login.php');
    exit;
}

$student_id = $_GET['student_id'] ?? null;
$class_name = $_GET['class'] ?? null;
$academic_year_id = $_GET['academic_year'] ?? null;
$term_id = $_GET['term'] ?? null;

if (!$student_id || !$class_name || !$academic_year_id || !$term_id) {
    die("Missing required parameters.");
}

$stmt = $pdo->prepare("SELECT CONCAT(last_name, ', ', first_name, ' ', COALESCE(middle_name, '')) AS name FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch();
if (!$student) {
    die("Student not found.");
}
$student_name = trim($student['name']);

$stmt = $pdo->prepare("SELECT id FROM classes WHERE name = ?");
$stmt->execute([$class_name]);
$class_info = $stmt->fetch();
if (!$class_info) {
    die("Class not found.");
}
$class_id = $class_info['id'];

$stmt = $pdo->prepare("
    SELECT e.id AS enrollment_id, c.id AS course_id, c.code, c.name AS course_name
    FROM enrollments e
    JOIN courses c ON e.course_id = c.id
    JOIN class_courses cc ON cc.course_id = c.id AND cc.class_id = ?
    WHERE e.student_id = ? AND e.academic_year_id = ? AND e.term_id = ?
    ORDER BY c.name
");
$stmt->execute([$class_id, $student_id, $academic_year_id, $term_id]);
$enrollments = $stmt->fetchAll();

if (empty($enrollments)) {
    die("No enrollments found for this student.");
}

// Get all mock exam data for each enrollment
$mock_exams_data = [];
foreach ($enrollments as $enrollment) {
    $stmt = $pdo->prepare("SELECT mock_exam_number, score, grade, comments FROM mock_exams WHERE enrollment_id = ? AND term_id = ? AND academic_year_id = ? ORDER BY mock_exam_number");
    $stmt->execute([$enrollment['enrollment_id'], $term_id, $academic_year_id]);
    $mock_exams = $stmt->fetchAll();
    
    $mock_exams_data[$enrollment['course_id']] = [
        'course_name' => $enrollment['course_name'],
        'course_code' => $enrollment['code'],
        'mock_exams' => $mock_exams
    ];
}

$role = $_SESSION['role'];

if ($role == 'admin') {
    include '../includes/admin_header.php';
} else {
    include '../includes/teacher_header.php';
}
?>

<style>
    .view-mock-exams-container {
        padding: 20px;
        background-color: #f8f9fa;
        min-height: 100vh;
    }

    .page-header {
        background: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%);
        color: white;
        padding: 30px;
        border-radius: 15px;
        margin-bottom: 30px;
        text-align: center;
        box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    }

    .page-header h2 {
        margin: 0;
        font-size: 2.5em;
        font-weight: 700;
    }

    .results-container {
        background: white;
        border-radius: 15px;
        padding: 25px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
    }

    .results-container h3 {
        color: #2c3e50;
        margin-bottom: 20px;
        font-size: 1.8em;
    }

    .back-link {
        display: inline-block;
        padding: 10px 20px;
        background: #6c757d;
        color: white;
        text-decoration: none;
        border-radius: 8px;
        font-weight: 600;
        margin-bottom: 20px;
        transition: background-color 0.3s ease;
    }

    .back-link:hover {
        background: #5a6268;
        color: white;
        text-decoration: none;
    }

    .results-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        border-radius: 8px;
        overflow: hidden;
    }

    .results-table th,
    .results-table td {
        padding: 12px 15px;
        text-align: left;
        border-bottom: 1px solid #ddd;
    }

    .results-table th {
        background: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%);
        color: white;
        font-weight: 600;
        font-size: 0.9em;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .results-table tr:nth-child(even) {
        background-color: #f8f9fa;
    }

    .results-table tr:hover {
        background-color: #f3e5f5;
    }

    .results-table .course-name {
        font-weight: 600;
        color: #2c3e50;
        min-width: 200px;
    }

    .no-data {
        color: #999;
        font-style: italic;
    }

    .grade-cell {
        font-weight: 600;
    }
</style>

<div class="view-mock-exams-container">
    <div class="page-header">
        <h2>📊 Mock Exam Results</h2>
    </div>

    <div class="results-container">
        <a href="final_year_grades.php?class=<?php echo urlencode($class_name); ?>&academic_year=<?php echo $academic_year_id; ?>&term=<?php echo $term_id; ?>" class="back-link">← Back to Students</a>
        <a href="print_mock.php?student_id=<?php echo $student_id; ?>&class=<?php echo urlencode($class_name); ?>&academic_year=<?php echo $academic_year_id; ?>&term=<?php echo $term_id; ?>" target="_blank" class="back-link" style="background: #28a745; margin-left: 10px;">🖨️ Print Report</a>

        <h3>Mock Exam Results for <?php echo htmlspecialchars($student_name); ?> in <?php echo htmlspecialchars($class_name); ?></h3>

        <table class="results-table">
            <thead>
                <tr>
                    <th>Course</th>
                    <?php for ($i = 1; $i <= 10; $i++): ?>
                        <th>Mock #<?php echo $i; ?></th>
                    <?php endfor; ?>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($mock_exams_data as $course_data): ?>
                    <tr>
                        <td class="course-name"><?php echo htmlspecialchars($course_data['course_name']); ?></td>
                        <?php 
                        $mock_exams_by_number = [];
                        foreach ($course_data['mock_exams'] as $mock) {
                            $mock_exams_by_number[$mock['mock_exam_number']] = $mock;
                        }
                        
                        for ($i = 1; $i <= 10; $i++): 
                            if (isset($mock_exams_by_number[$i])): 
                                $mock = $mock_exams_by_number[$i];
                        ?>
                            <td class="grade-cell" title="<?php echo htmlspecialchars($mock['comments'] ?? ''); ?>">
                                <?php echo htmlspecialchars($mock['score'] ?? '-'); ?> (<?php echo htmlspecialchars($mock['grade'] ?? '-'); ?>)
                            </td>
                        <?php else: ?>
                            <td class="no-data">-</td>
                        <?php endif; endfor; ?>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<?php
if ($role == 'admin') {
    include '../includes/admin_footer.php';
} else {
    include '../includes/teacher_footer.php';
}
?>
