<?php
require_once '../config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'teacher') {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Get teacher id from staff table (since teachers are now in staff table)
$stmt = $pdo->prepare("SELECT id FROM staff WHERE user_id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher not found.");
}

$teacher_id = $teacher['id'];

try {
    // Fetch classes assigned to the teacher
    $classes_stmt = $pdo->prepare("SELECT DISTINCT c.id, c.name FROM classes c JOIN teacher_classes tc ON c.id = tc.class_id WHERE tc.teacher_id = ?");
    $classes_stmt->execute([$teacher_id]);
    $classes = $classes_stmt->fetchAll();
} catch (PDOException $e) {
    error_log("Error fetching classes for teacher $teacher_id: " . $e->getMessage());
    $classes = [];
}

$selected_class_id = $_GET['class_id'] ?? null;
$selected_date = $_GET['date'] ?? null;

$attendance_records = [];

if ($selected_class_id && $selected_date) {
    // Get the class name
    $class_stmt = $pdo->prepare("SELECT name FROM classes WHERE id = ?");
    $class_stmt->execute([$selected_class_id]);
    $class = $class_stmt->fetch();
    $class_name = $class['name'];

    // Fetch all students in the class and their attendance status for the selected date
    $attendance_stmt = $pdo->prepare("
        SELECT s.id, CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as student_name, a.status
        FROM students s
        LEFT JOIN attendance a ON a.student_id = s.id AND a.class_id = ? AND a.date = ? AND (a.marked_by = ? OR a.marked_by IS NULL)
        WHERE s.class = ?
        ORDER BY CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) ASC
    ");
    $attendance_stmt->execute([$selected_class_id, $selected_date, $teacher_id, $class_name]);
    $attendance_records = $attendance_stmt->fetchAll();
}

include '../includes/teacher_header.php';
?>

<style>
    .view-attendance-container {
        padding: 20px;
        background-color: #f8f9fa;
        min-height: 100vh;
    }

    .page-header {
        background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
        color: white;
        padding: 30px;
        border-radius: 15px;
        margin-bottom: 30px;
        text-align: center;
        box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    }

    .page-header h2 {
        margin: 0;
        font-size: 2.5em;
        font-weight: 700;
    }

    .filter-form {
        background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
        padding: 30px;
        border-radius: 20px;
        box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        margin-bottom: 30px;
        display: flex;
        flex-wrap: wrap;
        gap: 40px;
        align-items: center;
        justify-content: center;
        position: relative;
        overflow: hidden;
    }

    .filter-form::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, #3498db, #2980b9, #27ae60);
    }

    .filter-group {
        display: flex;
        flex-direction: column;
        gap: 10px;
        min-width: 240px;
        position: relative;
        z-index: 1;
    }

    .filter-form label {
        font-weight: 700;
        color: #2c3e50;
        font-size: 1em;
        display: flex;
        align-items: center;
        gap: 10px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .filter-form label::before {
        content: '📅';
        font-size: 1.2em;
        opacity: 0.8;
    }

    .filter-form label[for="class_id"]::before {
        content: '🏫';
    }

    .filter-form select,
    .filter-form input[type="date"] {
        padding: 14px 18px;
        border: 2px solid #e1e8ed;
        border-radius: 12px;
        font-size: 1em;
        background-color: #fff;
        transition: all 0.3s ease;
        box-shadow: 0 3px 6px rgba(0,0,0,0.08);
        position: relative;
    }

    .filter-form select:focus,
    .filter-form input[type="date"]:focus {
        outline: none;
        border-color: #3498db;
        box-shadow: 0 0 0 4px rgba(52, 152, 219, 0.15), 0 4px 8px rgba(0,0,0,0.1);
        transform: translateY(-2px);
        background-color: #f8f9ff;
    }

    .filter-form select:hover,
    .filter-form input[type="date"]:hover {
        border-color: #bdc3c7;
        box-shadow: 0 4px 8px rgba(0,0,0,0.1);
    }

    .filter-form select {
        cursor: pointer;
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236b7280' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
        background-position: right 0.5rem center;
        background-repeat: no-repeat;
        background-size: 1.5em 1.5em;
        padding-right: 2.5rem;
    }

    .attendance-table-container {
        background: white;
        border-radius: 15px;
        padding: 25px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
        overflow-x: auto;
    }

    .attendance-table-container h3 {
        color: #2c3e50;
        margin-bottom: 20px;
        font-size: 1.8em;
    }

    .table-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
    }

    .table-header h3 {
        margin: 0;
        color: #2c3e50;
        font-size: 1.5em;
        font-weight: 600;
    }

    .btn-print {
        background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
        color: white;
        padding: 10px 15px;
        text-decoration: none;
        border-radius: 5px;
        font-weight: 600;
        transition: all 0.3s ease;
        display: inline-flex;
        align-items: center;
        gap: 5px;
    }

    .btn-print:hover {
        background: linear-gradient(135deg, #34495e 0%, #2c3e50 100%);
        text-decoration: none;
        color: white;
    }

    table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    th, td {
        padding: 15px;
        text-align: left;
        border-bottom: 1px solid #ddd;
    }

    th {
        background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
        color: white;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    tr:nth-child(even) {
        background-color: #f8f9fa;
    }

    tr:hover {
        background-color: #e8f5e8;
    }

    @media (max-width: 768px) {
        .filter-form {
            flex-direction: column;
            align-items: stretch;
        }

        .filter-form label {
            margin-bottom: 5px;
        }

        table {
            font-size: 0.9em;
        }

        th, td {
            padding: 10px;
        }
    }
</style>

<div class="view-attendance-container">
    <div class="page-header">
        <h2>📋 View Attendance Records</h2>
    </div>

    <form method="GET" class="filter-form">
        <div class="filter-group">
            <label for="class_id">Select Class:</label>
            <select name="class_id" id="class_id" required onchange="this.form.submit()">
                <option value="">-- Select Class --</option>
                <?php foreach ($classes as $class): ?>
                    <option value="<?php echo $class['id']; ?>" <?php if ($class['id'] == $selected_class_id) echo 'selected'; ?>>
                        <?php echo htmlspecialchars($class['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="filter-group">
            <label for="date">Select Date:</label>
            <input type="date" name="date" id="date" value="<?php echo htmlspecialchars($selected_date); ?>" required onchange="this.form.submit()">
        </div>
    </form>

    <?php if ($selected_class_id && $selected_date): ?>
        <?php if (count($attendance_records) > 0): ?>
            <div class="attendance-table-container">
                <div class="table-header">
                    <h3>Attendance for Class: <?php echo htmlspecialchars($class_name); ?> on <?php echo htmlspecialchars(date('F j, Y', strtotime($selected_date))); ?></h3>
                    <a href="../admin/print_attendance.php?class_id=<?php echo urlencode($selected_class_id); ?>&date=<?php echo urlencode($selected_date); ?>" class="btn-print" target="_blank">
                        <i class="fas fa-print"></i> Print Attendance
                    </a>
                </div>
                <table>
                    <thead>
                        <tr>
                            <th>Student Name</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($attendance_records as $record): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($record['student_name']); ?></td>
                                <td><?php echo htmlspecialchars($record['status'] ? ucfirst($record['status']) : 'Not Marked'); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="attendance-table-container">
                <p>No attendance records found for the selected class and date.</p>
            </div>
        <?php endif; ?>
    <?php endif; ?>
</div>

<?php
include '../includes/teacher_footer.php';
?>
