<?php
require_once '../config.php';

if (!isset($_SESSION['role']) || $_SESSION['role'] != 'teacher') {
    header('Location: ../login.php');
    exit;
}

// Get teacher's ID
$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT id, CONCAT(first_name, ' ', last_name) as name FROM staff WHERE id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher record not found. Please contact your administrator.");
}

$teacher_id = $teacher['id'];
$teacher_name = $teacher['name'];

// Debug output
echo "<!-- Debug: Teacher ID: $teacher_id, Name: $teacher_name -->\n";

// Get filter values
$selected_class = $_GET['class'] ?? null;
$selected_term_id = $_GET['term_id'] ?? null;
$selected_academic_year_id = $_GET['academic_year_id'] ?? null;
$selected_student = $_GET['student'] ?? null;

// Get classes where this teacher has enrolled students in their courses
$stmt = $pdo->prepare("
    SELECT DISTINCT s.class
    FROM students s
    JOIN enrollments e ON s.id = e.student_id
    JOIN courses c ON e.course_id = c.id
    JOIN teacher_courses tc ON c.id = tc.course_id
    WHERE tc.teacher_id = ?
    ORDER BY s.class
");
$stmt->execute([$teacher_id]);
$teacher_classes = $stmt->fetchAll(PDO::FETCH_COLUMN);

// Get all terms for selection
$stmt = $pdo->query("SELECT id, name FROM terms ORDER BY start_date DESC");
$active_terms = $stmt->fetchAll();

// Get all academic years for selection
$stmt = $pdo->query("SELECT id, year FROM academic_years ORDER BY start_date DESC");
$active_academic_years = $stmt->fetchAll();

// Get detailed student results with filters
// Use INNER JOIN for grades when no academic year filter is applied to only show courses with results
$join_type = empty($selected_academic_year_id) ? 'INNER' : 'LEFT';

$query = "
        SELECT * FROM (
        SELECT
            s.id as student_id,
            CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) COLLATE utf8mb4_unicode_ci as student_name,
            s.class COLLATE utf8mb4_unicode_ci as student_class,
            c.name COLLATE utf8mb4_unicode_ci as course_name,
            g.converted_class_mark,
            g.converted_exam_mark,
            g.converted_final_mark,
            g.assessment_total,
            g.exam_score,
            g.final_score,
            g.comments COLLATE utf8mb4_unicode_ci as comments,
            g.graded_at,
            t.name COLLATE utf8mb4_unicode_ci as term_name,
            ay.year COLLATE utf8mb4_unicode_ci as academic_year,
            sub.name COLLATE utf8mb4_unicode_ci as subject_name,
            sub.code COLLATE utf8mb4_unicode_ci as subject_code,
            'regular' COLLATE utf8mb4_unicode_ci as grade_type,
            CAST(g.grade AS CHAR) COLLATE utf8mb4_unicode_ci as grade_value
        FROM students s
        JOIN enrollments e ON s.id = e.student_id
        JOIN courses c ON e.course_id = c.id
        JOIN teacher_courses tc ON c.id = tc.course_id
        JOIN classes cl ON s.class = cl.name
        JOIN levels lv ON cl.level_id = lv.id
        {$join_type} JOIN grades g ON e.id = g.enrollment_id
        LEFT JOIN terms t ON g.term_id = t.id
        LEFT JOIN academic_years ay ON g.academic_year_id = ay.id
        LEFT JOIN subjects sub ON c.subject_id = sub.id
        WHERE tc.teacher_id = ?
";

$params = [$teacher_id];

if ($selected_class) {
    $query .= " AND s.class = ?";
    $params[] = $selected_class;
}
if ($selected_term_id) {
    $query .= " AND g.term_id = ?";
    $params[] = $selected_term_id;
}
if ($selected_academic_year_id) {
    $query .= " AND g.academic_year_id = ?";
    $params[] = $selected_academic_year_id;
}
if ($selected_student) {
    $query .= " AND s.id = ?";
    $params[] = $selected_student;
}

$query .= "

        UNION ALL

        SELECT
            s.id as student_id,
            CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) COLLATE utf8mb4_unicode_ci as student_name,
            s.class COLLATE utf8mb4_unicode_ci as student_class,
            c.name COLLATE utf8mb4_unicode_ci as course_name,
            (pg.exercise1 + pg.homework1) as converted_class_mark,
            (pg.exam_score * 0.5) as converted_exam_mark,
            ((pg.exercise1 + pg.homework1) + (pg.exam_score * 0.5)) as converted_final_mark,
            NULL as assessment_total,
            pg.exam_score,
            pg.final_score,
            pg.comments COLLATE utf8mb4_unicode_ci as comments,
            pg.created_at as graded_at,
            t.name COLLATE utf8mb4_unicode_ci as term_name,
            ay.year COLLATE utf8mb4_unicode_ci as academic_year,
            sub.name COLLATE utf8mb4_unicode_ci as subject_name,
            sub.code COLLATE utf8mb4_unicode_ci as subject_code,
            'preschool' COLLATE utf8mb4_unicode_ci as grade_type,
            CAST(pg.grade AS CHAR) COLLATE utf8mb4_unicode_ci as grade_value
        FROM students s
        JOIN enrollments e ON s.id = e.student_id
        JOIN courses c ON e.course_id = c.id
        JOIN teacher_courses tc ON c.id = tc.course_id
        JOIN classes cl ON s.class = cl.name
        JOIN levels lv ON cl.level_id = lv.id
        LEFT JOIN preschool_grades pg ON e.id = pg.enrollment_id
        LEFT JOIN terms t ON pg.term_id = t.id
        LEFT JOIN academic_years ay ON pg.academic_year_id = ay.id
        LEFT JOIN subjects sub ON c.subject_id = sub.id
        WHERE tc.teacher_id = ? AND lv.id = 3
";

$params[] = $teacher_id;

if ($selected_class) {
    $query .= " AND s.class = ?";
    $params[] = $selected_class;
}
if ($selected_term_id) {
    $query .= " AND pg.term_id = ?";
    $params[] = $selected_term_id;
}
if ($selected_academic_year_id) {
    $query .= " AND pg.academic_year_id = ?";
    $params[] = $selected_academic_year_id;
}
if ($selected_student) {
    $query .= " AND s.id = ?";
    $params[] = $selected_student;
}

$query .= "
    ) AS combined_results
    ORDER BY student_class, student_name, academic_year DESC, term_name DESC, graded_at DESC
";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$results_data = $stmt->fetchAll();

// Debug output
echo "<!-- Debug: Number of results: " . count($results_data) . " -->\n";
if (count($results_data) > 0) {
    echo "<!-- Debug: Sample result: " . json_encode($results_data[0]) . " -->\n";
}

// Fetch students for filter dropdown (only those taught by this teacher)
$stmt = $pdo->prepare("SELECT DISTINCT s.id, CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as name FROM students s JOIN enrollments e ON s.id = e.student_id JOIN courses c ON e.course_id = c.id JOIN teacher_courses tc ON c.id = tc.course_id WHERE tc.teacher_id = ? ORDER BY s.first_name, s.last_name");
$stmt->execute([$teacher_id]);
$teacher_students = $stmt->fetchAll();

// Check if any filters are applied
$filters_applied = !empty($selected_class) || !empty($selected_term_id) || !empty($selected_academic_year_id) || !empty($selected_student);

    include '../includes/teacher_header.php';
?>

<style>
    .results-container {
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    .page-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 30px;
        border-radius: 15px;
        margin-bottom: 30px;
        text-align: center;
        box-shadow: 0 10px 30px rgba(102, 126, 234, 0.3);
        position: relative;
        overflow: hidden;
    }

    .page-header h2 {
        margin: 0;
        font-size: 2.5em;
        font-weight: 700;
        position: relative;
        z-index: 1;
    }

    .filters-section {
        background: white;
        padding: 30px;
        border-radius: 15px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        margin-bottom: 30px;
        border: 1px solid #e9ecef;
    }

    .filters-section h3 {
        margin: 0 0 25px 0;
        color: #333;
        font-size: 1.5em;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .filter-form {
        display: flex;
        flex-wrap: wrap;
        gap: 25px;
        align-items: end;
    }

    .filter-group {
        display: flex;
        flex-direction: column;
        min-width: 220px;
        flex: 1;
    }

    .filter-group label {
        font-weight: 600;
        color: #555;
        margin-bottom: 8px;
        font-size: 0.95em;
    }

    .filter-group select {
        padding: 12px 16px;
        border: 2px solid #e9ecef;
        border-radius: 10px;
        font-size: 1em;
        background: white;
        transition: all 0.3s ease;
        cursor: pointer;
    }

    .filter-group select:focus {
        outline: none;
        border-color: #667eea;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        transform: translateY(-1px);
    }

    .filter-btn {
        padding: 14px 30px;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border: none;
        border-radius: 10px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        display: flex;
        align-items: center;
        gap: 8px;
        font-size: 1em;
        min-height: 48px;
    }

    .filter-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
    }

    .filter-btn:active {
        transform: translateY(0);
    }

    .results-table-container {
        background: white;
        border-radius: 15px;
        overflow: hidden;
        box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        border: 1px solid #e9ecef;
    }

    .results-table-container h3 {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        margin: 0;
        padding: 25px 30px;
        font-size: 1.5em;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    table {
        width: 100%;
        border-collapse: collapse;
        font-size: 0.95em;
    }

    th, td {
        padding: 18px 20px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    th {
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        font-weight: 700;
        color: #333;
        text-transform: uppercase;
        font-size: 0.8em;
        letter-spacing: 1px;
        position: sticky;
        top: 0;
        z-index: 10;
    }

    tr:nth-child(even) {
        background: #fafbfc;
    }

    tr:hover {
        background: linear-gradient(135deg, #e3f2fd 0%, #f3e5f5 100%);
        transition: background-color 0.3s ease;
        transform: scale(1.01);
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    }

    .student-name {
        font-weight: 600;
        color: #333;
        font-size: 1.05em;
    }

    .score-display {
        font-weight: 700;
        font-size: 1.1em;
        color: #333;
    }

    .no-data {
        text-align: center;
        padding: 80px 20px;
        color: #666;
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    }

    .no-data i {
        font-size: 5em;
        color: #ccc;
        margin-bottom: 25px;
        display: block;
        opacity: 0.7;
    }

    .no-data p {
        margin: 10px 0;
        font-size: 1.1em;
    }

    .no-data p:first-of-type {
        font-weight: 600;
        color: #333;
    }

    .no-grades {
        background: white;
        border-radius: 15px;
        padding: 60px 30px;
        text-align: center;
        box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        border: 1px solid #e9ecef;
        margin-bottom: 30px;
    }

    .no-grades-icon {
        font-size: 4em;
        color: #ccc;
        margin-bottom: 20px;
        display: block;
    }

    .no-grades h3 {
        color: #666;
        margin: 0 0 15px 0;
        font-size: 1.5em;
        font-weight: 600;
    }

    .no-grades p {
        color: #888;
        margin: 0;
        font-size: 1.1em;
    }

    .student-results-section {
        background: white;
        border-radius: 15px;
        margin-bottom: 30px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        border: 1px solid #e9ecef;
        overflow: hidden;
    }

    .student-results-section h3 {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        margin: 0;
        padding: 20px 30px;
        font-size: 1.3em;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .grades-table-container {
        overflow-x: auto;
    }

    .grades-table {
        width: 100%;
        border-collapse: collapse;
        font-size: 0.95em;
    }

    .grades-table th,
    .grades-table td {
        padding: 15px 20px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .grades-table th {
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        font-weight: 700;
        color: #333;
        text-transform: uppercase;
        font-size: 0.8em;
        letter-spacing: 1px;
        position: sticky;
        top: 0;
        z-index: 10;
    }

    .grades-table tr:nth-child(even) {
        background: #fafbfc;
    }

    .grades-table tr:hover {
        background: linear-gradient(135deg, #e3f2fd 0%, #f3e5f5 100%);
        transition: background-color 0.3s ease;
    }

    .subject-info {
        line-height: 1.4;
    }

    .subject-info small {
        display: block;
        color: #666;
        font-size: 0.85em;
        font-weight: normal;
    }

    .print-btn {
        background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
        color: white;
        border: none;
        padding: 10px 20px;
        border-radius: 8px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        margin-left: 20px;
        font-size: 0.9em;
        display: inline-flex;
        align-items: center;
        gap: 5px;
    }

    .print-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(40, 167, 69, 0.3);
    }

    .print-btn:active {
        transform: translateY(0);
    }

    /* Responsive Design */
    @media (max-width: 1024px) {
        .filter-form {
            gap: 20px;
        }

        .filter-group {
            min-width: 180px;
        }
    }

    @media (max-width: 768px) {
        .results-container {
            padding: 15px;
        }

        .page-header {
            padding: 25px 20px;
            margin-bottom: 20px;
        }

        .page-header h2 {
            font-size: 2em;
        }

        .page-header p {
            font-size: 1em;
        }

        .filter-form {
            flex-direction: column;
            align-items: stretch;
            gap: 20px;
        }

        .filter-group {
            min-width: auto;
        }

        .filter-btn {
            width: 100%;
            justify-content: center;
        }

        .results-table-container h3 {
            padding: 20px;
            font-size: 1.3em;
        }

        .student-results-section h3 {
            padding: 18px 20px;
            font-size: 1.2em;
        }

        .print-btn {
            margin-left: 10px;
            padding: 8px 16px;
            font-size: 0.8em;
        }

        .grades-table-container {
            overflow-x: auto;
        }

        .grades-table {
            font-size: 0.9em;
            min-width: 600px;
        }

        .grades-table th,
        .grades-table td {
            padding: 12px 15px;
        }

        table {
            font-size: 0.9em;
        }

        th, td {
            padding: 12px 15px;
        }

        .no-grades {
            padding: 40px 20px;
        }

        .no-grades-icon {
            font-size: 3em;
        }

        .no-grades h3 {
            font-size: 1.3em;
        }
    }

    @media (max-width: 480px) {
        .results-container {
            padding: 10px;
        }

        .page-header {
            padding: 20px 15px;
            border-radius: 10px;
        }

        .page-header h2 {
            font-size: 1.8em;
        }

        .filter-form {
            gap: 12px;
        }

        .results-table-container h3 {
            padding: 18px 15px;
            font-size: 1.2em;
        }

        .student-results-section h3 {
            padding: 15px 18px;
            font-size: 1.1em;
        }

        .grades-table {
            font-size: 0.8em;
            min-width: 500px;
        }

        .grades-table th,
        .grades-table td {
            padding: 8px 10px;
        }

        table {
            font-size: 0.85em;
        }

        th, td {
            padding: 10px 12px;
        }

        .student-name {
            font-size: 1em;
        }

        .no-grades {
            padding: 30px 15px;
        }

        .no-grades-icon {
            font-size: 2.5em;
        }

        .no-grades h3 {
            font-size: 1.2em;
        }

        .subject-info small {
            font-size: 0.8em;
        }
    }
</style>

<div class="results-container">
    <div class="page-header">
        <h2>📋 Students Results</h2>
        <p>View detailed results of students per selected course, term, and academic year</p>
    </div>

    <!-- Filters Section -->
    <div class="filters-section">
        <h3>🔍 Filter Results</h3>
        <form method="get" action="students_results.php" class="filter-form">
            <div class="filter-group">
                <label for="class">Class:</label>
                <select id="class" name="class">
                    <?php foreach ($teacher_classes as $class): ?>
                        <option value="<?php echo htmlspecialchars($class); ?>" <?php echo ($selected_class == $class) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($class); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label for="term_id">Term/Semester:</label>
                <select id="term_id" name="term_id">
                    <option value="">All Terms</option>
                    <?php foreach ($active_terms as $term): ?>
                        <option value="<?php echo $term['id']; ?>" <?php echo ($selected_term_id == $term['id']) ? 'selected' : ''; ?>>
<?php echo htmlspecialchars($term['name'] ?? ''); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label for="academic_year_id">Academic Year:</label>
                <select id="academic_year_id" name="academic_year_id">
                    <option value="">All Academic Years</option>
                    <?php foreach ($active_academic_years as $academic_year): ?>
                        <option value="<?php echo $academic_year['id']; ?>" <?php echo ($selected_academic_year_id == $academic_year['id']) ? 'selected' : ''; ?>>
<?php echo htmlspecialchars($academic_year['year'] ?? ''); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label for="student">Student:</label>
                <select id="student" name="student">
                    <option value="">All Students</option>
                    <?php foreach ($teacher_students as $student): ?>
<option value="<?php echo htmlspecialchars($student['id'] ?? ''); ?>" <?php echo ($selected_student == ($student['id'] ?? '')) ? 'selected' : ''; ?>>
<?php echo htmlspecialchars($student['name'] ?? ''); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <button type="submit" class="filter-btn">🔍 Apply Filters</button>
        </form>
    </div>

    <!-- Results Display -->
    <?php if (!$filters_applied): ?>
        <div class="no-grades">
            <div class="no-grades-icon">🔍</div>
            <h3>Apply Filters to View Results</h3>
            <p>Please select the desired filters above and click "Apply Filters" to view student results.</p>
        </div>
    <?php elseif (empty($results_data)): ?>
        <div class="no-grades">
            <div class="no-grades-icon">📝</div>
            <h3>No Results Available</h3>
            <p>No student results found for the selected filters. Try adjusting your filters or ensure students have been graded.</p>
        </div>
    <?php else: ?>
            <?php
            // Group results by student
            $results_by_student = [];
            foreach ($results_data as $result) {
                $student_key = $result['student_id'] . ' - ' . $result['student_name'];
                $results_by_student[$student_key][] = $result;
            }

            foreach ($results_by_student as $student_key => $student_results): ?>
                <div class="student-results-section" id="student-results-<?php echo htmlspecialchars(explode(' - ', $student_key)[0]); ?>">
<h3><?php echo htmlspecialchars(explode(' - ', $student_key)[1] ?? ''); ?> - Class <?php echo htmlspecialchars($student_results[0]['student_class'] ?? ''); ?></h3>
                    <a href="print_report_card.php?student_id=<?php echo htmlspecialchars(explode(' - ', $student_key)[0]); ?>&class=<?php echo urlencode($selected_class); ?>&term_id=<?php echo urlencode($selected_term_id); ?>&academic_year_id=<?php echo urlencode($selected_academic_year_id); ?>&student=<?php echo urlencode($selected_student); ?>" target="_blank" class="print-btn">🖨️ Print Report Card</a>
                    <?php
                    // Group student results by term
                    $results_by_term = [];
                    foreach ($student_results as $result) {
                        $term_key = $result['term_name'] ?: 'No Term';
                        $results_by_term[$term_key][] = $result;
                    }

                    foreach ($results_by_term as $term_name => $term_results): ?>
                        <h4 style="color: #667eea; margin-top: 20px; margin-bottom: 10px; font-size: 1.2em;"><?php echo htmlspecialchars($term_name); ?></h4>
                        <div class="grades-table-container">
                            <table class="grades-table">
                                <thead>
                                    <tr>
                                        <th>Subject</th>
                                        <th>Assessment</th>
                                        <th>Exam</th>
                                        <th>Total</th>
                                        <th>Grade</th>
                                        <th>Comments</th>
                                        <th>Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($term_results as $result): ?>
                                        <tr>
                                            <td>
<div class="subject-info">
    <?php
        $course = trim((string)($result['course_name'] ?? ''));
        $subject = trim((string)($result['subject_name'] ?? ''));
        $code = trim((string)($result['subject_code'] ?? ''));

        if ($course && $subject) {
            echo htmlspecialchars($course . ' — ' . $subject);
        } elseif ($course) {
            echo htmlspecialchars($course);
        } elseif ($subject) {
            echo htmlspecialchars($subject);
        } else {
            echo 'N/A';
        }

        if ($code) {
            echo ' <small>' . htmlspecialchars($code) . '</small>';
        }
    ?>
</div>
                                            </td>
                                            <td><?php echo htmlspecialchars($result['converted_class_mark'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($result['converted_exam_mark'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($result['converted_final_mark'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($result['grade_value'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($result['comments'] ?? ''); ?></td>
                                            <td><?php echo $result['graded_at'] ? date('M d, Y', strtotime($result['graded_at'])) : 'N/A'; ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endforeach; ?>
    <?php endif; ?>
</div>

<script>
function printReportCard(studentId) {
    // Extract student results section content
    const content = document.getElementById('student-results-' + studentId).innerHTML;

    // Extract additional info from the page for school, class, term, academic year
    // Assuming these are available as selected filter values or can be passed in some way
    const classSelect = document.getElementById('class');
    const termSelect = document.getElementById('term_id');
    const academicYearSelect = document.getElementById('academic_year_id');

    const selectedClass = classSelect ? classSelect.options[classSelect.selectedIndex].text : 'N/A';
    const selectedTerm = termSelect ? termSelect.options[termSelect.selectedIndex].text : 'N/A';
    const selectedAcademicYear = academicYearSelect ? academicYearSelect.options[academicYearSelect.selectedIndex].text : 'N/A';

    const schoolName = "Professional School Name"; // Could be dynamic if available

    const printWindow = window.open('', '', 'width=900,height=700');
    printWindow.document.write(
        '<html>' +
        '<head>' +
            '<title>Student Report Card</title>' +
            '<style>' +
                'body {' +
                    'font-family: \'Segoe UI\', Tahoma, Geneva, Verdana, sans-serif;' +
                    'margin: 30px 40px;' +
                    'color: #333;' +
                    'background: #f9f9f9;' +
                '}' +
                '.report-container {' +
                    'max-width: 800px;' +
                    'margin: auto;' +
                    'background: white;' +
                    'padding: 30px 40px;' +
                    'box-shadow: 0 0 15px rgba(0,0,0,0.1);' +
                    'border-radius: 12px;' +
                '}' +
                '.header {' +
                    'text-align: center;' +
                    'margin-bottom: 30px;' +
                '}' +
                '.school-name {' +
                    'font-size: 2em;' +
                    'font-weight: 700;' +
                    'color: #5a2a83;' +
                    'margin-bottom: 5px;' +
                '}' +
                '.report-title {' +
                    'font-size: 1.5em;' +
                    'color: #667eea;' +
                    'font-weight: 600;' +
                    'margin-bottom: 20px;' +
                '}' +
                '.info-section {' +
                    'display: flex;' +
                    'justify-content: space-between;' +
                    'margin-bottom: 30px;' +
                    'font-size: 1em;' +
                    'color: #555;' +
                '}' +
                '.info-item {' +
                    'flex: 1;' +
                    'padding: 0 10px;' +
                '}' +
                'table {' +
                    'width: 100%;' +
                    'border-collapse: collapse;' +
                    'font-size: 0.95em;' +
                '}' +
                'th, td {' +
                    'border: 1px solid #ccc;' +
                    'padding: 12px 15px;' +
                    'text-align: left;' +
                '}' +
                'th {' +
                    'background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);' +
                    'color: white;' +
                    'font-weight: 700;' +
                    'text-transform: uppercase;' +
                    'letter-spacing: 1px;' +
                '}' +
                'tr:nth-child(even) {' +
                    'background: #f7f9fc;' +
                '}' +
                'tr:hover {' +
                    'background: #e1e7ff;' +
                    'transition: background-color 0.3s ease;' +
                '}' +
                '.comments-cell {' +
                    'max-width: 250px;' +
                    'white-space: pre-wrap;' +
                    'word-wrap: break-word;' +
                '}' +
                '.footer {' +
                    'margin-top: 40px;' +
                    'text-align: center;' +
                    'font-size: 0.9em;' +
                    'color: #888;' +
                '}' +
            '</style>' +
        '</head>' +
        '<body>' +
            '<div class="report-container">' +
                '<div class="header">' +
                    '<div class="school-name">' + schoolName + '</div>' +
                    '<div class="report-title">Student Report Card</div>' +
                '</div>' +
                '<div class="info-section">' +
                    '<div class="info-item"><strong>Class:</strong> ' + selectedClass + '</div>' +
                    '<div class="info-item"><strong>Term/Semester:</strong> ' + selectedTerm + '</div>' +
                    '<div class="info-item"><strong>Academic Year:</strong> ' + selectedAcademicYear + '</div>' +
                '</div>' +
                '<div id="print-content">' + content + '</div>' +
                '<div class="footer">' +
                    'Generated on ' + new Date().toLocaleDateString() +
                '</div>' +
            '</div>' +
        '</body>' +
        '</html>'
    );
    printWindow.document.close();
    printWindow.focus();
    printWindow.print();
    printWindow.close();
}

document.addEventListener('DOMContentLoaded', function() {
    // Auto-submit form when filters change
    const filters = document.querySelectorAll('select[name="class"], select[name="term_id"], select[name="academic_year_id"], select[name="student"]');
    filters.forEach(filter => {
        filter.addEventListener('change', function() {
            this.closest('form').submit();
        });
    });
});
</script>

<?php
include '../includes/footer.php';
?>
