<?php
require_once '../config.php';

if (!isset($_SESSION['role']) || $_SESSION['role'] != 'teacher') {
    header('Location: ../login.php');
    exit;
}

// Get teacher's ID
$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT id, CONCAT(first_name, ' ', last_name) as name FROM staff WHERE user_id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher record not found. Please contact your administrator.");
}

$teacher_id = $teacher['id'];
$teacher_name = $teacher['name'];

// Get filter values
$selected_class = $_GET['class'] ?? null;
$selected_course_id = $_GET['course_id'] ?? null;
$selected_term_id = $_GET['term_id'] ?? null;
$selected_academic_year_id = $_GET['academic_year_id'] ?? null;

// Pagination
$page = (int)($_GET['page'] ?? 1);
$limit = 50;
$offset = ($page - 1) * $limit;

// Get courses taught by this teacher
$stmt = $pdo->prepare("SELECT DISTINCT c.id, c.name FROM courses c INNER JOIN teacher_courses tc ON c.id = tc.course_id WHERE tc.teacher_id = ? ORDER BY c.name");
$stmt->execute([$teacher_id]);
$teacher_courses = $stmt->fetchAll();

// Get active terms for selection
$stmt = $pdo->query("SELECT id, name FROM terms WHERE is_active = TRUE ORDER BY start_date DESC");
$active_terms = $stmt->fetchAll();

// Get active academic years for selection
$stmt = $pdo->query("SELECT id, year FROM academic_years WHERE is_active = TRUE ORDER BY start_date DESC");
$active_academic_years = $stmt->fetchAll();

// Get classes taught by this teacher, filtered by selected term or active term
$term_filter = $selected_term_id;
if (!$term_filter) {
    $active_term = $pdo->query("SELECT id FROM terms WHERE is_active = TRUE LIMIT 1")->fetch();
    $term_filter = $active_term ? $active_term['id'] : null;
}

if ($term_filter) {
    $stmt = $pdo->prepare("
        SELECT DISTINCT c.name
        FROM classes c
        INNER JOIN class_courses cc ON c.id = cc.class_id
        INNER JOIN teacher_courses tc ON cc.course_id = tc.course_id
        INNER JOIN course_terms ct ON cc.course_id = ct.course_id
        WHERE tc.teacher_id = ? AND ct.term_id = ?
        ORDER BY c.name
    ");
    $stmt->execute([$teacher_id, $term_filter]);
} else {
    // If no active term, show no classes to avoid errors
    $teacher_classes = [];
}
$teacher_classes = $stmt->fetchAll(PDO::FETCH_COLUMN);

// Get student performance data with filters
$query = "
    SELECT
        s.id as student_id,
        CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as student_name,
        s.class as student_class,
        c.name as course_name,
        COUNT(g.id) as total_grades,
        COUNT(g.id) * 100 as total_expected_marks,
        SUM(
            CASE g.grade
                WHEN 'A+' THEN 95
                WHEN 'A' THEN 90
                WHEN 'A-' THEN 85
                WHEN 'B+' THEN 80
                WHEN 'B' THEN 75
                WHEN 'B-' THEN 70
                WHEN 'C+' THEN 65
                WHEN 'C' THEN 60
                WHEN 'C-' THEN 55
                WHEN 'D+' THEN 50
                WHEN 'D' THEN 45
                WHEN 'D-' THEN 40
                WHEN 'F' THEN 35
                ELSE 0
            END
        ) as total_marks_obtained,
        AVG(
            CASE g.grade
                WHEN 'A+' THEN 95
                WHEN 'A' THEN 90
                WHEN 'A-' THEN 85
                WHEN 'B+' THEN 80
                WHEN 'B' THEN 75
                WHEN 'B-' THEN 70
                WHEN 'C+' THEN 65
                WHEN 'C' THEN 60
                WHEN 'C-' THEN 55
                WHEN 'D+' THEN 50
                WHEN 'D' THEN 45
                WHEN 'D-' THEN 40
                WHEN 'F' THEN 35
                ELSE 0
            END
        ) as average_score,
        MAX(g.graded_at) as last_graded
    FROM students s
    JOIN enrollments e ON s.id = e.student_id
    JOIN courses c ON e.course_id = c.id
    JOIN teacher_courses tc ON c.id = tc.course_id
    LEFT JOIN grades g ON e.id = g.enrollment_id
    WHERE tc.teacher_id = ?
";

$params = [$teacher_id];

if ($selected_class) {
    $query .= " AND s.class = ?";
    $params[] = $selected_class;
}
if ($selected_course_id) {
    $query .= " AND c.id = ?";
    $params[] = $selected_course_id;
}
if ($selected_term_id) {
    $query .= " AND g.term_id = ?";
    $params[] = $selected_term_id;
}
if ($selected_academic_year_id) {
    $query .= " AND g.academic_year_id = ?";
    $params[] = $selected_academic_year_id;
}

$query .= "
    GROUP BY s.id, s.first_name, s.middle_name, s.last_name, s.class, c.id, c.name
    ORDER BY s.id, average_score DESC
";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$raw_data = $stmt->fetchAll();

// Group data by student
$students = [];
foreach ($raw_data as $row) {
    $sid = $row['student_id'];
    if (!isset($students[$sid])) {
        $students[$sid] = [
            'student_id' => $row['student_id'],
            'student_name' => $row['student_name'],
            'student_class' => $row['student_class'],
            'courses' => [],
            'total_marks_obtained' => 0,
            'total_expected_marks' => 0,
            'last_graded' => null,
            'average_score' => 0,
            'cumulative_average' => 0,
            'area_of_strength' => '',
            'area_of_weakness' => '',
            'performance_level' => ''
        ];
    }
    $students[$sid]['courses'][] = [
        'course_name' => $row['course_name'],
        'average_score' => $row['average_score'] ?? 0,
        'total_marks_obtained' => $row['total_marks_obtained'] ?? 0,
        'total_expected_marks' => $row['total_expected_marks'] ?? 0,
        'last_graded' => $row['last_graded']
    ];
    $students[$sid]['total_marks_obtained'] += $row['total_marks_obtained'] ?? 0;
    $students[$sid]['total_expected_marks'] += $row['total_expected_marks'] ?? 0;
    if ($row['last_graded'] && (!$students[$sid]['last_graded'] || $row['last_graded'] > $students[$sid]['last_graded'])) {
        $students[$sid]['last_graded'] = $row['last_graded'];
    }
}

// Calculate averages, strengths, weaknesses, and performance levels
foreach ($students as &$student) {
    $courses = $student['courses'];
    $scores = array_column($courses, 'average_score');
    $student['average_score'] = count($scores) > 0 ? array_sum($scores) / count($scores) : 0;
    $student['cumulative_average'] = $student['average_score'];

    if (count($courses) > 0) {
        $max_score = max($scores);
        $min_score = min($scores);
        $strength_courses = array_filter($courses, fn($c) => $c['average_score'] == $max_score);
        $weakness_courses = array_filter($courses, fn($c) => $c['average_score'] == $min_score);
        $student['area_of_strength'] = reset($strength_courses)['course_name'];
        $student['area_of_weakness'] = reset($weakness_courses)['course_name'];
    }

    $avg = $student['average_score'];
    if ($avg >= 90) $student['performance_level'] = 'Excellent';
    elseif ($avg >= 80) $student['performance_level'] = 'Very Good';
    elseif ($avg >= 70) $student['performance_level'] = 'Good';
    elseif ($avg >= 60) $student['performance_level'] = 'Satisfactory';
    else $student['performance_level'] = 'Needs Improvement';
}

$performance_data = array_values($students);

// Pagination
$total_students = count($performance_data);
$paginated_data = array_slice($performance_data, $offset, $limit);

// Calculate overall statistics
$excellent_count = 0;
$very_good_count = 0;
$good_count = 0;
$satisfactory_count = 0;
$needs_improvement_count = 0;

foreach ($performance_data as $student) {
    switch ($student['performance_level']) {
        case 'Excellent':
            $excellent_count++;
            break;
        case 'Very Good':
            $very_good_count++;
            break;
        case 'Good':
            $good_count++;
            break;
        case 'Satisfactory':
            $satisfactory_count++;
            break;
        case 'Needs Improvement':
            $needs_improvement_count++;
            break;
    }
}
?>
<?php include '../includes/teacher_header.php'; ?>

<style>
/* ===== MAIN CONTAINER ===== */
.performance-container {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    line-height: 1.6;
}

/* ===== PAGE HEADER ===== */
.page-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 30px;
    border-radius: 15px;
    margin-bottom: 30px;
    text-align: center;
    box-shadow: 0 10px 30px rgba(102, 126, 234, 0.3);
    position: relative;
    overflow: hidden;
}

.page-header::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="rgba(255,255,255,0.1)"/><circle cx="75" cy="75" r="1" fill="rgba(255,255,255,0.1)"/><circle cx="50" cy="10" r="0.5" fill="rgba(255,255,255,0.1)"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
    opacity: 0.1;
}

.page-header h2 {
    margin: 0;
    font-size: 2.5em;
    font-weight: 700;
    position: relative;
    z-index: 1;
}

.page-header p {
    margin: 10px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

/* ===== STATISTICS CARDS ===== */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.stat-card {
    background: white;
    padding: 25px;
    border-radius: 15px;
    text-align: center;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    transition: all 0.3s ease;
    border: 1px solid #e9ecef;
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #667eea, #764ba2);
    transform: scaleX(0);
    transition: transform 0.3s ease;
}

.stat-card:hover {
    transform: translateY(-8px);
    box-shadow: 0 12px 40px rgba(0,0,0,0.15);
}

.stat-card:hover::before {
    transform: scaleX(1);
}

.stat-card h4 {
    margin: 0 0 15px 0;
    color: #666;
    font-size: 0.9em;
    text-transform: uppercase;
    letter-spacing: 1px;
    font-weight: 600;
}

.stat-card .value {
    font-size: 2.8em;
    font-weight: 700;
    margin: 0;
    background: linear-gradient(135deg, #667eea, #764ba2);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

/* Performance level colors with better contrast */
.excellent { color: #28a745; }
.very-good { color: #20c997; }
.good { color: #ffc107; }
.satisfactory { color: #fd7e14; }
.needs-improvement { color: #dc3545; }

/* ===== FILTERS SECTION ===== */
.filters-section {
    background: white;
    padding: 30px;
    border-radius: 15px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    margin-bottom: 30px;
    border: 1px solid #e9ecef;
}

.filters-section h3 {
    margin: 0 0 25px 0;
    color: #333;
    font-size: 1.5em;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 10px;
}

.filter-form {
    display: flex;
    flex-wrap: wrap;
    gap: 25px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
    min-width: 220px;
    flex: 1;
}

.filter-group label {
    font-weight: 600;
    color: #555;
    margin-bottom: 8px;
    font-size: 0.95em;
}

.filter-group select {
    padding: 12px 16px;
    border: 2px solid #e9ecef;
    border-radius: 10px;
    font-size: 1em;
    background: white;
    transition: all 0.3s ease;
    cursor: pointer;
}

.filter-group select:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    transform: translateY(-1px);
}

.filter-btn {
    padding: 14px 30px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border: none;
    border-radius: 10px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 1em;
    min-height: 48px;
}

.filter-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
}

.filter-btn:active {
    transform: translateY(0);
}

/* ===== PERFORMANCE TABLE ===== */
.performance-table-container {
    background: white;
    border-radius: 15px;
    overflow: hidden;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    border: 1px solid #e9ecef;
}

.performance-table-container h3 {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    margin: 0;
    padding: 25px 30px;
    font-size: 1.5em;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 10px;
}

table {
    width: 100%;
    border-collapse: collapse;
    font-size: 0.95em;
}

th, td {
    padding: 18px 20px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

th {
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    font-weight: 700;
    color: #333;
    text-transform: uppercase;
    font-size: 0.8em;
    letter-spacing: 1px;
    position: sticky;
    top: 0;
    z-index: 10;
}

tr:nth-child(even) {
    background: #fafbfc;
}

tr:hover {
    background: linear-gradient(135deg, #e3f2fd 0%, #f3e5f5 100%);
    transition: background-color 0.3s ease;
    transform: scale(1.01);
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.student-name {
    font-weight: 600;
    color: #333;
    font-size: 1.05em;
}

.performance-badge {
    padding: 8px 16px;
    border-radius: 25px;
    font-weight: 600;
    font-size: 0.85em;
    text-align: center;
    display: inline-block;
    min-width: 120px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    transition: all 0.3s ease;
}

.performance-badge:hover {
    transform: scale(1.05);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.performance-badge.excellent {
    background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
    color: #155724;
    border: 1px solid #c3e6cb;
}

.performance-badge.very-good {
    background: linear-gradient(135deg, #d1ecf1 0%, #bee5eb 100%);
    color: #0c5460;
    border: 1px solid #bee5eb;
}

.performance-badge.good {
    background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
    color: #856404;
    border: 1px solid #ffeaa7;
}

.performance-badge.satisfactory {
    background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.performance-badge.needs-improvement {
    background: linear-gradient(135deg, #f5c6cb 0%, #f1aeb5 100%);
    color: #721c24;
    border: 1px solid #f1aeb5;
}

.score-display {
    font-weight: 700;
    font-size: 1.2em;
    color: #333;
}

/* ===== NO DATA STATE ===== */
.no-data {
    text-align: center;
    padding: 80px 20px;
    color: #666;
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
}

.no-data i {
    font-size: 5em;
    color: #ccc;
    margin-bottom: 25px;
    display: block;
    opacity: 0.7;
}

.no-data p {
    margin: 10px 0;
    font-size: 1.1em;
}

.no-data p:first-of-type {
    font-weight: 600;
    color: #333;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1024px) {
    .stats-grid {
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    }

    .filter-form {
        gap: 20px;
    }

    .filter-group {
        min-width: 180px;
    }
}

@media (max-width: 768px) {
    .performance-container {
        padding: 15px;
    }

    .page-header {
        padding: 25px 20px;
        margin-bottom: 20px;
    }

    .page-header h2 {
        font-size: 2em;
    }

    .page-header p {
        font-size: 1em;
    }

    .stats-grid {
        grid-template-columns: repeat(auto-fit, minmax(160px, 1fr));
        gap: 15px;
    }

    .stat-card {
        padding: 20px;
    }

    .stat-card .value {
        font-size: 2.2em;
    }

    .filters-section {
        padding: 25px 20px;
    }

    .filter-form {
        flex-direction: column;
        align-items: stretch;
        gap: 20px;
    }

    .filter-group {
        min-width: auto;
    }

    .filter-btn {
        width: 100%;
        justify-content: center;
    }

    .performance-table-container h3 {
        padding: 20px;
        font-size: 1.3em;
    }

    table {
        font-size: 0.9em;
    }

    th, td {
        padding: 12px 15px;
    }

    .performance-badge {
        min-width: 100px;
        font-size: 0.8em;
        padding: 6px 12px;
    }

    .score-display {
        font-size: 1.1em;
    }
}

@media (max-width: 480px) {
    .performance-container {
        padding: 10px;
    }

    .page-header {
        padding: 20px 15px;
        border-radius: 10px;
    }

    .page-header h2 {
        font-size: 1.8em;
    }

    .stats-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }

    .stat-card {
        padding: 18px;
        border-radius: 10px;
    }

    .stat-card .value {
        font-size: 2em;
    }

    .filters-section {
        padding: 20px 15px;
        border-radius: 10px;
    }

    .performance-table-container {
        border-radius: 10px;
    }

    .performance-table-container h3 {
        padding: 18px 15px;
        font-size: 1.2em;
    }

    table {
        font-size: 0.85em;
    }

    th, td {
        padding: 10px 12px;
    }

    .student-name {
        font-size: 1em;
    }

    .performance-badge {
        min-width: 90px;
        font-size: 0.75em;
        padding: 5px 10px;
    }

    .no-data {
        padding: 60px 15px;
    }

    .no-data i {
        font-size: 4em;
    }
}

/* ===== ACCESSIBILITY IMPROVEMENTS ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* Focus states for keyboard navigation */
.filter-group select:focus,
.filter-btn:focus {
    outline: 2px solid #667eea;
    outline-offset: 2px;
}

/* High contrast mode support */
@media (prefers-contrast: high) {
    .stat-card,
    .filters-section,
    .performance-table-container {
        border: 2px solid #000;
    }

    .performance-badge {
        border: 2px solid #000;
    }
}

/* ===== PRINT STYLES ===== */
@media print {
    .performance-container {
        background: white;
        padding: 0;
    }

    .page-header,
    .filters-section {
        background: white !important;
        color: black !important;
        box-shadow: none;
        border: 1px solid #000;
    }

    .stat-card,
    .performance-table-container {
        box-shadow: none;
        border: 1px solid #000;
    }

    .filter-btn {
        display: none;
    }

    .performance-badge {
        border: 1px solid #000;
        background: white !important;
        color: black !important;
    }
}
</style>

<div class="performance-container">
    <div class="page-header">
        <h2>📊 Student Performance Analytics</h2>
        <p>Track and analyze student performance across your courses</p>
    </div>

    <!-- Statistics Cards -->
    <div class="stats-grid">
        <div class="stat-card">
            <h4>Total Students</h4>
            <p class="value"><?php echo $total_students; ?></p>
        </div>
        <div class="stat-card">
            <h4 class="excellent">Excellent (90-100%)</h4>
            <p class="value excellent"><?php echo $excellent_count; ?></p>
        </div>
        <div class="stat-card">
            <h4 class="very-good">Very Good (80-89%)</h4>
            <p class="value very-good"><?php echo $very_good_count; ?></p>
        </div>
        <div class="stat-card">
            <h4 class="good">Good (70-79%)</h4>
            <p class="value good"><?php echo $good_count; ?></p>
        </div>
        <div class="stat-card">
            <h4 class="satisfactory">Satisfactory (60-69%)</h4>
            <p class="value satisfactory"><?php echo $satisfactory_count; ?></p>
        </div>
        <div class="stat-card">
            <h4 class="needs-improvement">Needs Improvement (<60%)</h4>
            <p class="value needs-improvement"><?php echo $needs_improvement_count; ?></p>
        </div>
    </div>

    <!-- Filters Section -->
    <div class="filters-section">
        <h3>🔍 Filter Performance Data</h3>
        <form method="get" action="students_performance.php" class="filter-form">
            <div class="filter-group">
                <label for="class">Class:</label>
                <select id="class" name="class">
                    <option value="">All Classes</option>
                    <?php foreach ($teacher_classes as $class): ?>
                        <option value="<?php echo htmlspecialchars($class); ?>" <?php echo ($selected_class == $class) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($class); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label for="course_id">Course:</label>
                <select id="course_id" name="course_id">
                    <option value="">All Courses</option>
                    <?php foreach ($teacher_courses as $course): ?>
                        <option value="<?php echo $course['id']; ?>" <?php echo ($selected_course_id == $course['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($course['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label for="term_id">Term/Semester:</label>
                <select id="term_id" name="term_id">
                    <option value="">All Terms</option>
                    <?php foreach ($active_terms as $term): ?>
                        <option value="<?php echo $term['id']; ?>" <?php echo ($selected_term_id == $term['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($term['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label for="academic_year_id">Academic Year:</label>
                <select id="academic_year_id" name="academic_year_id">
                    <option value="">All Academic Years</option>
                    <?php foreach ($active_academic_years as $academic_year): ?>
                        <option value="<?php echo $academic_year['id']; ?>" <?php echo ($selected_academic_year_id == $academic_year['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($academic_year['year']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <button type="submit" class="filter-btn">🔍 Apply Filters</button>
        </form>
    </div>

    <!-- Performance Table -->
    <div class="performance-table-container">
        <h3>📈 Student Performance Details</h3>
        <?php if (empty($performance_data)): ?>
            <div class="no-data">
                <i>📊</i>
                <p>No performance data found for the selected filters.</p>
                <p>Try adjusting your filters or ensure students have been graded.</p>
            </div>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>Student Name</th>
                        <th>Class</th>
                        <th>Total Expected Marks</th>
                        <th>Total Marks Obtained</th>
                        <th>Average Score</th>
                        <th>Cumulative Average</th>
                        <th>Area of Strength</th>
                        <th>Area of Weakness</th>
                        <th>Performance Level</th>
                        <th>Last Graded</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($performance_data as $student): ?>
                        <tr>
                            <td class="student-name"><?php echo htmlspecialchars($student['student_name']); ?></td>
                            <td><?php echo htmlspecialchars($student['student_class']); ?></td>
                            <td><?php echo htmlspecialchars($student['total_expected_marks']); ?></td>
                            <td><?php echo htmlspecialchars($student['total_marks_obtained'] ?? 0); ?></td>
                            <td class="score-display">
                                <?php echo $student['average_score'] ? number_format($student['average_score'], 1) . '%' : 'N/A'; ?>
                            </td>
                            <td class="score-display">
                                <?php echo $student['cumulative_average'] ? number_format($student['cumulative_average'], 1) . '%' : 'N/A'; ?>
                            </td>
                            <td><?php echo htmlspecialchars($student['area_of_strength'] ?? 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($student['area_of_weakness'] ?? 'N/A'); ?></td>
                            <td>
                                <span class="performance-badge <?php echo strtolower(str_replace(' ', '-', $student['performance_level'])); ?>">
                                    <?php echo htmlspecialchars($student['performance_level']); ?>
                                </span>
                            </td>
                            <td><?php echo htmlspecialchars($student['last_graded'] ?? 'Never'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>

<script>
// Add interactive features
document.addEventListener('DOMContentLoaded', function() {
    // Auto-submit form when filters change
    const filters = document.querySelectorAll('select[name="class"], select[name="course_id"], select[name="term_id"], select[name="academic_year_id"]');
    filters.forEach(filter => {
        filter.addEventListener('change', function() {
            this.closest('form').submit();
        });
    });

    // Add hover effects to performance badges
    const badges = document.querySelectorAll('.performance-badge');
    badges.forEach(badge => {
        badge.addEventListener('mouseenter', function() {
            this.style.transform = 'scale(1.05)';
            this.style.transition = 'transform 0.2s ease';
        });
        badge.addEventListener('mouseleave', function() {
            this.style.transform = 'scale(1)';
        });
    });
});
</script>

<?php
include '../includes/footer.php';
?>
