<?php
require_once '../config.php';

if (!isset($_SESSION['role']) || $_SESSION['role'] != 'teacher') {
    header('Location: ../login.php');
    exit;
}

// Get parameters
$student_id = $_GET['student_id'] ?? null;
$selected_class = $_GET['class'] ?? null;
$selected_term_id = $_GET['term_id'] ?? null;
$selected_academic_year_id = $_GET['academic_year_id'] ?? null;

if (!$student_id) {
    die("Student ID is required.");
}

// Fetch school settings
$school_name = 'Professional School Name';
$school_logo_url = '';
try {
    $stmt = $pdo->prepare("SELECT school_name, logo_path FROM school_settings LIMIT 1");
    $stmt->execute();
    $school_info = $stmt->fetch();
    if ($school_info) {
        $school_name = htmlspecialchars($school_info['school_name']);
        $logo_path = $school_info['logo_path'];
        if ($logo_path && !preg_match('/^https?:\/\//', $logo_path) && !preg_match('/^\//', $logo_path)) {
            $school_logo_url = '../' . $logo_path;
        } else {
            $school_logo_url = $logo_path;
        }
        $school_logo_url = htmlspecialchars($school_logo_url);
    }
} catch (Exception $e) {
    // Use default if error
}

// Get teacher's ID
$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT id FROM staff WHERE id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher record not found. Please contact your administrator.");
}

$teacher_id = $teacher['id'];

// Fetch student name
$stmt = $pdo->prepare("SELECT CONCAT(first_name, ' ', COALESCE(middle_name, ''), ' ', last_name) as name FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch();
if (!$student) {
    die("Student not found.");
}
$student_name = htmlspecialchars($student['name']);

// Get detailed student results with filters
$query = "
    SELECT * FROM (
        SELECT
            g.converted_class_mark,
            g.converted_exam_mark,
            g.converted_final_mark,
            g.comments COLLATE utf8mb4_unicode_ci as comments,
            g.graded_at,
            t.name COLLATE utf8mb4_unicode_ci as term_name,
            ay.year COLLATE utf8mb4_unicode_ci as academic_year,
            sub.name COLLATE utf8mb4_unicode_ci as subject_name,
            sub.code COLLATE utf8mb4_unicode_ci as subject_code,
            c.name COLLATE utf8mb4_unicode_ci as course_name,
            'regular' COLLATE utf8mb4_unicode_ci as grade_type,
            CAST(g.grade AS CHAR) COLLATE utf8mb4_unicode_ci as grade_value
        FROM enrollments e
        JOIN courses c ON e.course_id = c.id
        JOIN teacher_courses tc ON c.id = tc.course_id
        INNER JOIN grades g ON e.id = g.enrollment_id
        LEFT JOIN terms t ON g.term_id = t.id
        LEFT JOIN academic_years ay ON g.academic_year_id = ay.id
        LEFT JOIN subjects sub ON c.subject_id = sub.id
        WHERE e.student_id = ? AND tc.teacher_id = ?
";

$params = [$student_id, $teacher_id];

if ($selected_class) {
    $query .= " AND (SELECT class FROM students WHERE id = e.student_id) = ?";
    $params[] = $selected_class;
}
if ($selected_term_id) {
    $query .= " AND g.term_id = ?";
    $params[] = $selected_term_id;
}
if ($selected_academic_year_id) {
    $query .= " AND g.academic_year_id = ?";
    $params[] = $selected_academic_year_id;
}

$query .= "

        UNION ALL

        SELECT
            (pg.exercise1 + pg.homework1) as converted_class_mark,
            (pg.exam_score * 0.5) as converted_exam_mark,
            ((pg.exercise1 + pg.homework1) + (pg.exam_score * 0.5)) as converted_final_mark,
            pg.comments COLLATE utf8mb4_unicode_ci as comments,
            pg.created_at as graded_at,
            t.name COLLATE utf8mb4_unicode_ci as term_name,
            ay.year COLLATE utf8mb4_unicode_ci as academic_year,
            sub.name COLLATE utf8mb4_unicode_ci as subject_name,
            sub.code COLLATE utf8mb4_unicode_ci as subject_code,
            c.name COLLATE utf8mb4_unicode_ci as course_name,
            'preschool' COLLATE utf8mb4_unicode_ci as grade_type,
            CAST(pg.grade AS CHAR) COLLATE utf8mb4_unicode_ci as grade_value
        FROM enrollments e
        JOIN courses c ON e.course_id = c.id
        JOIN teacher_courses tc ON c.id = tc.course_id
        LEFT JOIN preschool_grades pg ON e.id = pg.enrollment_id
        LEFT JOIN terms t ON pg.term_id = t.id
        LEFT JOIN academic_years ay ON pg.academic_year_id = ay.id
        LEFT JOIN subjects sub ON c.subject_id = sub.id
        WHERE e.student_id = ? AND tc.teacher_id = ?
";

$params[] = $student_id;
$params[] = $teacher_id;

if ($selected_class) {
    $query .= " AND (SELECT class FROM students WHERE id = e.student_id) = ?";
    $params[] = $selected_class;
}
if ($selected_term_id) {
    $query .= " AND pg.term_id = ?";
    $params[] = $selected_term_id;
}
if ($selected_academic_year_id) {
    $query .= " AND pg.academic_year_id = ?";
    $params[] = $selected_academic_year_id;
}

$query .= "
    ) AS combined_results
    ORDER BY academic_year DESC, term_name DESC, graded_at DESC
";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$results = $stmt->fetchAll();

// Fetch conduct grades
$conduct_grades = [];
$stmt = $pdo->prepare("
    SELECT * FROM other_grades
    WHERE student_id = ?
    ORDER BY academic_year_id DESC, term_id DESC
");
$stmt->execute([$student_id]);
$conduct_results = $stmt->fetchAll();
foreach ($conduct_results as $cg) {
    $term_key = '';
    if ($cg['academic_year_id']) {
        $stmt_ay = $pdo->prepare("SELECT year FROM academic_years WHERE id = ?");
        $stmt_ay->execute([$cg['academic_year_id']]);
        $ay = $stmt_ay->fetch();
        $term_key .= $ay ? $ay['year'] . ' - ' : '';
    }
    if ($cg['term_id']) {
        $stmt_term = $pdo->prepare("SELECT name FROM terms WHERE id = ?");
        $stmt_term->execute([$cg['term_id']]);
        $term = $stmt_term->fetch();
        $term_key .= $term ? $term['name'] : 'No Term';
    } else {
        $term_key .= 'No Term';
    }
    $conduct_grades[$term_key] = $cg;
}

// Group results by academic year and term
$results_by_term = [];
foreach ($results as $result) {
    $term_key = ($result['academic_year'] ? $result['academic_year'] . ' - ' : '') . ($result['term_name'] ?: 'No Term');
    $results_by_term[$term_key][] = $result;
}

// Calculate metrics for each term
$term_metrics = [];
foreach ($results_by_term as $term_key => $term_results) {
    $total_obtained = 0;
    $subject_count = count($term_results);
    $total_expected = $subject_count * 100; // Assuming 100 max per subject
    $strengths = [];
    $weaknesses = [];

    foreach ($term_results as $result) {
        $final_mark = is_numeric($result['converted_final_mark']) ? (float)$result['converted_final_mark'] : 0;
        $total_obtained += $final_mark;

        $subject_name = $result['course_name'] ?: $result['subject_name'];
        if ($final_mark >= 80) {
            $strengths[] = $subject_name;
        } elseif ($final_mark < 50) {
            $weaknesses[] = $subject_name;
        }
    }

    $average = $subject_count > 0 ? round($total_obtained / $subject_count, 2) : 0;

    $term_metrics[$term_key] = [
        'total_expected' => $total_expected,
        'total_obtained' => $total_obtained,
        'average' => $average,
        'strengths' => $strengths,
        'weaknesses' => $weaknesses,
        'subject_count' => $subject_count
    ];
}

// Calculate accumulative average across all terms
$total_marks_all = 0;
$total_subjects_all = 0;
foreach ($term_metrics as $metrics) {
    $total_marks_all += $metrics['total_obtained'];
    $total_subjects_all += $metrics['subject_count'];
}
$accumulative_average = $total_subjects_all > 0 ? round($total_marks_all / $total_subjects_all, 2) : 0;

// Get class, term, academic year names for display
$class_name = $selected_class ?: 'All Classes';
$term_name = 'All Terms';
$academic_year_name = 'All Academic Years';

if ($selected_term_id) {
    $stmt = $pdo->prepare("SELECT name FROM terms WHERE id = ?");
    $stmt->execute([$selected_term_id]);
    $term = $stmt->fetch();
    $term_name = $term ? htmlspecialchars($term['name']) : 'N/A';
}

if ($selected_academic_year_id) {
    $stmt = $pdo->prepare("SELECT year FROM academic_years WHERE id = ?");
    $stmt->execute([$selected_academic_year_id]);
    $ay = $stmt->fetch();
    $academic_year_name = $ay ? htmlspecialchars($ay['year']) : 'N/A';
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Report Card</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            color: #333;
            background: white;
        }

        .report-container {
            max-width: 800px;
            margin: auto;
            background: white;
            padding: 20px;
            border: 2px solid #333;
        }

        .header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #333;
            padding-bottom: 20px;
        }

        .school-logo {
            max-height: 80px;
            margin-bottom: 10px;
        }

        .report-title {
            font-size: 1.5em;
            color: #667eea;
            font-weight: 600;
            margin: 10px 0;
        }

        .student-info {
            text-align: center;
            margin-bottom: 20px;
            font-size: 1.5em;
        }

        .info-section {
            display: flex;
            justify-content: space-between;
            margin-bottom: 20px;
            font-size: 1.2em;
            color: #555;
        }

        .info-item {
            flex: 1;
            padding: 0 10px;
            text-align: center;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            font-size: 1.5em;
            margin-top: 15px;
        }

        th, td {
            border: 1px solid #ccc;
            padding: 8px 10px;
            text-align: left;
            vertical-align: top;
        }

        th {
            background: #f0f0f0;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 1px;
            font-size: 1.0em;
        }

        tr:nth-child(even) {
            background: #f9f9f9;
        }

        .comments-cell {
            max-width: 140px;
            word-wrap: break-word;
            font-size: 0.8em;
        }

        .page-break {
            page-break-before: always;
        }

        .academic-section {
            page-break-inside: avoid;
        }

        .footer {
            margin-top: 30px;
            text-align: center;
            font-size: 0.9em;
            color: #888;
            border-top: 1px solid #ccc;
            padding-top: 10px;
        }

        @media print {
            body {
                margin: 0;
                background: white;
                font-size: 12px;
            }
            .report-container {
                border: none;
                padding: 10px;
                max-width: none;
            }
            .no-print {
                display: none;
            }
            .page-break {
                page-break-before: always;
            }
            .academic-section {
                page-break-inside: avoid;
            }
            table {
                font-size: 1.4em;
            }
            th, td {
                padding: 6px 8px;
            }
        }
    </style>
</head>
<body>
    <div class="report-container">
        <div class="header">
            <?php if ($school_logo_url): ?>
                <img src="<?php echo $school_logo_url; ?>" alt="School Logo" class="school-logo" />
            <?php endif; ?>
            <h2 class="report-title">Student Report Card</h2>
        </div>

        <div class="student-info">
            <strong><?php echo $student_name; ?></strong>
        </div>

        <div class="info-section">
            <div class="info-item"><strong>Class:</strong> <?php echo $class_name; ?></div>
            <div class="info-item"><strong>Term/Semester:</strong> <?php echo $term_name; ?></div>
            <div class="info-item"><strong>Academic Year:</strong> <?php echo $academic_year_name; ?></div>
        </div>

        <?php if (!empty($results)): ?>
            <!-- Academic Performance Section -->
            <div class="academic-section">
                <?php foreach ($results_by_term as $term_name => $term_results): ?>
                    <h3 style="color: #667eea; margin-top: 20px; margin-bottom: 8px; font-size: 1.1em;"><?php echo htmlspecialchars($term_name); ?></h3>

                    <!-- Academic Performance Table -->
                    <table>
                        <thead>
                            <tr>
                                <th style="width: 25%;">Subject</th>
                                <th style="width: 12%;">Assessment</th>
                                <th style="width: 12%;">Exam</th>
                                <th style="width: 12%;">Total</th>
                                <th style="width: 8%;">Grade</th>
                                <th style="width: 23%;">Comments</th>
                                <th style="width: 8%;">Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($term_results as $result): ?>
                                <tr>
                                    <td>
                                        <?php
                                            $course = trim((string)($result['course_name'] ?? ''));
                                            $subject = trim((string)($result['subject_name'] ?? ''));
                                            $code = trim((string)($result['subject_code'] ?? ''));

                                            if ($course && $subject) {
                                                echo htmlspecialchars($course . ' — ' . $subject);
                                            } elseif ($course) {
                                                echo htmlspecialchars($course);
                                            } elseif ($subject) {
                                                echo htmlspecialchars($subject);
                                            } else {
                                                echo 'N/A';
                                            }

                                            if ($code) {
                                                echo '<br><small>' . htmlspecialchars($code) . '</small>';
                                            }
                                        ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($result['converted_class_mark'] ?? 'N/A'); ?></td>
                                    <td><?php echo htmlspecialchars($result['converted_exam_mark'] ?? 'N/A'); ?></td>
                                    <td><?php echo htmlspecialchars($result['converted_final_mark'] ?? 'N/A'); ?></td>
                                    <td><?php echo htmlspecialchars($result['grade_value'] ?? 'N/A'); ?></td>
                                    <td class="comments-cell"><?php echo htmlspecialchars($result['comments'] ?? ''); ?></td>
                                    <td style="font-size: 0.65em;"><?php echo $result['graded_at'] ? date('M d', strtotime($result['graded_at'])) : 'N/A'; ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>

                    <!-- Performance Summary -->
                    <?php if (isset($term_metrics[$term_name])): ?>
                        <div style="margin-top: 15px; padding: 10px; background: #f9f9f9; border-radius: 5px; font-size: 1.5em;">
                            <div style="display: flex; justify-content: space-between; flex-wrap: wrap; gap: 10px;">
                                <div><strong>Expected:</strong> <?php echo $term_metrics[$term_name]['total_expected']; ?></div>
                                <div><strong>Obtained:</strong> <?php echo $term_metrics[$term_name]['total_obtained']; ?></div>
                                <div><strong>Average:</strong> <?php echo $term_metrics[$term_name]['average']; ?>%</div>
                            </div>
                        </div>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>

            <!-- Conduct Grades Section - New Page -->
            <div class="page-break">
                <div class="header">
                    <?php if ($school_logo_url): ?>
                        <img src="<?php echo $school_logo_url; ?>" alt="School Logo" class="school-logo" />
                    <?php endif; ?>
                    <h2 class="report-title">Student Report Card - Conduct Assessment</h2>
                </div>

                <div class="student-info">
                    <strong><?php echo $student_name; ?></strong>
                </div>

                <div class="info-section">
                    <div class="info-item"><strong>Class:</strong> <?php echo $class_name; ?></div>
                    <div class="info-item"><strong>Term/Semester:</strong> <?php echo $term_name; ?></div>
                    <div class="info-item"><strong>Academic Year:</strong> <?php echo $academic_year_name; ?></div>
                </div>

                <?php foreach ($results_by_term as $term_name => $term_results): ?>
                    <?php if (isset($conduct_grades[$term_name])): ?>
                        <h3 style="color: #667eea; margin-top: 30px; margin-bottom: 10px;"><?php echo htmlspecialchars($term_name); ?> - Conduct Assessment</h3>
                        <div style="padding: 15px; background: #e8f4f8; border-radius: 5px;">
                            <table style="width: 100%; border-collapse: collapse; font-size: 1.5em;">
                                <thead>
                                    <tr>
                                        <th style="border: 1px solid #ccc; padding: 8px; background: #f0f0f0;">Aspect</th>
                                        <th style="border: 1px solid #ccc; padding: 8px; background: #f0f0f0;">Grade</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td style="border: 1px solid #ccc; padding: 8px;">Appearance</td>
                                        <td style="border: 1px solid #ccc; padding: 8px;"><?php echo htmlspecialchars($conduct_grades[$term_name]['appearance'] ?? 'N/A'); ?></td>
                                    </tr>
                                    <tr>
                                        <td style="border: 1px solid #ccc; padding: 8px;">Conduct</td>
                                        <td style="border: 1px solid #ccc; padding: 8px;"><?php echo htmlspecialchars($conduct_grades[$term_name]['conduct'] ?? 'N/A'); ?></td>
                                    </tr>
                                    <tr>
                                        <td style="border: 1px solid #ccc; padding: 8px;">Interest</td>
                                        <td style="border: 1px solid #ccc; padding: 8px;"><?php echo htmlspecialchars($conduct_grades[$term_name]['interest'] ?? 'N/A'); ?></td>
                                    </tr>
                                    <tr>
                                        <td style="border: 1px solid #ccc; padding: 8px;">Class Participation</td>
                                        <td style="border: 1px solid #ccc; padding: 8px;"><?php echo htmlspecialchars($conduct_grades[$term_name]['class_participation'] ?? 'N/A'); ?></td>
                                    </tr>
                                    <tr>
                                        <td style="border: 1px solid #ccc; padding: 8px;">Punctuality</td>
                                        <td style="border: 1px solid #ccc; padding: 8px;"><?php echo htmlspecialchars($conduct_grades[$term_name]['pe_punctuality'] ?? 'N/A'); ?></td>
                                    </tr>
                                </tbody>
                            </table>
                            <?php
                            $conduct_strengths = [];
                            $conduct_weaknesses = [];
                            $conduct_aspects = ['appearance', 'conduct', 'interest', 'class_participation', 'pe_punctuality'];
                            foreach ($conduct_aspects as $aspect) {
                                $grade = $conduct_grades[$term_name][$aspect] ?? '';
                                if (in_array($grade, ['A', 'B'])) {
                                    $conduct_strengths[] = ucfirst(str_replace('_', ' ', $aspect));
                                } elseif (in_array($grade, ['D', 'E', 'F'])) {
                                    $conduct_weaknesses[] = ucfirst(str_replace('_', ' ', $aspect));
                                }
                            }
                            ?>
                            <?php if (!empty($conduct_strengths)): ?>
                                <div style="margin-top: 10px; font-size: 1.5em;"><strong>Areas of Strength:</strong> <?php echo htmlspecialchars(implode(', ', $conduct_strengths)); ?></div>
                            <?php endif; ?>
                            <?php if (!empty($conduct_weaknesses)): ?>
                                <div style="margin-top: 10px; font-size: 1.5em;"><strong>Areas of Weakness:</strong> <?php echo htmlspecialchars(implode(', ', $conduct_weaknesses)); ?></div>
                            <?php endif; ?>
                            <?php if (!empty($conduct_grades[$term_name]['class_teacher_remarks'])): ?>
                                <div style="margin-top: 10px; font-size: 1.5em;"><strong>Class Teacher Remarks:</strong> <?php echo htmlspecialchars($conduct_grades[$term_name]['class_teacher_remarks']); ?></div>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>


        <?php else: ?>
            <p>No results found for the selected filters.</p>
        <?php endif; ?>

                <!-- Reopening Date and Signature Section -->
                <div style="margin-top: 40px; padding: 20px; border-top: 2px solid #333; font-size: 1.5em;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                        <div style="text-align: left;">
                            <strong>Reopening Date:</strong> 8th January 2026
                        </div>
                        <div style="text-align: right;">
                            <div style="margin-bottom: 40px;">Head Teacher's Signature:</div>
                            <div style="border-bottom: 2px solid #333; width: 200px; margin-left: auto;"></div>
                        </div>
                    </div>
                </div>

        <div class="footer">
            Generated on <?php echo date('F d, Y'); ?>
        </div>

    <script>
        window.onload = function() {
            window.print();
        };
    </script>
</body>
</html>
