<?php
ob_start();

ini_set('display_errors', 1);

ini_set('display_startup_errors', 1);

error_reporting(E_ALL);

require_once '../config.php';


if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'teacher'])) {
    header('Location: ../login.php');
    exit;
}

require_once '../includes/user_activity_logger.php';

$message = '';

// Get current user and teacher ID
$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT id FROM staff WHERE user_id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher not found.");
}

$teacher_id = $teacher['id'];

$student_id = $_GET['student_id'] ?? null;
$class_name = $_GET['class'] ?? null;

if (!$student_id || !$class_name) {
    die("Student ID and class are required.");
}

// Get student full name
$stmt = $pdo->prepare("SELECT CONCAT(last_name, ', ', first_name, ' ', COALESCE(middle_name, '')) AS name FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch();

if (!$student) {
    die("Student not found.");
}

$student_name = trim($student['name']);

// Get class level_id
$stmt = $pdo->prepare("SELECT level_id FROM classes WHERE name = ?");
$stmt->execute([$class_name]);
$class_info = $stmt->fetch();

if (!$class_info) {
    die("Class not found.");
}

$class_level_id = $class_info['level_id'];
$is_preschool = ($class_level_id == 3);

// Load grading system for this level
$grading_system = $pdo->prepare("SELECT grade, min_score, max_score, description FROM grading_systems WHERE level_id = ? ORDER BY min_score DESC");
$grading_system->execute([$class_level_id]);
$grading_rules = $grading_system->fetchAll(PDO::FETCH_ASSOC);

// If preschool and no grading rules, use primary level rules
if ($class_level_id == 3 && empty($grading_rules)) {
    $grading_system->execute([1]);
    $grading_rules = $grading_system->fetchAll(PDO::FETCH_ASSOC);
}

// Get academic years and terms
$active_academic_years = $pdo->query("SELECT id, year, is_active FROM academic_years ORDER BY start_date DESC")->fetchAll();
$current_academic_year = null;
foreach ($active_academic_years as $year) {
    if ($year['is_active']) {
        $current_academic_year = $year;
        break;
    }
}

$active_terms = $pdo->query("SELECT id, name, is_active FROM terms ORDER BY start_date DESC")->fetchAll();
$current_term = null;
foreach ($active_terms as $term) {
    if ($term['is_active']) {
        $current_term = $term;
        break;
    }
}

$selected_academic_year_id = $_GET['academic_year_id'] ?? ($current_academic_year ? $current_academic_year['id'] : null);
$selected_term_id = $_GET['term_id'] ?? ($current_term ? $current_term['id'] : null);

// Get courses for teacher in this class and term
$stmt = $pdo->prepare("
    SELECT DISTINCT c.id, c.code, c.name
    FROM courses c
    JOIN teacher_courses tc ON c.id = tc.course_id
    JOIN class_courses cc ON c.id = cc.course_id
    JOIN classes cl ON cc.class_id = cl.id
    JOIN course_terms ct ON c.id = ct.course_id
    WHERE cl.name = ? AND tc.teacher_id = ? AND ct.term_id = ?
    ORDER BY c.name
");
$stmt->execute([$class_name, $teacher_id, $selected_term_id]);
$courses = $stmt->fetchAll();

$existing_grades = [];
foreach ($courses as $course) {
    if ($is_preschool) {
        $query = "
            SELECT g.grade, g.comments, g.exercise1, g.homework1, g.exam_score, g.final_score, g.converted_exam_mark
            FROM enrollments e
            LEFT JOIN preschool_grades g ON e.id = g.enrollment_id
            WHERE e.student_id = ? AND e.course_id = ?";
        $params = [$student_id, $course['id']];

        if ($selected_term_id) {
            $query .= " AND e.term_id = ?";
            $params[] = $selected_term_id;
        }
        if ($selected_academic_year_id) {
            $query .= " AND e.academic_year_id = ?";
            $params[] = $selected_academic_year_id;
        }
        if ($selected_term_id) {
            $query .= " AND (g.term_id = ? OR g.term_id IS NULL)";
            $params[] = $selected_term_id;
        }
        if ($selected_academic_year_id) {
            $query .= " AND (g.academic_year_id = ? OR g.academic_year_id IS NULL)";
            $params[] = $selected_academic_year_id;
        }

        $stmt2 = $pdo->prepare($query);
        $stmt2->execute($params);
        $grade = $stmt2->fetch();

        $existing_grades[$course['id']] = [
            'grade' => $grade['grade'] ?? '',
            'comments' => $grade['comments'] ?? '',
            'exercise1' => $grade['exercise1'] ?? '',
            'homework1' => $grade['homework1'] ?? '',
            'total_assessment' => '',
            'exam_score' => $grade['exam_score'] ?? '',
            'converted_exam_mark' => $grade['converted_exam_mark'] ?? '',
            'final_score' => $grade['final_score'] ?? ''
        ];
    } else {
        $query = "
            SELECT g.grade, g.comments, g.exercise1, g.exercise2, g.exercise3, g.exercise4,
                   g.class_test1, g.class_test2, g.class_test3,
                   g.homework1, g.homework2, g.homework3, g.homework4,
                   g.exam_score, g.assessment_total, g.final_score, g.converted_class_mark, g.converted_exam_mark
            FROM enrollments e
            LEFT JOIN grades g ON e.id = g.enrollment_id
            WHERE e.student_id = ? AND e.course_id = ?";
        $params = [$student_id, $course['id']];

        if ($selected_term_id) {
            $query .= " AND e.term_id = ?";
            $params[] = $selected_term_id;
        }
        if ($selected_academic_year_id) {
            $query .= " AND e.academic_year_id = ?";
            $params[] = $selected_academic_year_id;
        }
        if ($selected_term_id) {
            $query .= " AND (g.term_id = ? OR g.term_id IS NULL)";
            $params[] = $selected_term_id;
        }
        if ($selected_academic_year_id) {
            $query .= " AND (g.academic_year_id = ? OR g.academic_year_id IS NULL)";
            $params[] = $selected_academic_year_id;
        }

        $stmt2 = $pdo->prepare($query);
        $stmt2->execute($params);
        $grade = $stmt2->fetch();

        $existing_grades[$course['id']] = [
            'grade' => $grade['grade'] ?? '',
            'comments' => $grade['comments'] ?? '',
            'exercise1' => $grade['exercise1'] ?? '',
            'exercise2' => $grade['exercise2'] ?? '',
            'exercise3' => $grade['exercise3'] ?? '',
            'exercise4' => $grade['exercise4'] ?? '',
            'class_test1' => $grade['class_test1'] ?? '',
            'class_test2' => $grade['class_test2'] ?? '',
            'class_test3' => $grade['class_test3'] ?? '',
            'homework1' => $grade['homework1'] ?? '',
            'homework2' => $grade['homework2'] ?? '',
            'homework3' => $grade['homework3'] ?? '',
            'homework4' => $grade['homework4'] ?? '',
            'exam_score' => $grade['exam_score'] ?? '',
            'assessment_total' => $grade['assessment_total'] ?? '',
            'final_score' => $grade['final_score'] ?? '',
            'converted_class_mark' => $grade['converted_class_mark'] ?? '',
            'converted_exam_mark' => $grade['converted_exam_mark'] ?? ''
        ];
    }
}

// Handle Grade Submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_student_grades'])) {
    $course_ids = $_POST['course_id'] ?? [];
    $exercise1_scores = $_POST['exercise1'] ?? [];
    $exercise2_scores = $_POST['exercise2'] ?? [];
    $exercise3_scores = $_POST['exercise3'] ?? [];
    $exercise4_scores = $_POST['exercise4'] ?? [];
    $class_test1_scores = $_POST['class_test1'] ?? [];
    $class_test2_scores = $_POST['class_test2'] ?? [];
    $class_test3_scores = $_POST['class_test3'] ?? [];
    $homework1_scores = $_POST['homework1'] ?? [];
    $homework2_scores = $_POST['homework2'] ?? [];
    $homework3_scores = $_POST['homework3'] ?? [];
    $homework4_scores = $_POST['homework4'] ?? [];
    $exam_scores = $_POST['exam_score'] ?? [];
    $assessment_totals = $_POST['assessment_total'] ?? [];
    $final_scores = $_POST['final_score'] ?? [];
    $grades = $_POST['grade'] ?? [];
    $comments = $_POST['comments'] ?? [];
    $term_id = $_POST['term_id'] ?? null;
    $academic_year_id = $_POST['academic_year_id'] ?? null;

    $is_ajax = isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';

    if (!$term_id || !$academic_year_id) {
        $message = "Please select both term and academic year.";
        if ($is_ajax) {
            if (ob_get_length()) {
                ob_end_clean();
            }
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'message' => $message]);
            exit;
        }
    } else {
        $saved_count = 0;
        $debug_info = [];
        $debug_log = [];

        $debug_log[] = "POST data received: " . json_encode($_POST);
        $debug_log[] = "Course IDs: " . json_encode($course_ids);
        $debug_log[] = "Term ID: $term_id, Academic Year ID: $academic_year_id";

        $logger = new UserActivityLogger();

        foreach ($course_ids as $course_id) {
            $debug_log[] = "Processing course ID: $course_id";

            $stmt = $pdo->prepare("SELECT id FROM enrollments WHERE student_id = ? AND course_id = ? AND academic_year_id = ? AND term_id = ?");
            $stmt->execute([$student_id, $course_id, $academic_year_id, $term_id]);
            $enrollment = $stmt->fetch();

            if ($enrollment) {
                $enrollment_id = $enrollment['id'];

                $grade = $grades[$course_id] ?? '';
                $comment = $comments[$course_id] ?? '';

                if ($is_preschool) {
                    $exercise1 = $exercise1_scores[$course_id] ?? null;
                    $homework1 = $homework1_scores[$course_id] ?? null;
                    $exam_score = $exam_scores[$course_id] ?? null;
                    $final_score = $final_scores[$course_id] ?? null;

                    try {
                        $stmtCheck = $pdo->prepare("SELECT id FROM preschool_grades WHERE enrollment_id = ? AND term_id = ? AND academic_year_id = ?");
                        $stmtCheck->execute([$enrollment_id, $term_id, $academic_year_id]);
                        $existing = $stmtCheck->fetch();

                        if ($existing) {
                            $stmtUpdate = $pdo->prepare("UPDATE preschool_grades SET grade = ?, comments = ?, exercise1 = ?, homework1 = ?, exam_score = ?, final_score = ? WHERE enrollment_id = ? AND term_id = ? AND academic_year_id = ?");
                            $result = $stmtUpdate->execute([$grade, $comment, $exercise1, $homework1, $exam_score, $final_score, $enrollment_id, $term_id, $academic_year_id]);
                        } else {
                            $stmtInsert = $pdo->prepare("INSERT INTO preschool_grades (enrollment_id, grade, comments, exercise1, homework1, exam_score, final_score, term_id, academic_year_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
                            $result = $stmtInsert->execute([$enrollment_id, $grade, $comment, $exercise1, $homework1, $exam_score, $final_score, $term_id, $academic_year_id]);
                        }

                        if ($result) {
                            $saved_count++;
                        }
                    } catch (Exception $e) {
                        // Log or handle error here
                    }
                } else {
                    $exercise1 = $exercise1_scores[$course_id] ?? null;
                    $exercise2 = $exercise2_scores[$course_id] ?? null;
                    $exercise3 = $exercise3_scores[$course_id] ?? null;
                    $exercise4 = $exercise4_scores[$course_id] ?? null;
                    $class_test1 = $class_test1_scores[$course_id] ?? null;
                    $class_test2 = $class_test2_scores[$course_id] ?? null;
                    $class_test3 = $class_test3_scores[$course_id] ?? null;
                    $homework1 = $homework1_scores[$course_id] ?? null;
                    $homework2 = $homework2_scores[$course_id] ?? null;
                    $homework3 = $homework3_scores[$course_id] ?? null;
                    $homework4 = $homework4_scores[$course_id] ?? null;
                    $exam_score = $exam_scores[$course_id] ?? null;
                    $assessment_total = $assessment_totals[$course_id] ?? null;
                    $final_score = $final_scores[$course_id] ?? null;

                    $converted_class_mark = ($assessment_total !== null && $assessment_total !== '') ? floatval($assessment_total) * 0.3 : null;
                    $converted_exam_mark = ($exam_score !== null && $exam_score !== '') ? floatval($exam_score) * 0.7 : null;
                    $converted_final_mark = ($final_score !== null && $final_score !== '') ? floatval($final_score) : null;

                    try {
                        $stmtCheck = $pdo->prepare("SELECT id FROM grades WHERE enrollment_id = ? AND term_id = ? AND academic_year_id = ?");
                        $stmtCheck->execute([$enrollment_id, $term_id, $academic_year_id]);
                        $existing = $stmtCheck->fetch();

                        if ($existing) {
                            $stmtUpdate = $pdo->prepare("UPDATE grades SET grade = ?, comments = ?, exercise1 = ?, exercise2 = ?, exercise3 = ?, exercise4 = ?, class_test1 = ?, class_test2 = ?, class_test3 = ?, homework1 = ?, homework2 = ?, homework3 = ?, homework4 = ?, exam_score = ?, assessment_total = ?, final_score = ?, converted_class_mark = ?, converted_exam_mark = ?, converted_final_mark = ? WHERE enrollment_id = ? AND term_id = ? AND academic_year_id = ?");
                            $result = $stmtUpdate->execute([$grade, $comment, $exercise1, $exercise2, $exercise3, $exercise4, $class_test1, $class_test2, $class_test3, $homework1, $homework2, $homework3, $homework4, $exam_score, $assessment_total, $final_score, $converted_class_mark, $converted_exam_mark, $converted_final_mark, $enrollment_id, $term_id, $academic_year_id]);
                        } else {
                            $stmtInsert = $pdo->prepare("INSERT INTO grades (enrollment_id, grade, comments, exercise1, exercise2, exercise3, exercise4, class_test1, class_test2, class_test3, homework1, homework2, homework3, homework4, exam_score, assessment_total, final_score, converted_class_mark, converted_exam_mark, converted_final_mark, term_id, academic_year_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                            $result = $stmtInsert->execute([$enrollment_id, $grade, $comment, $exercise1, $exercise2, $exercise3, $exercise4, $class_test1, $class_test2, $class_test3, $homework1, $homework2, $homework3, $homework4, $exam_score, $assessment_total, $final_score, $converted_class_mark, $converted_exam_mark, $converted_final_mark, $term_id, $academic_year_id]);
                        }

                        if ($result) {
                            $saved_count++;
                        }
                    } catch (Exception $e) {
                        // Log or handle error here
                    }


                }
            }
        }

        if ($saved_count > 0) {
            $logger->logActivity($_SESSION['user_id'], 'Saved/Updated grades', "Saved grades for student_id: $student_id for $saved_count courses in academic_year_id: $academic_year_id, term_id: $term_id.");
            $message = "Grades saved successfully for $saved_count courses.";
        } else {
            $message = "No grades were updated.";
        }

        if ($is_ajax) {
            if (ob_get_length()) {
                ob_end_clean();
            }
            header('Content-Type: application/json');
            echo json_encode(['success' => true, 'message' => $message, 'saved_count' => $saved_count]);
            exit;
        } else {
            header("Location: assign_grades.php?class=" . urlencode($class_name) . "&message=" . urlencode($message));
            exit;
        }
    }
}

$role = $_SESSION['role'];

if ($role == 'admin') {
    include '../includes/admin_header.php';
} else {
    include '../includes/teacher_header.php';
}
?>

<style>
    /* Basic styling omitted for brevity, use the existing styles from your current file */
    .add-grades-container {
        padding: 20px;
        background-color: #f8f9fa;
        min-height: 100vh;
    }
    .page-header {
        background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
        color: white;
        padding: 30px;
        border-radius: 15px;
        margin-bottom: 30px;
        text-align: center;
        box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    }
    .page-header h2 {
        margin: 0;
        font-size: 1.5em;
        font-weight: 700;
        text-align: left;
    }
    .message {
        padding: 15px 20px;
        border-radius: 8px;
        margin-bottom: 20px;
        font-weight: 500;
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }
    .grades-form-container {
        background: white;
        border-radius: 15px;
        padding: 25px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
    }
    .grades-form-container h3 {
        color: #2c3e50;
        margin-bottom: 20px;
        font-size: 1.8em;
    }
    .back-link {
        display: inline-block;
        padding: 10px 20px;
        background: #6c757d;
        color: white;
        text-decoration: none;
        border-radius: 8px;
        font-weight: 600;
        margin-bottom: 20px;
        transition: background-color 0.3s ease;
    }
    .back-link:hover {
        background: #5a6268;
        color: white;
        text-decoration: none;
    }
    .grades-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        border-radius: 8px;
        overflow: hidden;
    }
    .grades-table th,
    .grades-table td {
        padding: 12px 15px;
        text-align: left;
        border-bottom: 1px solid #ddd;
    }
    .grades-table th {
        background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
        color: white;
        font-weight: 600;
        font-size: 0.9em;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    .grades-table tr:nth-child(even) {
        background-color: #f8f9fa;
    }
    .grades-table tr:hover {
        background-color: #e8f5e8;
    }
    .grades-table .course-name {
        font-weight: 600;
        color: #2c3e50;
        min-width: 200px;
    }
    .grades-table input,
    .grades-table select,
    .grades-table textarea {
        width: 90px;
        min-width: 90px;
        padding: 8px 10px;
        border: 1px solid #ddd;
        border-radius: 4px;
        font-size: 0.9em;
        transition: border-color 0.3s ease;
        box-sizing: border-box;
    }
    .grades-table input:focus,
    .grades-table select:focus,
    .grades-table textarea:focus {
        outline: none;
        border-color: #27ae60;
        box-shadow: 0 0 0 2px rgba(39, 174, 96, 0.1);
    }
    .grades-table input[readonly] {
        background-color: #f8f9fa;
        cursor: not-allowed;
    }
    .grades-table input.invalid {
        border-color: red;
        background-color: #ffe6e6;
    }
    .grades-table textarea {
        resize: vertical;
        min-height: 60px;
    }
    .term-selection {
        margin-bottom: 20px;
        padding: 15px;
        background: #f8f9fa;
        border-radius: 8px;
        border: 1px solid #ddd;
    }
    .term-selection label {
        font-weight: 600;
        color: #2c3e50;
        margin-right: 10px;
    }
    .term-selection select {
        padding: 8px 12px;
        border: 1px solid #ddd;
        border-radius: 4px;
        font-size: 1em;
        background-color: #fff;
    }
    .form-group {
        margin-bottom: 20px;
    }
    .form-group label {
        display: block;
        margin-bottom: 8px;
        font-weight: 600;
        color: #2c3e50;
    }
    .form-group input,
    .form-group textarea,
    .form-group select {
        width: 100%;
        padding: 12px 15px;
        border: 2px solid #ddd;
        border-radius: 8px;
        font-size: 1em;
        transition: border-color 0.3s ease;
        box-sizing: border-box;
        background-color: #fff;
    }
    .form-group input:focus,
    .form-group textarea:focus,
    .form-group select:focus {
        outline: none;
        border-color: #27ae60;
        box-shadow: 0 0 0 3px rgba(39, 174, 96, 0.1);
    }
    .form-group textarea {
        resize: vertical;
        min-height: 80px;
    }
    .form-group input[readonly] {
        background-color: #f8f9fa;
        cursor: not-allowed;
    }
    .btn-primary {
        padding: 12px 24px;
        background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
        color: white;
        border: none;
        border-radius: 8px;
        font-size: 1em;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
    }
    .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(39, 174, 96, 0.3);
    }
    @media (max-width: 768px) {
        .tab-navigation {
            flex-wrap: wrap;
        }
        .tab-button {
            padding: 10px 15px;
            font-size: 0.8em;
        }
    }
</style>

<div class="add-grades-container">
    <div class="page-header">
        <h2>📝 <?php echo htmlspecialchars($class_name); ?> - <?php echo htmlspecialchars($current_academic_year ? $current_academic_year['year'] : 'Academic Year'); ?> - <?php echo htmlspecialchars($current_term ? $current_term['name'] : 'Term'); ?> - Grades for <?php echo htmlspecialchars($student_name); ?></h2>
    </div>

    <?php if ($message): ?>
        <div class="message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <div class="grades-form-container">
        <a href="assign_grades.php?class=<?php echo urlencode($class_name); ?>" class="back-link">← Back to Students</a>

        <h3>Grade Entry</h3>

        <form id="grades-form" method="POST">
            <div class="term-selection">
                <div style="display: flex; gap: 20px; flex-wrap: wrap; align-items: center;">
                    <input type="hidden" name="student_id" value="<?php echo htmlspecialchars($student_id); ?>">
                    <input type="hidden" name="class" value="<?php echo htmlspecialchars($class_name); ?>">

                    <div>
                        <label for="academic_year_id"><strong>Academic Year:</strong></label>
                        <select name="academic_year_id" id="academic_year_id" disabled style="margin-left: 10px; padding: 5px; background-color: #f8f9fa; cursor: not-allowed;">
                            <?php foreach ($active_academic_years as $year): ?>
                                <option value="<?php echo $year['id']; ?>" <?php echo ($selected_academic_year_id == $year['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($year['year']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <input type="hidden" name="academic_year_id" value="<?php echo htmlspecialchars($selected_academic_year_id); ?>">
                    </div>

                    <div>
                        <label for="term_id"><strong>Term/Semester:</strong></label>
                        <select name="term_id" id="term_id" disabled style="margin-left: 10px; padding: 5px; background-color: #f8f9fa; cursor: not-allowed;">
                            <?php foreach ($active_terms as $term): ?>
                                <option value="<?php echo $term['id']; ?>" <?php echo ($selected_term_id == $term['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($term['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <input type="hidden" name="term_id" value="<?php echo htmlspecialchars($selected_term_id); ?>">
                    </div>
                </div>
            </div>

            <table class="grades-table">
                <thead>
                    <?php if ($is_preschool): ?>
                        <tr>
                            <th>Course</th>
                            <th>Exercises (30 marks)</th>
                            <th>Homework (20 marks)</th>
                            <th>Total Assessment (50 marks)</th>
                            <th>Exam (100 marks)</th>
                            <th>Converted Exams (50 marks)</th>
                            <th>Final Score (100 marks)</th>
                            <th>Grade</th>
                            <th>Comments</th>
                        </tr>
                    <?php else: ?>
                        <tr>
                            <th>Course</th>
                            <th colspan="4">Exercise (40 marks total - 4×10)</th>
                            <th colspan="3">Class Test (40 marks total - 10,10,20)</th>
                            <th colspan="4">Homework (20 marks total - 4×5)</th>
                            <th>Exam (100 marks)</th>
                            <th>Assessment Total (100 marks)</th>
                            <th>Assessment Contribution (30%)</th>
                            <th>Exam Contribution (70%)</th>
                            <th>Final Score (100 marks)</th>
                            <th>Grade</th>
                            <th>Comments</th>
                        </tr>
                        <tr>
                            <th></th>
                            <th>E1 (10)</th>
                            <th>E2 (10)</th>
                            <th>E3 (10)</th>
                            <th>E4 (10)</th>
                            <th>CT1 (10)</th>
                            <th>CT2 (10)</th>
                            <th>CT3 (20)</th>
                            <th>H1 (5)</th>
                            <th>H2 (5)</th>
                            <th>H3 (5)</th>
                            <th>H4 (5)</th>
                            <th>Exam (100)</th>
                            <th>Assessment (100)</th>
                            <th>Assessment (30%)</th>
                            <th>Exam (70%)</th>
                            <th>Final (100)</th>
                            <th></th>
                            <th></th>
                        </tr>
                    <?php endif; ?>
                </thead>
                <tbody>
                    <?php foreach ($courses as $course): ?>
                        <tr>
                            <td class="course-name">
                                <input type="hidden" name="course_id[<?php echo $course['id']; ?>]" value="<?php echo $course['id']; ?>">
                                <?php echo htmlspecialchars($course['code'] ?? ''); ?> - <?php echo htmlspecialchars($course['name']); ?>
                            </td>
                            <?php if ($is_preschool): ?>
                                <td><input type="number" id="exercise1_<?php echo $course['id']; ?>" name="exercise1[<?php echo $course['id']; ?>]" min="0" max="30" step="0.01" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['exercise1'] ?? ''); ?>"></td>
                                <td><input type="number" id="homework1_<?php echo $course['id']; ?>" name="homework1[<?php echo $course['id']; ?>]" min="0" max="20" step="0.01" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['homework1'] ?? ''); ?>"></td>
                                <td><input type="number" id="total_assessment_<?php echo $course['id']; ?>" readonly step="0.01" style="width: 80px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['total_assessment'] ?? ''); ?>"></td>
                                <td><input type="number" id="exam_score_<?php echo $course['id']; ?>" name="exam_score[<?php echo $course['id']; ?>]" min="0" max="100" step="0.01" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['exam_score'] ?? ''); ?>"></td>
                                <td><input type="number" id="converted_exam_mark_<?php echo $course['id']; ?>" readonly step="0.01" style="width: 80px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['converted_exam_mark'] ?? ''); ?>"></td>
                                <td><input type="number" id="final_score_<?php echo $course['id']; ?>" name="final_score[<?php echo $course['id']; ?>]" readonly step="0.01" style="width: 80px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['final_score'] ?? ''); ?>"></td>
                                <td>
                                    <input type="text" id="grade_<?php echo $course['id']; ?>" name="grade[<?php echo $course['id']; ?>]" readonly value="<?php echo htmlspecialchars($existing_grades[$course['id']]['grade'] ?? ''); ?>">
                                </td>
                                <td>
                                    <textarea id="comments_<?php echo $course['id']; ?>" name="comments[<?php echo $course['id']; ?>]" readonly placeholder="Auto-filled based on grade"><?php echo htmlspecialchars($existing_grades[$course['id']]['comments'] ?? ''); ?></textarea>
                                </td>
                            <?php else: ?>
                                <td><input type="number" id="exercise1_<?php echo $course['id']; ?>" name="exercise1[<?php echo $course['id']; ?>]" min="0" step="0.01" style="width: 50px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['exercise1'] ?? ''); ?>"></td>
                                <td><input type="number" id="exercise2_<?php echo $course['id']; ?>" name="exercise2[<?php echo $course['id']; ?>]" min="0" step="0.01" style="width: 50px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['exercise2'] ?? ''); ?>"></td>
                                <td><input type="number" id="exercise3_<?php echo $course['id']; ?>" name="exercise3[<?php echo $course['id']; ?>]" min="0" step="0.01" style="width: 50px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['exercise3'] ?? ''); ?>"></td>
                                <td><input type="number" id="exercise4_<?php echo $course['id']; ?>" name="exercise4[<?php echo $course['id']; ?>]" min="0" step="0.01" style="width: 50px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['exercise4'] ?? ''); ?>"></td>
                                <td><input type="number" id="class_test1_<?php echo $course['id']; ?>" name="class_test1[<?php echo $course['id']; ?>]" min="0" step="0.01" style="width: 50px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['class_test1'] ?? ''); ?>"></td>
                                <td><input type="number" id="class_test2_<?php echo $course['id']; ?>" name="class_test2[<?php echo $course['id']; ?>]" min="0" step="0.01" style="width: 50px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['class_test2'] ?? ''); ?>"></td>
                                <td><input type="number" id="class_test3_<?php echo $course['id']; ?>" name="class_test3[<?php echo $course['id']; ?>]" min="0" step="0.01" style="width: 50px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['class_test3'] ?? ''); ?>"></td>
                                <td><input type="number" id="homework1_<?php echo $course['id']; ?>" name="homework1[<?php echo $course['id']; ?>]" min="0" step="0.01" style="width: 50px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['homework1'] ?? ''); ?>"></td>
                                <td><input type="number" id="homework2_<?php echo $course['id']; ?>" name="homework2[<?php echo $course['id']; ?>]" min="0" step="0.01" style="width: 50px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['homework2'] ?? ''); ?>"></td>
                                <td><input type="number" id="homework3_<?php echo $course['id']; ?>" name="homework3[<?php echo $course['id']; ?>]" min="0" step="0.01" style="width: 50px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['homework3'] ?? ''); ?>"></td>
                                <td><input type="number" id="homework4_<?php echo $course['id']; ?>" name="homework4[<?php echo $course['id']; ?>]" min="0" step="0.01" style="width: 50px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['homework4'] ?? ''); ?>"></td>
                                <td><input type="number" id="exam_score_<?php echo $course['id']; ?>" name="exam_score[<?php echo $course['id']; ?>]" min="0" step="0.01" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['exam_score'] ?? ''); ?>"></td>
                                <td><input type="number" id="assessment_total_<?php echo $course['id']; ?>" name="assessment_total[<?php echo $course['id']; ?>]" readonly step="0.01" style="width: 60px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['assessment_total'] ?? ''); ?>"></td>
                                <td><input type="number" id="assessment_contribution_<?php echo $course['id']; ?>" readonly step="0.01" style="width: 60px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['converted_class_mark'] ?? ''); ?>"></td>
                                <td><input type="number" id="exam_contribution_<?php echo $course['id']; ?>" readonly step="0.01" style="width: 60px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['converted_exam_mark'] ?? ''); ?>"></td>
                                <td><input type="number" id="final_score_<?php echo $course['id']; ?>" name="final_score[<?php echo $course['id']; ?>]" readonly step="0.01" style="width: 80px;" value="<?php echo htmlspecialchars($existing_grades[$course['id']]['final_score'] ?? ''); ?>"></td>
                                <td>
                                    <input type="text" id="grade_<?php echo $course['id']; ?>" name="grade[<?php echo $course['id']; ?>]" readonly value="<?php echo htmlspecialchars($existing_grades[$course['id']]['grade'] ?? ''); ?>">
                                </td>
                                <td>
                                    <textarea id="comments_<?php echo $course['id']; ?>" name="comments[<?php echo $course['id']; ?>]" readonly placeholder="Auto-filled based on grade"><?php echo htmlspecialchars($existing_grades[$course['id']]['comments'] ?? ''); ?></textarea>
                                </td>
                            <?php endif; ?>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <div style="margin-top: 20px; text-align: center;">
                <button id="save_grades_btn" type="button" class="btn-primary">💾 Save Grades</button>
            </div>
        </form>
    </div>
</div>

<script>
const gradingRules = <?php echo json_encode($grading_rules); ?>;

document.addEventListener('DOMContentLoaded', function() {
    const isPreschool = <?php echo $is_preschool ? 'true' : 'false'; ?>;
    const saveButton = document.getElementById('save_grades_btn');
    if (!saveButton) {
        console.error('Save grades button not found');
        return;
    }

    saveButton.addEventListener('click', function() {
        const academicYearId = document.querySelector('input[name="academic_year_id"]').value;
        const termId = document.querySelector('input[name="term_id"]').value;

        if (!academicYearId || !termId) {
            alert('Please select both academic year and term.');
            return;
        }

        const formData = new FormData();
        formData.append('save_student_grades', '1');
        formData.append('academic_year_id', academicYearId);
        formData.append('term_id', termId);

        const courseInputs = document.querySelectorAll('input[name^="course_id"]');
        courseInputs.forEach(input => {
            const courseId = input.value;
            formData.append(`course_id[${courseId}]`, courseId);
            if (isPreschool) {
                formData.append(`exercise1[${courseId}]`, document.getElementById(`exercise1_${courseId}`).value);
                formData.append(`homework1[${courseId}]`, document.getElementById(`homework1_${courseId}`).value);
                formData.append(`exam_score[${courseId}]`, document.getElementById(`exam_score_${courseId}`).value);
                formData.append(`final_score[${courseId}]`, document.getElementById(`final_score_${courseId}`).value);
            } else {
                formData.append(`exercise1[${courseId}]`, document.getElementById(`exercise1_${courseId}`).value);
                formData.append(`exercise2[${courseId}]`, document.getElementById(`exercise2_${courseId}`).value);
                formData.append(`exercise3[${courseId}]`, document.getElementById(`exercise3_${courseId}`).value);
                formData.append(`exercise4[${courseId}]`, document.getElementById(`exercise4_${courseId}`).value);
                formData.append(`class_test1[${courseId}]`, document.getElementById(`class_test1_${courseId}`).value);
                formData.append(`class_test2[${courseId}]`, document.getElementById(`class_test2_${courseId}`).value);
                formData.append(`class_test3[${courseId}]`, document.getElementById(`class_test3_${courseId}`).value);
                formData.append(`homework1[${courseId}]`, document.getElementById(`homework1_${courseId}`).value);
                formData.append(`homework2[${courseId}]`, document.getElementById(`homework2_${courseId}`).value);
                formData.append(`homework3[${courseId}]`, document.getElementById(`homework3_${courseId}`).value);
                formData.append(`homework4[${courseId}]`, document.getElementById(`homework4_${courseId}`).value);
                formData.append(`exam_score[${courseId}]`, document.getElementById(`exam_score_${courseId}`).value);
                formData.append(`assessment_total[${courseId}]`, document.getElementById(`assessment_total_${courseId}`).value);
                formData.append(`final_score[${courseId}]`, document.getElementById(`final_score_${courseId}`).value);
            }
            formData.append(`grade[${courseId}]`, document.getElementById(`grade_${courseId}`).value);
            formData.append(`comments[${courseId}]`, document.getElementById(`comments_${courseId}`).value);
        });

        fetch('add_grades.php?student_id=<?php echo urlencode($student_id); ?>&class=<?php echo urlencode($class_name); ?>', {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(data => {
            alert(data.message);
            if (data.success) {
                window.location.reload();
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred while saving grades.');
        });
    });
});
</script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const isPreschool = <?php echo $is_preschool ? 'true' : 'false'; ?>;

    // For each course row, calculate assessment total and contributions dynamically
    function calculateAndUpdate(courseId) {
        // Parse inputs as floats or zero
        function parseInput(id) {
            const el = document.getElementById(id);
            if (!el) return 0;
            let val = parseFloat(el.value);
            return isNaN(val) ? 0 : val;
        }

        if (isPreschool) {
            // For preschool, final score is exercises (30) + homework (20) + exam converted to 50 marks
            // Cap values at their maximums
            const exerciseScore = Math.min(parseInput(`exercise1_${courseId}`), 30);
            const homeworkScore = Math.min(parseInput(`homework1_${courseId}`), 20);
            const totalAssessment = exerciseScore + homeworkScore;
            const examScore = Math.min(parseInput(`exam_score_${courseId}`), 100);
            const finalScoreCalc = totalAssessment + (examScore * 0.5);

            const totalAssessmentInput = document.getElementById(`total_assessment_${courseId}`);
            if (totalAssessmentInput) {
                totalAssessmentInput.value = totalAssessment.toFixed(2);
            }

            const finalScoreInput = document.getElementById(`final_score_${courseId}`);
            if (finalScoreInput) {
                finalScoreInput.value = finalScoreCalc.toFixed(2);
            }

            // Set converted marks
            const convertedExamMark = examScore * 0.5;

            const convertedExamMarkInput = document.getElementById(`converted_exam_mark_${courseId}`);
            if (convertedExamMarkInput) {
                convertedExamMarkInput.value = convertedExamMark.toFixed(2);
            }

            // Update grade based on gradingRules and final score
            const gradeInput = document.getElementById(`grade_${courseId}`);
            const commentsInput = document.getElementById(`comments_${courseId}`);

            if (gradeInput && commentsInput) {
                let gradeFound = '';
                let commentFound = '';
                for (const rule of gradingRules) {
                    if (finalScoreCalc >= parseFloat(rule.min_score) && finalScoreCalc <= parseFloat(rule.max_score)) {
                        gradeFound = rule.grade;
                        commentFound = rule.description || '';
                        break;
                    }
                }
                gradeInput.value = gradeFound;
                commentsInput.value = commentFound;
            }
        } else {
            // Sum exercises (4 x 10 marks)
            const exerciseTotal = parseInput(`exercise1_${courseId}`) + parseInput(`exercise2_${courseId}`) + parseInput(`exercise3_${courseId}`) + parseInput(`exercise4_${courseId}`);

            // Sum class tests (10,10,20)
            const classTestTotal = parseInput(`class_test1_${courseId}`) + parseInput(`class_test2_${courseId}`) + parseInput(`class_test3_${courseId}`);

            // Sum homework (4 x 5)
            const homeworkTotal = parseInput(`homework1_${courseId}`) + parseInput(`homework2_${courseId}`) + parseInput(`homework3_${courseId}`) + parseInput(`homework4_${courseId}`);

            // Calculate assessment total (sum of exercises, class tests, homework)
            const assessmentTotalCalc = exerciseTotal + classTestTotal + homeworkTotal;

            // Set assessment_total input readonly but update its value here
            const assessmentTotalInput = document.getElementById(`assessment_total_${courseId}`);
            if (assessmentTotalInput) {
                assessmentTotalInput.value = assessmentTotalCalc.toFixed(2);
            }

            // Exam score input
            const examScore = parseInput(`exam_score_${courseId}`);

            // Calculate contributions
            const assessmentContribution = assessmentTotalCalc * 0.3; // 30%
            const examContribution = examScore * 0.7; // 70%

            // Update contribution inputs
            const assessmentContributionInput = document.getElementById(`assessment_contribution_${courseId}`);
            if (assessmentContributionInput) {
                assessmentContributionInput.value = assessmentContribution.toFixed(2);
            }

            const examContributionInput = document.getElementById(`exam_contribution_${courseId}`);
            if (examContributionInput) {
                 examContributionInput.value = examContribution.toFixed(2);
            }

            // Calculate final score (sum contributions)
            const finalScoreCalc = assessmentContribution + examContribution;

            const finalScoreInput = document.getElementById(`final_score_${courseId}`);
            if (finalScoreInput) {
                finalScoreInput.value = finalScoreCalc.toFixed(2);
            }

            // Update grade based on gradingRules and final score
            const gradeInput = document.getElementById(`grade_${courseId}`);
            const commentsInput = document.getElementById(`comments_${courseId}`);

            if (gradeInput && commentsInput) {
                let gradeFound = '';
                let commentFound = '';
                for (const rule of gradingRules) {
                    if (finalScoreCalc >= parseFloat(rule.min_score) && finalScoreCalc <= parseFloat(rule.max_score)) {
                        gradeFound = rule.grade;
                        commentFound = rule.description || '';
                        break;
                    }
                }
                gradeInput.value = gradeFound;
                commentsInput.value = commentFound;
            }
        }
    }

    // Attach event listeners to inputs to trigger calculations on change
    let inputSelectors;
    if (isPreschool) {
        inputSelectors = ['exercise1', 'homework1', 'exam_score'];
    } else {
        inputSelectors = [
            'exercise1', 'exercise2', 'exercise3', 'exercise4',
            'class_test1', 'class_test2', 'class_test3',
            'homework1', 'homework2', 'homework3', 'homework4',
            'exam_score'
        ];
    }

    const courses = <?php echo json_encode(array_column($courses, 'id')); ?>;

    courses.forEach(courseId => {
        inputSelectors.forEach(prefix => {
            const input = document.getElementById(`${prefix}_${courseId}`);
            if (input) {
                input.addEventListener('input', () => {
                    calculateAndUpdate(courseId);
                });
                input.addEventListener('change', () => {
                    calculateAndUpdate(courseId);
                });
            }
        });

        // Enforce max values for preschool inputs
        if (isPreschool) {
            const exerciseInput = document.getElementById(`exercise1_${courseId}`);
            if (exerciseInput) {
                const enforceMaxExercise = () => {
                    if (parseFloat(exerciseInput.value) > 30) exerciseInput.value = 30;
                };
                exerciseInput.addEventListener('input', enforceMaxExercise);
                exerciseInput.addEventListener('change', enforceMaxExercise);
                exerciseInput.addEventListener('blur', enforceMaxExercise);
            }
            const homeworkInput = document.getElementById(`homework1_${courseId}`);
            if (homeworkInput) {
                const enforceMaxHomework = () => {
                    if (parseFloat(homeworkInput.value) > 20) homeworkInput.value = 20;
                };
                homeworkInput.addEventListener('input', enforceMaxHomework);
                homeworkInput.addEventListener('change', enforceMaxHomework);
                homeworkInput.addEventListener('blur', enforceMaxHomework);
            }
            const examInput = document.getElementById(`exam_score_${courseId}`);
            if (examInput) {
                const enforceMaxExam = () => {
                    if (parseFloat(examInput.value) > 100) examInput.value = 100;
                };
                examInput.addEventListener('input', enforceMaxExam);
                examInput.addEventListener('change', enforceMaxExam);
                examInput.addEventListener('blur', enforceMaxExam);
            }
        }

        // Initial calculation on page load
        calculateAndUpdate(courseId);
    });
});
</script>