<?php
require_once 'config.php';
require_once 'includes/user_activity_logger.php';
require_once 'includes/csrf_protection.php';
require_once 'includes/input_validation.php';
require_once 'includes/password_security.php';

// Fetch school logo from settings
$school_settings = $pdo->query("SELECT logo_path FROM school_settings LIMIT 1")->fetch(PDO::FETCH_ASSOC);
$logo_path = $school_settings['logo_path'] ?? 'img/dps.jpg';

$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Validate CSRF token
    if (!CSRFProtection::validateRequest()) {
        $error = 'Security validation failed. Please refresh the page and try again.';
    } else {
        // Sanitize and validate input
        $username = InputValidation::getPostData('username', 'string');
        $password = InputValidation::getPostData('password', 'string');

        // Validate required fields
        $validationErrors = InputValidation::validateRequired($_POST, ['username', 'password']);
        if (!empty($validationErrors)) {
            $error = implode('<br>', $validationErrors);
        } elseif (empty($username) || empty($password)) {
            $error = 'Please fill in all fields.';
        } else {
            // Check student credentials in student_users table only
            $stmt = $pdo->prepare("SELECT id, password FROM student_users WHERE username = ?");
            $stmt->execute([$username]);
            $user = $stmt->fetch();

            if ($user && PasswordSecurity::verifyPassword($password, $user['password'])) {
                // Set session for student
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['username'] = $username;
                $_SESSION['role'] = 'student';

                // Log successful login
                UserActivityLogger::logActivity($user['id'], 'login', "Student portal login successful from IP: " . $_SERVER['REMOTE_ADDR']);

                // Redirect to student dashboard
                header('Location: student/student_dashboard.php');
                exit;
            } else {
                // Log failed login attempt
                UserActivityLogger::logActivity(null, 'login_failed', "Failed login attempt for username: $username from IP: " . $_SERVER['REMOTE_ADDR']);
                $error = 'Invalid username or password.';
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - School Management System</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-image: url('img/dps.jpg');
            background-size: cover;
            background-position: center;
            background-repeat: no-repeat;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .top-bar {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 15px 0;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            z-index: 1000;
        }

        .top-bar .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .top-bar .logo {
            font-size: 1.8rem;
            font-weight: 700;
            color: #667eea;
            display: flex;
            align-items: center;
            text-shadow: 0 2px 4px rgba(102, 126, 234, 0.3);
            letter-spacing: 0.5px;
        }

        .top-bar .logo i {
            margin-right: 12px;
            font-size: 2rem;
            color: #764ba2;
            filter: drop-shadow(0 2px 4px rgba(118, 75, 162, 0.3));
        }

        .login-container {
            background: rgba(255, 255, 255, 0.98);
            backdrop-filter: blur(20px);
            padding: 40px;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
            width: 100%;
            max-width: 380px;
            border: 2px solid #667eea;
        }

        .login-header {
            text-align: center;
            margin-bottom: 30px;
        }

        .login-header .icon {
            margin-bottom: 15px;
        }

        .login-header .icon img {
            width: 8rem;
            height: 8rem;
            object-fit: contain;
        }

        .login-header h1 {
            color: #2d3748;
            margin-bottom: 8px;
            font-size: 2rem;
            font-weight: 600;
        }

        .login-header p {
            color: #718096;
            font-size: 0.95rem;
        }

        .form-group {
            margin-bottom: 20px;
            position: relative;
        }

        .form-group input {
            width: 100%;
            padding: 15px 15px 15px 45px;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            font-size: 16px;
            transition: all 0.3s ease;
            background: #f8fafc;
            outline: none;
        }

        .form-group input:focus {
            border-color: #667eea;
            background: white;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
            transform: translateY(-1px);
        }

        .form-group .input-icon {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #a0aec0;
            font-size: 1.1rem;
            transition: color 0.3s ease;
        }

        .form-group input:focus + .input-icon {
            color: #667eea;
        }

        .btn {
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
        }

        .btn:active {
            transform: translateY(0);
        }

        .error-message {
            padding: 12px 15px;
            margin-bottom: 20px;
            border-radius: 8px;
            background: #fed7d7;
            color: #c53030;
            border: 1px solid #fbb6ce;
            font-size: 14px;
        }

        .links-section {
            margin-top: 25px;
            text-align: center;
        }

        .back-link {
            color: #718096;
            text-decoration: none;
            font-size: 14px;
            transition: color 0.3s ease;
        }

        .back-link:hover {
            color: #667eea;
        }

        .forgot-password {
            color: #667eea;
            text-decoration: none;
            font-size: 14px;
            font-weight: 500;
            transition: color 0.3s ease;
        }

        .forgot-password:hover {
            color: #764ba2;
            text-decoration: underline;
        }

        .demo-info {
            margin-top: 25px;
            padding: 20px;
            background: linear-gradient(135deg, #f0f4ff 0%, #e6fffa 100%);
            border-radius: 12px;
            border-left: 4px solid #667eea;
        }

        .demo-info h4 {
            margin-bottom: 12px;
            color: #667eea;
            font-size: 16px;
            font-weight: 600;
        }

        .demo-info p {
            margin-bottom: 6px;
            font-size: 14px;
            color: #4a5568;
            font-weight: 500;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .login-container {
                margin: 20px;
                padding: 30px 25px;
            }

            .login-header h1 {
                font-size: 1.8rem;
            }

            .login-header .icon {
                font-size: 3rem;
            }

            .top-bar .container {
                padding: 0 15px;
            }

            .top-bar .logo {
                font-size: 1.3rem;
            }

            .top-bar .logo i {
                font-size: 1.5rem;
            }
        }

        @media (max-width: 480px) {
            body {
                padding: 10px;
            }

            .login-container {
                padding: 25px 20px;
            }

            .login-header h1 {
                font-size: 1.6rem;
            }

            .login-header .icon {
                font-size: 2.5rem;
            }

            .top-bar {
                padding: 12px 0;
            }

            .top-bar .logo {
                font-size: 1.2rem;
            }

            .top-bar .logo i {
                font-size: 1.4rem;
            }
        }
    </style>
</head>
<body>
    <!-- Top Navigation Bar -->
    <div class="top-bar">
        <div class="container">
            <div class="logo">
                <i class="fas fa-graduation-cap"></i>
                <span>Four River International School SIS</span>
            </div>
            <!-- Removed nav-links for Home, Library, Login -->
        </div>
    </div>

    <!-- Login Container -->
    <div class="login-container">
        <div class="login-header">
            <div class="icon"><img src="<?php echo htmlspecialchars($logo_path); ?>" alt="School Logo"></div>
            <h1>Student Portal</h1>
            <p>Please login to continue</p>
        </div>

        <?php if ($error): ?>
            <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>

        <form method="post" action="login.php">
            <?php echo CSRFProtection::getHiddenInput(); ?>
            <div class="form-group">
                <input type="text" id="username" name="username" placeholder="Username" required autofocus>
                <i class="fas fa-user input-icon"></i>
            </div>

            <div class="form-group">
                <input type="password" id="password" name="password" placeholder="Password" required>
                <i class="fas fa-lock input-icon"></i>
            </div>

            <button type="submit" class="btn"><i class="fas fa-sign-in-alt"></i> Login</button>
        </form>

        <div class="links-section">
            <a href="forgot_password.php" class="forgot-password"><i class="fas fa-key"></i> Forgot Password?</a>
        </div>

    </div>

    <script src="/school/js/script.js"></script>
</body>
</html>
