<?php
require_once '../config.php';

// Check if user is logged in and is librarian or admin
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['librarian', 'admin'])) {
    header('Location: ../login.php');
    exit();
}

$message = '';
$message_type = '';

// Handle return book
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['return_book'])) {
    $borrowing_id = intval($_POST['borrowing_id']);

    try {
        // Get borrowing details
        $stmt = $pdo->prepare("SELECT book_id FROM library_borrowings WHERE id = ?");
        $stmt->execute([$borrowing_id]);
        $borrowing = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($borrowing) {
            // Update borrowing status
            $update_borrowing = $pdo->prepare("UPDATE library_borrowings SET status = 'returned', return_date = CURDATE() WHERE id = ?");
            $update_borrowing->execute([$borrowing_id]);

            // Update available copies
            $update_book = $pdo->prepare("UPDATE library_books SET available_copies = available_copies + 1 WHERE id = ?");
            $update_book->execute([$borrowing['book_id']]);

            $message = "Book returned successfully!";
            $message_type = "success";
        }
    } catch (PDOException $e) {
        $message = "Error returning book: " . $e->getMessage();
        $message_type = "error";
    }
}

// Fetch overdue books
try {
    $stmt = $pdo->prepare("
        SELECT lb.*, bk.title, bk.author, u.username,
               DATEDIFF(CURDATE(), lb.due_date) as days_overdue
        FROM library_borrowings lb
        JOIN library_books bk ON lb.book_id = bk.id
        JOIN users u ON lb.user_id = u.id
        WHERE lb.status IN ('borrowed', 'overdue') AND lb.due_date < CURDATE()
        ORDER BY lb.due_date ASC
    ");
    $stmt->execute();
    $overdue_books = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $overdue_books = [];
    $message = "Error loading overdue books: " . $e->getMessage();
    $message_type = "error";
}

$page_title = "Overdue Books - Library";
include '../includes/library_header.php';
?>

<div class="container-fluid">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2><i class="fas fa-exclamation-triangle"></i> Overdue Books</h2>
        <div class="text-muted">
            Total Overdue: <span class="badge badge-danger"><?php echo count($overdue_books); ?></span>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
            <?php echo htmlspecialchars($message); ?>
            <button type="button" class="close" data-dismiss="alert">
                <span>&times;</span>
            </button>
        </div>
    <?php endif; ?>

    <?php if (empty($overdue_books)): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> No overdue books at this time.
        </div>
    <?php else: ?>
        <div class="table-responsive">
            <table class="table table-striped table-hover">
                <thead class="thead-dark">
                    <tr>
                        <th>Book Title</th>
                        <th>Author</th>
                        <th>Borrower</th>
                        <th>Due Date</th>
                        <th>Days Overdue</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($overdue_books as $item): ?>
                        <tr class="<?php echo $item['days_overdue'] > 30 ? 'table-danger' : 'table-warning'; ?>">
                            <td><?php echo htmlspecialchars($item['title']); ?></td>
                            <td><?php echo htmlspecialchars($item['author']); ?></td>
                            <td><?php echo htmlspecialchars($item['username']); ?></td>
                            <td><?php echo date('M d, Y', strtotime($item['due_date'])); ?></td>
                            <td>
                                <span class="badge badge-danger"><?php echo $item['days_overdue']; ?> days</span>
                            </td>
                            <td>
                                <button class="btn btn-sm btn-success" onclick="returnBook(<?php echo $item['id']; ?>, '<?php echo htmlspecialchars($item['title']); ?>')">
                                    <i class="fas fa-undo"></i> Return
                                </button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<!-- Return Book Modal -->
<div class="modal fade" id="returnBookModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Return Book</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <p>Are you sure you want to mark this book as returned?</p>
                    <p><strong>Book:</strong> <span id="return-book-title"></span></p>
                    <input type="hidden" id="return_borrowing_id" name="borrowing_id">
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="return_book" class="btn btn-success">Confirm Return</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function returnBook(borrowingId, bookTitle) {
    document.getElementById('return_borrowing_id').value = borrowingId;
    document.getElementById('return-book-title').textContent = bookTitle;
    $('#returnBookModal').modal('show');
}
</script>

<?php include '../includes/library_footer.php'; ?>
