<?php
require_once '../config.php';
require_once 'settings.php';

// Check if user is logged in and is admin or librarian
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['admin', 'librarian'])) {
    header('Location: ../login.php');
    exit();
}

$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['update_settings'])) {
        $new_borrowing_period = intval($_POST['borrowing_period_days']);
        $new_max_books_per_user = intval($_POST['max_books_per_user']);
        $new_fine_per_day = floatval($_POST['fine_per_day']);

        // Validate inputs
        if ($new_borrowing_period < 1 || $new_borrowing_period > 365) {
            $message = "Borrowing period must be between 1 and 365 days.";
            $message_type = "error";
        } elseif ($new_max_books_per_user < 1 || $new_max_books_per_user > 50) {
            $message = "Maximum books per user must be between 1 and 50.";
            $message_type = "error";
        } elseif ($new_fine_per_day < 0 || $new_fine_per_day > 100) {
            $message = "Fine per day must be between 0 and 100.";
            $message_type = "error";
        } else {
            // Update settings file
            $settings_content = "<?php\n";
            $settings_content .= "// Library settings configuration\n\n";
            $settings_content .= "// Borrowing period length in days\n";
            $settings_content .= "// This value will be used to set the due date when issuing a book\n";
            $settings_content .= "\$library_borrowing_period_days = $new_borrowing_period;\n\n";
            $settings_content .= "// Maximum books a user can borrow at once\n";
            $settings_content .= "\$library_max_books_per_user = $new_max_books_per_user;\n\n";
            $settings_content .= "// Fine per day for overdue books\n";
            $settings_content .= "\$library_fine_per_day = $new_fine_per_day;\n\n";
            $settings_content .= "// You can add other library-related settings here as needed\n";
            $settings_content .= "?>\n";

            if (file_put_contents('settings.php', $settings_content)) {
                $message = "Settings updated successfully!";
                $message_type = "success";

                // Reload settings
                require_once 'settings.php';
            } else {
                $message = "Error updating settings file.";
                $message_type = "error";
            }
        }
    }
}

$page_title = "Library Settings";
include '../includes/library_header.php';
?>
<link rel="stylesheet" href="../css/library_custom.css">

<style>
/* Page Header */
.container-fluid h2 {
    color: #8e44ad;
    font-weight: 700;
    margin-bottom: 30px;
    font-size: 2em;
}

/* Settings Form */
.settings-form {
    background: white;
    border-radius: 15px;
    box-shadow: 0 20px 40px rgba(0,0,0,0.1);
    padding: 30px;
    margin-bottom: 30px;
}

.settings-form .form-group {
    margin-bottom: 25px;
}

.settings-form label {
    font-weight: 600;
    color: #495057;
    margin-bottom: 8px;
    display: block;
}

.settings-form .form-control {
    border-radius: 8px;
    border: 2px solid #e9ecef;
    padding: 12px 15px;
    font-size: 14px;
    transition: all 0.3s ease;
    background-color: #fff;
    color: #495057;
}

.settings-form .form-control:focus {
    border-color: #8e44ad;
    box-shadow: 0 0 0 0.2rem rgba(142, 68, 173, 0.25);
    outline: none;
}

.settings-form .form-text {
    color: #6c757d;
    font-size: 0.875em;
    margin-top: 5px;
}

/* Buttons */
.btn-update-settings {
    border-radius: 25px;
    padding: 12px 30px;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(142, 68, 173, 0.3);
    border: none;
    color: white;
    background: linear-gradient(135deg, #8e44ad 0%, #9b59b6 100%);
    cursor: pointer;
}

.btn-update-settings:hover {
    background: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(155, 89, 182, 0.4);
}

/* Alert Messages */
.alert {
    border-radius: 10px;
    border: none;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}

.alert-success {
    background-color: #d4edda;
    color: #155724;
}

.alert-danger {
    background-color: #f8d7da;
    color: #721c24;
}

/* Current Settings Display */
.current-settings {
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border-radius: 15px;
    padding: 25px;
    margin-bottom: 30px;
    border: 1px solid #dee2e6;
}

.current-settings h4 {
    color: #8e44ad;
    margin-bottom: 20px;
    font-weight: 700;
}

.setting-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 10px 0;
    border-bottom: 1px solid #dee2e6;
}

.setting-item:last-child {
    border-bottom: none;
}

.setting-label {
    font-weight: 600;
    color: #495057;
}

.setting-value {
    color: #8e44ad;
    font-weight: 700;
}

/* Responsive Design */
@media (max-width: 768px) {
    .container-fluid h2 {
        font-size: 1.5em;
        margin-bottom: 20px;
    }

    .settings-form {
        padding: 20px;
    }

    .current-settings {
        padding: 20px;
    }

    .setting-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 5px;
    }

    .btn-update-settings {
        width: 100%;
        margin-top: 15px;
    }
}
</style>

<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2><i class="fas fa-cog"></i> Library Settings</h2>
            </div>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="close" data-dismiss="alert">
                        <span>&times;</span>
                    </button>
                </div>
            <?php endif; ?>

            <!-- Current Settings -->
            <div class="current-settings">
                <h4><i class="fas fa-info-circle"></i> Current Settings</h4>
                <div class="setting-item">
                    <span class="setting-label">Borrowing Period:</span>
                    <span class="setting-value"><?php echo $library_borrowing_period_days; ?> days</span>
                </div>
                <div class="setting-item">
                    <span class="setting-label">Maximum Books per User:</span>
                    <span class="setting-value"><?php echo isset($library_max_books_per_user) ? $library_max_books_per_user : 'Not set'; ?></span>
                </div>
                <div class="setting-item">
                    <span class="setting-label">Fine per Day (Overdue):</span>
                    <span class="setting-value">₵<?php echo isset($library_fine_per_day) ? number_format($library_fine_per_day, 2) : 'Not set'; ?></span>
                </div>
            </div>

            <!-- Settings Form -->
            <div class="settings-form">
                <h4><i class="fas fa-edit"></i> Update Settings</h4>
                <form method="POST">
                    <div class="form-group">
                        <label for="borrowing_period_days">Borrowing Period (Days) *</label>
                        <input type="number" class="form-control" id="borrowing_period_days" name="borrowing_period_days"
                               value="<?php echo $library_borrowing_period_days; ?>" min="1" max="365" required>
                        <small class="form-text text-muted">Number of days books can be borrowed (1-365 days)</small>
                    </div>

                    <div class="form-group">
                        <label for="max_books_per_user">Maximum Books per User *</label>
                        <input type="number" class="form-control" id="max_books_per_user" name="max_books_per_user"
                               value="<?php echo isset($library_max_books_per_user) ? $library_max_books_per_user : 5; ?>" min="1" max="50" required>
                        <small class="form-text text-muted">Maximum number of books a user can borrow at once (1-50)</small>
                    </div>

                    <div class="form-group">
                        <label for="fine_per_day">Fine per Day (Overdue) *</label>
                        <input type="number" class="form-control" id="fine_per_day" name="fine_per_day" step="0.01"
                               value="<?php echo isset($library_fine_per_day) ? $library_fine_per_day : 0.50; ?>" min="0" max="100" required>
                        <small class="form-text text-muted">Daily fine for overdue books in cedis (0-100)</small>
                    </div>

                    <button type="submit" name="update_settings" class="btn btn-update-settings">
                        <i class="fas fa-save"></i> Update Settings
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<?php include '../includes/library_footer.php'; ?>
