<?php
require_once '../config.php';

header('Content-Type: application/json');

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Check authentication
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit;
}

// Check permissions - only admins and accounts users can view PO details
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Access denied']);
    exit;
}

$po_id = isset($_GET['po_id']) ? (int)$_GET['po_id'] : null;

if (!$po_id) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Purchase order ID required']);
    exit;
}

try {
    // Get purchase order details
    $stmt = $pdo->prepare("
        SELECT po.*,
               s.name AS supplier_name, s.contact_person, s.phone, s.email, s.address,
               u.username AS created_by_name,
               (
                   SELECT COUNT(*) FROM purchase_order_items poi WHERE poi.purchase_order_id = po.id
               ) as item_count,
               (
                   SELECT COALESCE(SUM(pop.amount), 0) FROM purchase_order_payments pop WHERE pop.purchase_order_id = po.id
               ) as total_paid
        FROM purchase_orders po
        JOIN suppliers s ON po.supplier_id = s.id
        LEFT JOIN users u ON po.created_by = u.id
        WHERE po.id = ?
    ");
    $stmt->execute([$po_id]);
    $po = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$po) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Purchase order not found']);
        exit;
    }

    // Get purchase order items
    $stmt = $pdo->prepare("
        SELECT poi.*,
               bc.name AS category_name
        FROM purchase_order_items poi
        LEFT JOIN budget_categories bc ON poi.category = bc.id
        WHERE poi.purchase_order_id = ?
        ORDER BY poi.id ASC
    ");
    $stmt->execute([$po_id]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get payment history
    $stmt = $pdo->prepare("
        SELECT pop.*,
               u.username AS recorded_by_name,
               ca.name AS cash_account_name
        FROM purchase_order_payments pop
        LEFT JOIN users u ON pop.recorded_by = u.id
        LEFT JOIN cash_accounts ca ON pop.cash_account_id = ca.id
        WHERE pop.purchase_order_id = ?
        ORDER BY pop.payment_date DESC
    ");
    $stmt->execute([$po_id]);
    $payments = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Calculate outstanding balance
    $outstanding_balance = $po['total_amount'] - $po['total_paid'];

    // Format the response
    $response = [
        'success' => true,
        'po' => [
            'id' => $po['id'],
            'po_number' => $po['po_number'],
            'supplier' => [
                'id' => $po['supplier_id'],
                'name' => $po['supplier_name'],
                'contact_person' => $po['contact_person'],
                'phone' => $po['phone'],
                'email' => $po['email'],
                'address' => $po['address']
            ],
            'total_amount' => floatval($po['total_amount']),
            'total_paid' => floatval($po['total_paid']),
            'outstanding_balance' => floatval($outstanding_balance),
            'expected_delivery_date' => $po['expected_delivery_date'],
            'status' => $po['status'],
            'created_by' => [
                'id' => $po['created_by'],
                'name' => $po['created_by_name']
            ],
            'created_at' => $po['created_at'],
            'updated_at' => $po['updated_at'],
            'item_count' => intval($po['item_count'])
        ],
        'items' => array_map(function($item) {
            return [
                'id' => $item['id'],
                'description' => $item['description'],
                'quantity_ordered' => floatval($item['quantity_ordered']),
                'unit_price' => floatval($item['unit_price']),
                'total_price' => floatval($item['quantity_ordered'] * $item['unit_price']),
                'category' => $item['category'],
                'category_name' => $item['category_name']
            ];
        }, $items),
        'payments' => array_map(function($payment) {
            return [
                'id' => $payment['id'],
                'amount' => floatval($payment['amount']),
                'payment_date' => $payment['payment_date'],
                'payment_method' => $payment['payment_method'],
                'reference_number' => $payment['reference_number'],
                'notes' => $payment['notes'],
                'recorded_by' => $payment['recorded_by_name'],
                'cash_account_name' => $payment['cash_account_name']
            ];
        }, $payments)
    ];

    echo json_encode($response);

} catch (PDOException $e) {
    error_log("Database error in get_po_details.php: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database error occurred']);
}
?>
