<?php
require_once '../config.php';
require_once '../includes/user_activity_logger.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}

// Check if user has permission to manage staff users
require_once '../includes/enhanced_access_control_fixed.php';
requirePermission('user_management');

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'update_staff':
                updateStaffUser();
                break;
            case 'deactivate_staff':
                deactivateStaffUser();
                break;
            case 'activate_staff':
                activateStaffUser();
                break;
        }
    }
}

// Update staff user
function updateStaffUser() {
    global $pdo;

    try {
        // Update staff_accounts table for username and password only
        $update_fields_users = ['username = ?'];
        $params_users = [$_POST['username']];

        // Update password only if provided
        if (!empty($_POST['password'])) {
            // Validate strong password
            if (!validateStrongPassword($_POST['password'])) {
                $_SESSION['error_message'] = 'Password must be at least 8 characters long and contain at least one uppercase letter, one lowercase letter, one number, and one special character.';
                return;
            }
            $update_fields_users[] = 'password = ?';
            $params_users[] = password_hash($_POST['password'], PASSWORD_DEFAULT);
        }

        $params_users[] = $_POST['user_id'];

        $stmt_users = $pdo->prepare("UPDATE staff_accounts SET " . implode(', ', $update_fields_users) . " WHERE id = ?");
        $stmt_users->execute($params_users);

        // Log staff user update activity
        $password_changed = !empty($_POST['password']) ? ' (password changed)' : '';
        $description = "Updated staff user ID: {$_POST['user_id']} - Username: {$_POST['username']}{$password_changed}";
        UserActivityLogger::logActivity($_SESSION['user_id'], 'update_staff_user', $description);

        $_SESSION['success_message'] = 'Staff user updated successfully!';
        header('Location: staff_accounts.php');
        exit;
    } catch (PDOException $e) {
        $_SESSION['error_message'] = 'Error updating staff user: ' . $e->getMessage();
    }
}

// Deactivate staff user
function deactivateStaffUser() {
    global $pdo;

    try {
        $stmt = $pdo->prepare("UPDATE staff_accounts SET status = 'inactive' WHERE id = ?");
        $stmt->execute([$_POST['user_id']]);

        // Log deactivation activity
        $description = "Deactivated staff user ID: {$_POST['user_id']}";
        UserActivityLogger::logActivity($_SESSION['user_id'], 'deactivate_staff_user', $description);

        $_SESSION['success_message'] = 'Staff user deactivated successfully!';
        header('Location: staff_accounts.php');
        exit;
    } catch (PDOException $e) {
        $_SESSION['error_message'] = 'Error deactivating staff user: ' . $e->getMessage();
    }
}

// Activate staff user
function activateStaffUser() {
    global $pdo;

    try {
        $stmt = $pdo->prepare("UPDATE staff_accounts SET status = 'active' WHERE id = ?");
        $stmt->execute([$_POST['user_id']]);

        // Log activation activity
        $description = "Activated staff user ID: {$_POST['user_id']}";
        UserActivityLogger::logActivity($_SESSION['user_id'], 'activate_staff_user', $description);

        $_SESSION['success_message'] = 'Staff user activated successfully!';
        header('Location: staff_accounts.php');
        exit;
    } catch (PDOException $e) {
        $_SESSION['error_message'] = 'Error activating staff user: ' . $e->getMessage();
    }
}

// Validate strong password
function validateStrongPassword($password) {
    // At least 8 characters, one uppercase, one lowercase, one number, one special character
    return preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/', $password);
}

// New code with search functionality added
$per_page = 10; // Staff per page
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$search_term = isset($_GET['search']) ? trim($_GET['search']) : '';
$offset = ($page - 1) * $per_page;

try {
    // Prepare base count query with optional WHERE clause for search
    $count_query = "SELECT COUNT(*) as total FROM staff_accounts su LEFT JOIN staff s ON su.id = s.user_id";
    $count_params = [];
    if ($search_term !== '') {
        $count_query .= " WHERE su.username LIKE ? OR s.first_name LIKE ? OR s.last_name LIKE ? OR s.role LIKE ?";
        $like_search = '%' . $search_term . '%';
        $count_params = [$like_search, $like_search, $like_search, $like_search];
    }
    $count_stmt = $pdo->prepare($count_query);
    $count_stmt->execute($count_params);
    $total_staff = $count_stmt->fetch()['total'];
    $total_pages = ceil($total_staff / $per_page);

    // Prepare base data query with optional WHERE clause for search
    $data_query = "
        SELECT su.id, su.username, su.created_at, su.status, s.first_name, s.last_name, s.role
        FROM staff_accounts su
        LEFT JOIN staff s ON su.id = s.user_id
    ";
    $data_params = [];
    if ($search_term !== '') {
        $data_query .= " WHERE su.username LIKE ? OR s.first_name LIKE ? OR s.last_name LIKE ? OR s.role LIKE ?";
        $like_search = '%' . $search_term . '%';
        $data_params = [$like_search, $like_search, $like_search, $like_search];
    }
    $data_query .= " ORDER BY su.created_at DESC LIMIT ? OFFSET ?";

    $data_stmt = $pdo->prepare($data_query);
    // Bind data params plus pagination params
    $bind_index = 1;
    foreach ($data_params as $param) {
        $data_stmt->bindValue($bind_index++, $param, PDO::PARAM_STR);
    }
    $data_stmt->bindValue($bind_index++, $per_page, PDO::PARAM_INT);
    $data_stmt->bindValue($bind_index++, $offset, PDO::PARAM_INT);
    $data_stmt->execute();
    $staff = $data_stmt->fetchAll();
} catch (PDOException $e) {
    $staff = [];
    $total_pages = 1;
    $_SESSION['error_message'] = 'Error fetching staff users: ' . $e->getMessage();
}

include '../includes/admin_header.php';
?>

<div class="staff-users-container">
    <div class="page-header">
        <h2>👥 Staff Accounts Management</h2>
        <p>Manage staff user accounts and their login credentials</p>
    </div>

    <!-- Success/Error Messages -->
    <?php if (isset($_SESSION['success_message'])): ?>
        <div class="alert alert-success">
            <?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?>
        </div>
    <?php endif; ?>

    <?php if (isset($_SESSION['error_message'])): ?>
        <div class="alert alert-error">
            <?php echo $_SESSION['error_message']; unset($_SESSION['error_message']); ?>
        </div>
    <?php endif; ?>

    <!-- Staff Users Table -->
    <div class="user-card">
        <div class="user-header">
            <h3 class="user-title">Staff Accounts</h3>
            <div class="table-filters">
                <form method="get" class="table-search-container" action="staff_accounts.php">
                    <input type="text" name="search" class="table-search" placeholder="Search staff..." value="<?php echo htmlspecialchars($search_term); ?>">
                    <button type="submit" style="margin-left: 5px; padding: 7px 12px;">Search</button>
                </form>
            </div>
        </div>

        <div class="table-container">
            <table class="user-table" id="staffTable">
                <thead>
                    <tr>
                        <th class="sortable">First Name</th>
                        <th class="sortable">Last Name</th>
                        <th class="sortable">Username</th>
                        <th class="sortable">Role</th>
                        <th class="sortable">Created Date</th>
                        <th class="sortable">Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($staff as $member): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($member['first_name'] ?? ''); ?></td>
                            <td><?php echo htmlspecialchars($member['last_name'] ?? ''); ?></td>
                            <td><?php echo htmlspecialchars($member['username']); ?></td>
                            <td>
                                <span class="role-badge <?php echo strtolower($member['role']); ?>">
                                    <?php echo ucfirst($member['role']); ?>
                                </span>
                            </td>
                            <td><?php echo date('M d, Y', strtotime($member['created_at'])); ?></td>
                            <td>
                                <span class="status-badge <?php echo strtolower($member['status'] ?? 'active'); ?>">
                                    <?php echo ucfirst($member['status'] ?? 'active'); ?>
                                </span>
                            </td>
                            <td>
                                <div class="action-buttons">
                                    <button onclick="editStaff(<?php echo htmlspecialchars(json_encode($member)); ?>)"
                                            class="btn-user btn-warning" style="font-size: 0.8em; padding: 5px 10px;">
                                        Edit
                                    </button>
                                    <?php if (($member['status'] ?? 'active') === 'active'): ?>
                                        <button onclick="deactivateStaff(<?php echo $member['id']; ?>)"
                                                class="btn-user btn-danger" style="font-size: 0.8em; padding: 5px 10px;">
                                            Deactivate
                                        </button>
                                    <?php else: ?>
                                        <button onclick="activateStaff(<?php echo $member['id']; ?>)"
                                                class="btn-user btn-success" style="font-size: 0.8em; padding: 5px 10px;">
                                            Activate
                                        </button>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
            <div class="pagination-container">
                <div class="pagination">
                    <?php
                    $base_url = '?page=';
                    ?>

                    <?php if ($page > 1): ?>
                        <a href="<?php echo $base_url . ($page - 1); ?>" class="page-link">&laquo; Previous</a>
                    <?php endif; ?>

                    <?php
                    $start_page = max(1, $page - 2);
                    $end_page = min($total_pages, $page + 2);

                    if ($start_page > 1): ?>
                        <a href="<?php echo $base_url . '1'; ?>" class="page-link">1</a>
                        <?php if ($start_page > 2): ?>
                            <span class="page-dots">...</span>
                        <?php endif; ?>
                    <?php endif; ?>

                    <?php for ($i = $start_page; $i <= $end_page; $i++): ?>
                        <a href="<?php echo $base_url . $i; ?>" class="page-link <?php echo $i === $page ? 'active' : ''; ?>">
                            <?php echo $i; ?>
                        </a>
                    <?php endfor; ?>

                    <?php if ($end_page < $total_pages): ?>
                        <?php if ($end_page < $total_pages - 1): ?>
                            <span class="page-dots">...</span>
                        <?php endif; ?>
                        <a href="<?php echo $base_url . $total_pages; ?>" class="page-link"><?php echo $total_pages; ?></a>
                    <?php endif; ?>

                    <?php if ($page < $total_pages): ?>
                        <a href="<?php echo $base_url . ($page + 1); ?>" class="page-link">Next &raquo;</a>
                    <?php endif; ?>
                </div>
                <div class="pagination-info">
                    Showing page <?php echo $page; ?> of <?php echo $total_pages; ?> (<?php echo $total_staff; ?> total staff)
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Edit Staff Modal -->
<div id="editStaffModal" class="modal">
    <div class="modal-backdrop"></div>
    <div class="modal-content">
        <div class="modal-header">
            <div class="modal-title">
                <i class="fas fa-user-edit"></i>
                <h3>Edit Staff User</h3>
            </div>
            <button type="button" class="modal-close" onclick="closeEditModal()" aria-label="Close modal">
                <i class="fas fa-times"></i>
            </button>
        </div>

        <div class="modal-body">
            <form method="POST" class="user-form" id="editStaffForm">
                <input type="hidden" name="action" value="update_staff">
                <input type="hidden" name="user_id" id="edit_user_id">

                <div class="form-section">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="edit_username" class="form-label">
                                <i class="fas fa-at"></i>
                                Username *
                            </label>
                            <div class="input-wrapper">
                                <input type="text" id="edit_username" name="username" required
                                       class="form-input" placeholder="Enter username">
                                <div class="input-icon">
                                    <i class="fas fa-at"></i>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label for="edit_password" class="form-label">
                                <i class="fas fa-lock"></i>
                                New Password
                            </label>
                            <div class="input-wrapper">
                                <input type="password" id="edit_password" name="password"
                                       class="form-input" placeholder="Leave blank to keep current">
                                <div class="input-icon">
                                    <i class="fas fa-key"></i>
                                </div>
                            </div>
                            <small class="form-hint">
                                Password must be at least 8 characters with uppercase, lowercase, number, and special character
                            </small>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label for="edit_password_confirm" class="form-label">
                                <i class="fas fa-lock"></i>
                                Confirm New Password
                            </label>
                            <div class="input-wrapper">
                                <input type="password" id="edit_password_confirm" name="password_confirm"
                                       class="form-input" placeholder="Confirm new password">
                                <div class="input-icon">
                                    <i class="fas fa-key"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="modal-footer">
                    <div class="form-actions">
                        <button type="button" onclick="closeEditModal()" class="btn-modal btn-secondary">
                            <i class="fas fa-times"></i>
                            Cancel
                        </button>
                        <button type="submit" class="btn-modal btn-primary">
                            <i class="fas fa-save"></i>
                            Update Staff
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
    .staff-users-container {
        padding: 20px;
    }

    .page-header {
        margin-bottom: 30px;
    }

    .page-header h2 {
        color: #2c3e50;
        margin-bottom: 10px;
        font-size: 2em;
    }

    .page-header p {
        color: #7f8c8d;
        margin: 0;
        font-size: 1.1em;
    }

    .table-search-container {
        display: flex;
        justify-content: flex-end;
        margin-bottom: 15px;
    }

    .table-search {
        padding: 8px 12px;
        border: 1px solid #ddd;
        border-radius: 5px;
        font-size: 0.9em;
        width: 250px;
    }

    .action-buttons {
        display: flex;
        gap: 5px;
        flex-wrap: wrap;
    }

    .status-badge {
        padding: 4px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: 500;
    }

    .status-badge.active { background-color: #2ecc71; color: white; }
    .status-badge.inactive { background-color: #e74c3c; color: white; }

    .role-badge {
        padding: 4px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: 500;
        text-transform: capitalize;
    }

    .role-badge.admin { background-color: #e74c3c; color: white; }
    .role-badge.super_admin { background-color: #9b59b6; color: white; }
    .role-badge.teacher { background-color: #3498db; color: white; }
    .role-badge.head_teacher { background-color: #f39c12; color: white; }
    .role-badge.accounts { background-color: #27ae60; color: white; }
    .role-badge.librarian { background-color: #8e44ad; color: white; }
    .role-badge.non_teaching { background-color: #95a5a6; color: white; }
    .role-badge.director { background-color: #d35400; color: white; }

    /* Enhanced Modal Styles */
    .modal {
        position: fixed;
        z-index: 9999;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        display: flex;
        align-items: center;
        justify-content: center;
        opacity: 0;
        visibility: hidden;
        transition: all 0.3s ease;
    }

    .modal.show {
        opacity: 1;
        visibility: visible;
    }

    .modal-backdrop {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.4);
        backdrop-filter: blur(1px);
    }

    .modal-content {
        background: white;
        border-radius: 16px;
        width: 90%;
        max-width: 600px;
        max-height: 90vh;
        overflow: hidden;
        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        transform: scale(0.9) translateY(-20px);
        transition: all 0.3s ease;
        position: relative;
        z-index: 10000;
    }

    .modal.show .modal-content {
        transform: scale(1) translateY(0);
    }

    .modal-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        padding: 24px 30px;
        display: flex;
        justify-content: space-between;
        align-items: center;
        border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    }

    .modal-title {
        display: flex;
        align-items: center;
        gap: 12px;
        color: white;
    }

    .modal-title i {
        font-size: 1.5em;
        opacity: 0.9;
    }

    .modal-title h3 {
        margin: 0;
        font-size: 1.4em;
        font-weight: 600;
    }

    .modal-close {
        background: rgba(255, 255, 255, 0.2);
        border: none;
        color: white;
        font-size: 1.2em;
        width: 40px;
        height: 40px;
        border-radius: 50%;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.3s ease;
    }

    .modal-close:hover {
        background: rgba(255, 255, 255, 0.3);
        transform: rotate(90deg);
    }

    .modal-body {
        padding: 30px;
        max-height: calc(90vh - 140px);
        overflow-y: auto;
    }

    .modal-footer {
        padding: 20px 30px;
        background: #f8f9fa;
        border-top: 1px solid #e9ecef;
        display: flex;
        justify-content: flex-end;
    }

    /* Enhanced Form Styles for Modal */
    .form-section {
        margin-bottom: 0;
    }

    .form-label {
        display: flex;
        align-items: center;
        gap: 8px;
        font-weight: 600;
        color: #374151;
        margin-bottom: 8px;
        font-size: 0.95em;
    }

    .form-label i {
        color: #6b7280;
        width: 16px;
    }

    .input-wrapper {
        position: relative;
        display: flex;
        align-items: center;
    }

    .form-input {
        width: 100%;
        padding: 12px 16px;
        border: 2px solid #e5e7eb;
        border-radius: 8px;
        font-size: 1em;
        transition: all 0.3s ease;
        background: white;
    }

    .form-input:focus {
        outline: none;
        border-color: #667eea;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }

    .input-icon {
        position: absolute;
        right: 12px;
        color: #9ca3af;
        pointer-events: none;
        font-size: 0.9em;
    }

    .form-hint {
        display: block;
        margin-top: 6px;
        font-size: 0.85em;
        color: #6b7280;
        font-style: italic;
    }

    /* Modal Button Styles */
    .btn-modal {
        padding: 12px 24px;
        border: none;
        border-radius: 8px;
        font-size: 0.95em;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        display: inline-flex;
        align-items: center;
        gap: 8px;
        min-width: 120px;
        justify-content: center;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .btn-modal:focus {
        outline: none;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.3);
    }

    .btn-primary {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
    }

    .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    }

    .btn-secondary {
        background: #6b7280;
        color: white;
        box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
    }

    .btn-secondary:hover {
        background: #4b5563;
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(107, 114, 128, 0.4);
    }

    /* Form Actions in Modal */
    .form-actions {
        display: flex;
        gap: 12px;
        justify-content: flex-end;
        margin: 0;
    }

    /* Pagination Styles */
    .pagination-container {
        margin-top: 20px;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 10px;
    }

    .pagination {
        display: flex;
        justify-content: center;
        align-items: center;
        gap: 5px;
        flex-wrap: wrap;
    }

    .page-link {
        padding: 8px 12px;
        border: 1px solid #ddd;
        background-color: #fff;
        color: #333;
        text-decoration: none;
        border-radius: 4px;
        transition: all 0.3s ease;
        font-size: 0.9em;
        min-width: 35px;
        text-align: center;
    }

    .page-link:hover {
        background-color: #f8f9fa;
        border-color: #adb5bd;
    }

    .page-link.active {
        background-color: #007bff;
        color: white;
        border-color: #007bff;
    }

    .page-dots {
        padding: 8px 4px;
        color: #6c757d;
        font-size: 0.9em;
    }

    .pagination-info {
        font-size: 0.85em;
        color: #6c757d;
        margin-top: 5px;
    }

    @media (max-width: 768px) {
        .table-search {
            width: 100%;
        }

        .modal-content {
            width: 95%;
            margin: 10% auto;
        }

        .modal-backdrop {
            background: rgba(0, 0, 0, 0.3);
            backdrop-filter: none;
        }

        .action-buttons {
            flex-direction: column;
        }

        .pagination {
            gap: 2px;
        }

        .page-link {
            padding: 6px 8px;
            font-size: 0.8em;
            min-width: 30px;
        }
    }
</style>

<script>
    // Edit staff function
    function editStaff(staff) {
        document.getElementById('edit_user_id').value = staff.id;
        document.getElementById('edit_username').value = staff.username;

        const modal = document.getElementById('editStaffModal');
        modal.classList.add('show');
        document.body.style.overflow = 'hidden'; // Prevent background scrolling
    }

    // Close edit modal
    function closeEditModal() {
        const modal = document.getElementById('editStaffModal');
        modal.classList.remove('show');
        document.body.style.overflow = ''; // Restore scrolling

        // Clear form after animation completes
        setTimeout(() => {
            document.getElementById('editStaffForm').reset();
        }, 300);
    }

    // Close modal when clicking outside
    window.onclick = function(event) {
        const modal = document.getElementById('editStaffModal');
        if (event.target === modal || event.target.classList.contains('modal-backdrop')) {
            closeEditModal();
        }
    }

    // Close modal on Escape key
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape') {
            const modal = document.getElementById('editStaffModal');
            if (modal.classList.contains('show')) {
                closeEditModal();
            }
        }
    });

    // Search functionality with debouncing
    let searchTimeout;
    document.querySelector('.table-search').addEventListener('input', function() {
        clearTimeout(searchTimeout);
        const searchTerm = this.value.toLowerCase();
        const rows = document.querySelectorAll('#staffTable tbody tr');

        searchTimeout = setTimeout(() => {
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(searchTerm) ? '' : 'none';
            });
        }, 300); // Debounce search for better performance
    });

    // Password strength validation
    document.getElementById('edit_password').addEventListener('input', function() {
        const password = this.value;
        const strengthIndicator = document.getElementById('password-strength');

        if (password.length === 0) {
            this.style.borderColor = '#e5e7eb';
            return;
        }

        // Strong password regex: at least 8 chars, uppercase, lowercase, number, special char
        const strongRegex = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/;

        if (strongRegex.test(password)) {
            this.style.borderColor = '#10b981'; // Green for strong
        } else {
            this.style.borderColor = '#ef4444'; // Red for weak
        }
    });

    // Form validation feedback
    document.getElementById('editStaffForm').addEventListener('submit', function(e) {
        const requiredFields = this.querySelectorAll('[required]');
        let isValid = true;

        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                field.style.borderColor = '#ef4444';
                isValid = false;
            } else {
                field.style.borderColor = '#10b981';
            }
        });

        // Validate password if provided
        const passwordField = document.getElementById('edit_password');
        const confirmPasswordField = document.getElementById('edit_password_confirm');
        if (passwordField.value) {
            if (!/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/.test(passwordField.value)) {
                passwordField.style.borderColor = '#ef4444';
                isValid = false;
                alert('Password must be at least 8 characters long and contain at least one uppercase letter, one lowercase letter, one number, and one special character.');
            } else if (passwordField.value !== confirmPasswordField.value) {
                confirmPasswordField.style.borderColor = '#ef4444';
                isValid = false;
                alert('Passwords do not match.');
            } else {
                confirmPasswordField.style.borderColor = '#10b981';
            }
        }

        if (!isValid) {
            e.preventDefault();
            // Show error message
            const errorMsg = document.createElement('div');
            errorMsg.className = 'alert alert-error';
            errorMsg.textContent = 'Please fill in all required fields and ensure password meets requirements.';
            errorMsg.style.position = 'fixed';
            errorMsg.style.top = '20px';
            errorMsg.style.right = '20px';
            errorMsg.style.zIndex = '9999';
            document.body.appendChild(errorMsg);

            setTimeout(() => {
                errorMsg.remove();
            }, 3000);
        }
    });

    // Reset field border colors on input
    document.querySelectorAll('.form-input').forEach(field => {
        field.addEventListener('input', function() {
            this.style.borderColor = '#e5e7eb';
        });
    });

    // Deactivate staff function
    function deactivateStaff(userId) {
        if (confirm('Are you sure you want to deactivate this staff user?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="deactivate_staff">
                <input type="hidden" name="user_id" value="${userId}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }

    // Activate staff function
    function activateStaff(userId) {
        if (confirm('Are you sure you want to activate this staff user?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="activate_staff">
                <input type="hidden" name="user_id" value="${userId}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
</script>

<?php
include '../includes/admin_footer.php';
?>
