<?php
require_once '../config.php';
require_once '../includes/access_control.php';

// requireFeatureAccess('facilities_management'); // Temporarily disabled

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_facility'])) {
        $name = trim($_POST['name']);
        $category_id = intval($_POST['category_id']);
        $location = trim($_POST['location']);
        $capacity = intval($_POST['capacity']);
        $description = trim($_POST['description']);
        $amenities = trim($_POST['amenities']);
        $rules = trim($_POST['rules']);
        $requires_approval = isset($_POST['requires_approval']) ? 1 : 0;
        $max_booking_hours = intval($_POST['max_booking_hours']);
        $advance_booking_days = intval($_POST['advance_booking_days']);

        try {
            $stmt = $pdo->prepare("INSERT INTO facilities (name, category_id, location, capacity, description, amenities, rules, requires_approval, max_booking_hours, advance_booking_days, added_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$name, $category_id, $location, $capacity, $description, $amenities, $rules, $requires_approval, $max_booking_hours, $advance_booking_days, $_SESSION['user_id']]);
            $message = "Facility added successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error adding facility: " . $e->getMessage();
            $message_type = "error";
        }
    }

    if (isset($_POST['add_facility_category'])) {
        $category_name = trim($_POST['category_name']);

        try {
            $stmt = $pdo->prepare("INSERT INTO facility_categories (name) VALUES (?)");
            $stmt->execute([$category_name]);
            $message = "Facility category added successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error adding facility category: " . $e->getMessage();
            $message_type = "error";
        }
    }
}

// Fetch categories
try {
    $categories_stmt = $pdo->query("SELECT * FROM facility_categories ORDER BY name");
    $categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $categories = [];
}

// Fetch facilities
try {
    $facilities_stmt = $pdo->query("SELECT f.*, fc.name as category_name FROM facilities f LEFT JOIN facility_categories fc ON f.category_id = fc.id ORDER BY f.name");
    $facilities = $facilities_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $facilities = [];
}

$page_title = "Facility Management";
include '../includes/admin_header.php';
?>

<div class="container-fluid">
    <h2>Facility Management</h2>

    <?php if ($message): ?>
        <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
            <?php echo htmlspecialchars($message); ?>
            <button type="button" class="close" data-dismiss="alert">
                <span>&times;</span>
            </button>
        </div>
    <?php endif; ?>

    <button class="btn btn-facilities mb-3" data-toggle="modal" data-target="#addFacilityModal">
        <i class="fas fa-plus"></i> Add New Facility
    </button>
    <button class="btn btn-facilities mb-3" data-toggle="modal" data-target="#addFacilityCategoryModal">
        <i class="fas fa-plus"></i> Add Facility Category
    </button>
    <button class="btn btn-facilities mb-3" data-toggle="modal" data-target="#viewFacilityCategoriesModal">
        <i class="fas fa-list"></i> View Facility Categories
    </button>

    <div class="table-responsive">
        <table class="table table-striped table-hover facilities-table">
            <thead class="thead-dark">
                <tr>
                    <th>Name</th>
                    <th>Category</th>
                    <th>Location</th>
                    <th>Capacity</th>
                    <th>Status</th>
                    <th>Requires Approval</th>
                    <th>Max Booking Hours</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($facilities as $facility): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($facility['name']); ?></td>
                        <td><?php echo htmlspecialchars($facility['category_name']); ?></td>
                        <td><?php echo htmlspecialchars($facility['location']); ?></td>
                        <td><?php echo htmlspecialchars($facility['capacity']); ?></td>
                        <td>
                            <span class="badge badge-<?php echo $facility['status'] === 'available' ? 'success' : 'secondary'; ?>">
                                <?php echo ucfirst($facility['status']); ?>
                            </span>
                        </td>
                        <td><?php echo $facility['requires_approval'] ? 'Yes' : 'No'; ?></td>
                        <td><?php echo htmlspecialchars($facility['max_booking_hours']); ?> hours</td>
                        <td>
                            <button class="btn btn-sm btn-info" onclick="viewBookings(<?php echo $facility['id']; ?>)">
                                <i class="fas fa-calendar"></i> Bookings
                            </button>
                            <button class="btn btn-sm btn-warning" onclick="editFacility(<?php echo $facility['id']; ?>)">
                                <i class="fas fa-edit"></i> Edit
                            </button>
                        </td>
                    </tr>
                <?php endforeach; ?>
                <?php if (empty($facilities)): ?>
                    <tr>
                        <td colspan="8" class="text-center">No facilities found.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Add Facility Modal -->
<div class="modal fade" id="addFacilityModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title">Add New Facility</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <label for="name"><i class="fas fa-tag"></i> Name *</label>
                            <input type="text" class="form-control" id="name" name="name" required>
                        </div>
                        <div class="form-group col-md-6">
                            <label for="category_id"><i class="fas fa-list"></i> Category *</label>
                            <select class="form-control" id="category_id" name="category_id" required>
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <label for="location"><i class="fas fa-map-marker-alt"></i> Location *</label>
                            <input type="text" class="form-control" id="location" name="location" required>
                        </div>
                        <div class="form-group col-md-6">
                            <label for="capacity"><i class="fas fa-users"></i> Capacity</label>
                            <input type="number" class="form-control" id="capacity" name="capacity" min="1">
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="description"><i class="fas fa-info-circle"></i> Description</label>
                        <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                    </div>
                    <div class="form-group">
                        <label for="amenities"><i class="fas fa-tools"></i> Amenities</label>
                        <textarea class="form-control" id="amenities" name="amenities" rows="2" placeholder="e.g., Projector, Whiteboard, Sound System"></textarea>
                    </div>
                    <div class="form-group">
                        <label for="rules"><i class="fas fa-gavel"></i> Rules & Regulations</label>
                        <textarea class="form-control" id="rules" name="rules" rows="3" placeholder="Facility usage rules and regulations"></textarea>
                    </div>
                    <div class="form-row">
                        <div class="form-group col-md-4">
                            <label for="max_booking_hours"><i class="fas fa-clock"></i> Max Booking Hours</label>
                            <input type="number" class="form-control" id="max_booking_hours" name="max_booking_hours" min="1" max="24" value="8">
                        </div>
                        <div class="form-group col-md-4">
                            <label for="advance_booking_days"><i class="fas fa-calendar-alt"></i> Advance Booking Days</label>
                            <input type="number" class="form-control" id="advance_booking_days" name="advance_booking_days" min="0" value="30">
                        </div>
                        <div class="form-group col-md-4">
                            <div class="form-check mt-4">
                                <input class="form-check-input" type="checkbox" id="requires_approval" name="requires_approval" checked>
                                <label class="form-check-label" for="requires_approval">
                                    <i class="fas fa-check-circle"></i> Requires Approval
                                </label>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="add_facility" class="btn btn-primary">Add Facility</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Add Facility Category Modal -->
<div class="modal fade" id="addFacilityCategoryModal" tabindex="-1" role="dialog" aria-labelledby="addFacilityCategoryModalLabel" aria-hidden="true">
  <div class="modal-dialog" role="document">
    <form method="POST" id="addFacilityCategoryForm">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title" id="addFacilityCategoryModalLabel">Add Facility Category</h5>
          <button type="button" class="close" data-dismiss="modal" aria-label="Close">
            <span aria-hidden="true">&times;</span>
          </button>
        </div>
        <div class="modal-body">
          <div class="form-group">
            <label for="facility_category_name">Category Name *</label>
            <input type="text" class="form-control" id="facility_category_name" name="category_name" required>
          </div>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
          <button type="submit" name="add_facility_category" class="btn btn-primary">Add Category</button>
        </div>
      </div>
    </form>
  </div>
</div>

<!-- View Facility Categories Modal -->
<div class="modal fade" id="viewFacilityCategoriesModal" tabindex="-1" role="dialog" aria-labelledby="viewFacilityCategoriesModalLabel" aria-hidden="true">
  <div class="modal-dialog" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="viewFacilityCategoriesModalLabel">Facility Categories</h5>
        <button type="button" class="close" data-dismiss="modal" aria-label="Close">
          <span aria-hidden="true">&times;</span>
        </button>
      </div>
      <div class="modal-body">
        <ul class="list-group" id="facilityCategoryList">
          <?php foreach ($categories as $category): ?>
            <li class="list-group-item"><?php echo htmlspecialchars($category['name']); ?></li>
          <?php endforeach; ?>
          <?php if (empty($categories)): ?>
            <li class="list-group-item text-center">No categories found.</li>
          <?php endif; ?>
        </ul>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>

<script>
function viewBookings(facilityId) {
    // TODO: Implement view bookings functionality
    alert('View bookings functionality to be implemented');
}

function editFacility(facilityId) {
    // TODO: Implement edit facility functionality
    alert('Edit facility functionality to be implemented');
}

$(document).ready(function() {
    $('#addFacilityCategoryForm').submit(function(e) {
        e.preventDefault();
        var categoryName = $('#facility_category_name').val().trim();
        if (categoryName === '') {
            alert('Category name is required.');
            return;
        }
        $.ajax({
            url: 'manage_facilities.php',
            type: 'POST',
            data: { add_facility_category: true, category_name: categoryName },
            success: function(response) {
                location.reload();
            },
            error: function() {
                alert('Error adding category.');
            }
        });
    });
});
</script>

<?php include '../includes/admin_footer.php'; ?>
