<?php
require_once '../config.php';

$message = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['create_fee'])) {
        $name = trim($_POST['name']);
        $description = trim($_POST['description']);
        $amount = $_POST['amount'];
        $academic_year_id = $_POST['academic_year_id'];
        $is_mandatory = isset($_POST['is_mandatory']) ? 1 : 0;

        if (empty($name) || empty($amount)) {
            $message = 'Please fill in all required fields.';
        } else {
            try {
                $stmt = $pdo->prepare("INSERT INTO fees (name, description, amount, academic_year_id, created_at) VALUES (?, ?, ?, ?, NOW())");
                $stmt->execute([$name, $description, $amount, $academic_year_id]);

                $message = 'Fee created successfully!';
            } catch (PDOException $e) {
                $message = 'Error creating fee: ' . $e->getMessage();
            }
        }
    } elseif (isset($_POST['edit_fee'])) {
        $fee_id = $_POST['fee_id'];
        $name = trim($_POST['name']);
        $description = trim($_POST['description']);
        $amount = $_POST['amount'];
        $academic_year_id = $_POST['academic_year_id'];

        if (empty($name) || empty($amount)) {
            $message = 'Please fill in all required fields.';
        } else {
            try {
                $stmt = $pdo->prepare("UPDATE fees SET name = ?, description = ?, amount = ?, academic_year_id = ? WHERE id = ?");
                $stmt->execute([$name, $description, $amount, $academic_year_id, $fee_id]);

                $message = 'Fee updated successfully!';
            } catch (PDOException $e) {
                $message = 'Error updating fee: ' . $e->getMessage();
            }
        }
    } elseif (isset($_POST['delete_fee'])) {
        $fee_id = $_POST['delete_fee'];

        try {
            $stmt = $pdo->prepare("DELETE FROM fees WHERE id = ?");
            $stmt->execute([$fee_id]);

            $message = 'Fee deleted successfully!';
        } catch (PDOException $e) {
            $message = 'Error deleting fee: ' . $e->getMessage();
        }
    }
}

// Get academic years for dropdown
$stmt = $pdo->query("SELECT id, year FROM academic_years ORDER BY year DESC");
$academic_years = $stmt->fetchAll();

// Get existing fees for display
$stmt = $pdo->query("SELECT f.*, ay.year as academic_year FROM fees f LEFT JOIN academic_years ay ON f.academic_year_id = ay.id ORDER BY f.created_at DESC");
$fees = $stmt->fetchAll();

// Get fee for edit
$edit_fee = null;
if (isset($_GET['edit'])) {
    $fee_id = $_GET['edit'];
    $stmt = $pdo->prepare("SELECT f.*, ay.year as academic_year FROM fees f LEFT JOIN academic_years ay ON f.academic_year_id = ay.id WHERE f.id = ?");
    $stmt->execute([$fee_id]);
    $edit_fee = $stmt->fetch();
}

include '../includes/admin_header.php';
?>

<?php if ($message): ?>
    <div class="message <?php echo strpos($message, 'Error') === 0 ? 'error' : 'success'; ?>">
        <?php echo htmlspecialchars($message); ?>
    </div>
<?php endif; ?>

<!-- Button to trigger modal -->
<button type="button" class="btn btn-primary" data-toggle="modal" data-target="#feeModal">
    Create New Fee
</button>

<!-- Fee Modal -->
<div class="modal fade" id="feeModal" tabindex="-1" role="dialog" aria-labelledby="feeModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <form method="post" action="add_fee.php" class="fee-form">
                <div class="modal-header">
                    <h5 class="modal-title" id="feeModalLabel"><?php echo $edit_fee ? 'Edit Fee' : 'Create New Fee'; ?></h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <?php if ($edit_fee): ?>
                        <input type="hidden" name="fee_id" value="<?php echo $edit_fee['id']; ?>">
                    <?php endif; ?>

                    <div class="form-group">
                        <label for="name">Fee Name *</label>
                        <input type="text" id="name" name="name" value="<?php echo htmlspecialchars($edit_fee['name'] ?? ''); ?>" required placeholder="e.g., Tuition Fee, Library Fee">
                    </div>

                    <div class="form-group">
                        <label for="description">Description</label>
                        <textarea id="description" name="description" rows="3" placeholder="Optional description of the fee"><?php echo htmlspecialchars($edit_fee['description'] ?? ''); ?></textarea>
                    </div>

                    <div class="form-group">
                        <label for="amount">Amount (GHC) *</label>
                        <input type="number" id="amount" name="amount" step="0.01" min="0" value="<?php echo htmlspecialchars($edit_fee['amount'] ?? ''); ?>" required placeholder="0.00">
                    </div>

                    <div class="form-group">
                        <label for="academic_year_id">Academic Year</label>
                        <select id="academic_year_id" name="academic_year_id">
                            <option value="">Select Academic Year (Optional)</option>
                            <?php foreach ($academic_years as $year): ?>
                                <option value="<?php echo $year['id']; ?>" <?php echo ($edit_fee && $edit_fee['academic_year_id'] == $year['id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars($year['year']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <!-- Removed is_mandatory checkbox as it is not used in the database -->
                    <!--
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input type="checkbox" id="is_mandatory" name="is_mandatory" checked>
                            <span class="checkmark"></span>
                            Mandatory Fee
                        </label>
                    </div>
                    -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="<?php echo $edit_fee ? 'edit_fee' : 'create_fee'; ?>" class="btn btn-primary"><?php echo $edit_fee ? 'Update Fee' : 'Create Fee'; ?></button>
                </div>
            </form>
        </div>
    </div>
</div>

<h2>Existing Fees</h2>
<div class="fees-table">
    <table>
        <thead>
            <tr>
                <th>Name</th>
                <th>Description</th>
                <th>Amount</th>
                <th>Academic Year</th>
                <th>Type</th>
                <th>Created</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($fees)): ?>
                <tr>
                    <td colspan="7" class="no-data">No fees created yet.</td>
                </tr>
            <?php else: ?>
                <?php foreach ($fees as $fee): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($fee['name']); ?></td>
                        <td><?php echo htmlspecialchars($fee['description'] ?: '-'); ?></td>
                        <td>GHC <?php echo number_format($fee['amount'], 2); ?></td>
                        <td><?php echo htmlspecialchars($fee['academic_year'] ?: 'All Years'); ?></td>
                        <td>All Fees</td>
                        <td><?php echo date('M d, Y', strtotime($fee['created_at'])); ?></td>
                        <td>
                            <a href="?edit=<?php echo $fee['id']; ?>" class="btn btn-small">Edit</a>
                            <form method="post" action="add_fee.php" style="display:inline;">
                                <input type="hidden" name="delete_fee" value="<?php echo $fee['id']; ?>">
                                <button type="submit" class="btn btn-small btn-danger" onclick="return confirm('Are you sure you want to delete this fee?')">Delete</button>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<style>
.form-container {
    background: #f8f9fa;
    padding: 20px;
    border-radius: 8px;
    margin-bottom: 30px;
    border: 1px solid #e9ecef;
}

.fee-form {
    max-width: 600px;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 5px;
    font-weight: 600;
    color: #495057;
}

.form-group input[type="text"],
.form-group input[type="number"],
.form-group textarea,
.form-group select {
    width: 100%;
    padding: 10px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.form-group textarea {
    resize: vertical;
    min-height: 80px;
}

.checkbox-label {
    display: flex;
    align-items: center;
    cursor: pointer;
    font-weight: normal;
}

.checkbox-label input[type="checkbox"] {
    margin-right: 10px;
}

.fees-table {
    background: white;
    border-radius: 8px;
    overflow: hidden;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.fees-table table {
    width: 100%;
    border-collapse: collapse;
}

.fees-table th,
.fees-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.fees-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.fees-table tr:hover {
    background: #f8f9fa;
}

.no-data {
    text-align: center;
    color: #6c757d;
    font-style: italic;
}

.message {
    padding: 12px 16px;
    margin-bottom: 20px;
    border-radius: 4px;
}

.message.success {
    background-color: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.message.error {
    background-color: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Auto-format amount input
    const amountInput = document.getElementById('amount');
    if (amountInput) {
        amountInput.addEventListener('blur', function() {
            const value = parseFloat(this.value);
            if (!isNaN(value)) {
                this.value = value.toFixed(2);
            }
        });
    }

    // Auto-open modal for edit
    <?php if ($edit_fee): ?>
        $('#feeModal').modal('show');
    <?php endif; ?>
});
</script>

<?php
include '../includes/admin_footer.php';
?>
